# -*- coding: utf-8 -*-
"""
/***************************************************************************
 LayerNote
                                 A QGIS plugin
 Aggiungi annotazioni contestuali a layer e feature
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-05-09
        git sha              : $Format:%H$
        copyright            : (C) 2025 by dr. Geol. Faustino Cetraro
        email                : geol-faustino@libero.it
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .layer_note_dockwidget import LayerNoteDockWidget
import os.path

from qgis.core import QgsProject, QgsMapLayer
from qgis.PyQt.QtWidgets import QMessageBox


class LayerNote:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(self.plugin_dir, 'i18n', f'layer_note_{locale}.qm')

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Layer Note')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'LayerNote')
        self.toolbar.setObjectName(u'LayerNote')

        #print "** INITIALIZING LayerNote"

        self.pluginIsActive = False
        self.dockwidget = None


    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('LayerNote', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = os.path.join(self.plugin_dir, 'icon.png')
        self.add_action(
            icon_path,
            text=self.tr(u'Layer Note'),
            callback=self.run,
            parent=self.iface.mainWindow())
        
        QgsProject.instance().cleared.connect(self.clear_plugin_state)
    #--------------------------------------------------------------------------

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        #print "** CLOSING LayerNote"

        # disconnects
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        # self.dockwidget = None

        self.pluginIsActive = False


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        #print "** UNLOAD LayerNote"

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Layer Note'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

        # Disconnetti il segnale
        QgsProject.instance().cleared.disconnect(self.clear_plugin_state)

    #--------------------------------------------------------------------------

    def run(self):
        """Run method that loads and starts the plugin"""

        if not self.pluginIsActive:
            self.pluginIsActive = True

            #print "** STARTING LayerNote"

            # dockwidget may not exist if:
            #    first run of plugin
            #    removed on close (see self.onClosePlugin method)
            if self.dockwidget == None:
                # Create the dockwidget (after translation) and keep reference
                self.dockwidget = LayerNoteDockWidget()

            # connect to provide cleanup on closing of dockwidget
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            # show the dockwidget
            # TODO: fix to allow choice of dock location
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()

            # Impostare il comportamento di ancoraggio
            self.dockwidget.setFloating(True)  # Permette il disancoraggio

            # Collegamento dei segnali
            self.dockwidget.saveButton.clicked.connect(self.save_note)
            self.dockwidget.layerComboBox.currentTextChanged.connect(self.load_note)
            self.dockwidget.clearButton.clicked.connect(self.clear_note)

            # Connetti il pulsante "Aggiorna lista" alla funzione di aggiornamento
            self.dockwidget.refreshButton.clicked.connect(self.populate_layer_combo)

            # Popola il combo box con i nomi dei layer vettoriali
            self.populate_layer_combo()

    def populate_layer_combo(self):
        self.dockwidget.layerComboBox.clear()
        layers = [layer for layer in QgsProject.instance().mapLayers().values() if layer.type() == layer.VectorLayer]
        for layer in layers:
            self.dockwidget.layerComboBox.addItem(layer.name())

    def save_note(self):
        layer_name = self.dockwidget.layerComboBox.currentText()
        note = self.dockwidget.noteTextEdit.toPlainText()

        if not layer_name:
            return

        note = self.dockwidget.noteTextEdit.toPlainText().strip()
        if not note:
            # Mostra un messaggio di avviso se il campo delle note è vuoto
            QMessageBox.warning(
                self.iface.mainWindow(),
                self.tr("Nota vuota"),
                self.tr(f"Non puoi salvare una nota vuota per il layer \"{layer_name}\". Inserisci del testo prima di salvare.")
            )
            return
        
        key = f"layer_note_{layer_name}"
        custom_vars = QgsProject.instance().customVariables()

        # Se esiste già una nota e l'utente sta per sovrascriverla
        if key in custom_vars and custom_vars[key].strip() and note.strip():
            reply = QMessageBox.question(
                self.iface.mainWindow(),
                self.tr("Conferma sovrascrittura"),
                self.tr(f"Esiste già una nota per il layer \"{layer_name}\". Vuoi sovrascriverla?"),
                QMessageBox.Yes | QMessageBox.No,
                QMessageBox.No
            )
            if reply != QMessageBox.Yes:
                return  # L'utente ha scelto di non sovrascrivere

        # Salva la nota
        custom_vars[key] = note
        QgsProject.instance().setCustomVariables(custom_vars)

        # Messaggio di conferma
        QMessageBox.information(
            self.iface.mainWindow(),
            self.tr("Nota salvata"),
            self.tr(f"La nota per il layer \"{layer_name}\" è stata salvata con successo.")
        )

    def load_note(self, layer_name):
        custom_vars = QgsProject.instance().customVariables()
        note = custom_vars.get(f"layer_note_{layer_name}", "")
        
        self.dockwidget.noteTextEdit.setPlainText(note)

    def clear_note(self):
        layer_name = self.dockwidget.layerComboBox.currentText()

        if not layer_name:
            return

        note = self.dockwidget.noteTextEdit.toPlainText().strip()
        if not note:
            return  # Non fare nulla se la nota è già vuota

        # Mostra un messaggio di conferma
        reply = QMessageBox.question(
            self.iface.mainWindow(),
            self.tr("Conferma cancellazione"),
            self.tr(f"Sei sicuro di voler cancellare la nota per il layer \"{layer_name}\"?"),
            QMessageBox.Yes | QMessageBox.No,
            QMessageBox.No
        )

        if reply == QMessageBox.Yes:
            # Cancella la nota dalla UI
            self.dockwidget.noteTextEdit.clear()

            # Rimuove la nota dal progetto
            custom_vars = QgsProject.instance().customVariables()
            key = f"layer_note_{layer_name}"
            if key in custom_vars:
                del custom_vars[key]
                QgsProject.instance().setCustomVariables(custom_vars)

            QMessageBox.information(
                self.iface.mainWindow(),
                self.tr("Nota cancellata"),
                self.tr(f"La nota per il layer \"{layer_name}\" è stata cancellata con successo.")
            )
        
    def onClosePlugin(self):
        """Pulisce quando il plugin viene chiuso"""

        # Disabilita il dock widget se chiuso
        if self.dockwidget:
            self.dockwidget.setFloating(False)
        self.pluginIsActive = False
    
    def clear_plugin_state(self):
        """Ripulisce l'interfaccia del plugin quando si avvia un nuovo progetto."""
        if self.dockwidget:
            self.dockwidget.layerComboBox.clear()
            self.dockwidget.noteTextEdit.clear()
