# -*- coding: utf-8 -*-
"""
/***************************************************************************
 LayerGridView
                                 A QGIS plugin
 The Layer Grid Plugin provides an intuitive dockable widget that presents a grid of map canvases.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-09-23
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Tharles de Sousa Andrade
        email                : irtharles@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QTimer
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import (
    QWidget, QAction, QLabel, QVBoxLayout, QGridLayout,
    QScrollArea, QDialog, QDialogButtonBox, QHBoxLayout,
)
from qgis.core import (
    Qgis, QgsProject, QgsCoordinateTransform, QgsMessageLog,
)
# Initialize Qt resources from file resources.py
from .resources import *
from .canvas import CustomMapCanvas

# Import the code for the DockWidget
from .layer_grid_view_dockwidget import LayerGridViewDockWidget
import os.path


class LayerGridView:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        self.iface = iface
        self.canvases = []
        self.layers = []
        self._transform_cache = {}
        self._sync_timer = None

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'LayerGridView_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Layer Grid View')
        self.toolbar = self.iface.addToolBar(u'LayerGridView')
        self.toolbar.setObjectName(u'LayerGridView')

        self.pluginIsActive = False
        self.dockwidget = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('LayerGridView', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar."""

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/layer_grid_view/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Layer Grid View'),
            callback=self.run,
            parent=self.iface.mainWindow())

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed."""

        # Disconnect signals
        try:
            self.iface.mapCanvas().extentsChanged.disconnect(self._on_extent_changed)
        except TypeError:
            pass
        try:
            QgsProject.instance().crsChanged.disconnect(self._on_project_crs_changed)
        except TypeError:
            pass

        if self.dockwidget:
            self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        # Stop pending sync timer
        if self._sync_timer and self._sync_timer.isActive():
            self._sync_timer.stop()

        # Clean up canvases
        for canvas in self.canvases:
            canvas.deleteLater()
        self.canvases = []
        self.layers = []
        self._transform_cache.clear()

        if self.dockwidget:
            self.dockwidget.hide()
        self.dockwidget = None
        self.pluginIsActive = False

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        if hasattr(self.iface, 'removePluginMenu'):
            for action in self.actions:
                self.iface.removePluginMenu(
                    self.tr(u'&Layer Grid View'),
                    action)
                self.iface.removeToolBarIcon(action)

            if self.toolbar is not None:
                del self.toolbar
        else:
            QgsMessageLog.logMessage(
                "QgsInterface does not have the method removePluginMenu",
                'LayerGrid', Qgis.Warning
            )

    def _transform_extent(self, extent, src_crs, dst_crs):
        """Transform extent between CRSs, with cached transform."""
        if not src_crs.isValid() or not dst_crs.isValid():
            return extent
        if src_crs == dst_crs:
            return extent

        cache_key = (src_crs.authid(), dst_crs.authid())
        if cache_key not in self._transform_cache:
            self._transform_cache[cache_key] = QgsCoordinateTransform(
                src_crs, dst_crs, QgsProject.instance()
            )

        try:
            return self._transform_cache[cache_key].transformBoundingBox(extent)
        except Exception:
            QgsMessageLog.logMessage(
                f"Transform failed: {src_crs.authid()} -> {dst_crs.authid()}",
                'LayerGrid', Qgis.Warning
            )
            return extent

    def updateGrid(self, layers):
        """Build the grid of canvases, one per layer."""
        try:
            # Clear existing widgets from the layout
            for i in reversed(range(self.layout.count())):
                widget = self.layout.itemAt(i).widget()
                if widget is not None:
                    self.layout.removeWidget(widget)
                    widget.deleteLater()

            self.canvases = []
            project_crs = QgsProject.instance().crs()

            row = 0
            col = 0
            for layer in layers:
                if not layer.isValid():
                    QgsMessageLog.logMessage(
                        f"Layer '{layer.name()}' is not valid, skipping.",
                        'LayerGrid', Qgis.Warning
                    )
                    continue

                label = QLabel(layer.name())
                label.setStyleSheet("font-size: 12px;")
                label.setAlignment(Qt.AlignCenter)

                canvas = CustomMapCanvas()
                canvas.setCanvasColor(Qt.gray)
                canvas.setDestinationCrs(project_crs)
                canvas.setLayers([layer])

                # Transform layer extent to project CRS
                layer_extent = self._transform_extent(
                    layer.extent(), layer.crs(), project_crs
                )
                canvas.setExtent(layer_extent)
                canvas.refresh()
                canvas.showLoading()

                self.canvases.append(canvas)
                self.layout.addWidget(label, row, col)
                self.layout.addWidget(canvas, row + 1, col)

                col += 1
                if col > 2:
                    col = 0
                    row += 2

        except Exception as e:
            if self.iface:
                msg = self.iface.messageBar().createMessage("GRID", f"Error -> {e}")
                self.iface.messageBar().pushWidget(msg, level=Qgis.Critical)
            else:
                QgsMessageLog.logMessage(
                    f"Error in updateGrid: {e}", 'LayerGrid', Qgis.Critical
                )

    def _on_extent_changed(self):
        """Debounced handler - schedules sync_zoom after 50ms."""
        if self._sync_timer and self._sync_timer.isActive():
            self._sync_timer.stop()
        self._sync_timer = QTimer()
        self._sync_timer.setSingleShot(True)
        self._sync_timer.timeout.connect(self._do_sync_zoom)
        self._sync_timer.start(50)

    def _do_sync_zoom(self):
        """Sync extent from main canvas to all grid canvases."""
        if not self.canvases or not self.iface:
            return

        try:
            main_canvas = self.iface.mapCanvas()
            main_extent = main_canvas.extent()
            main_crs = main_canvas.mapSettings().destinationCrs()
            project_crs = QgsProject.instance().crs()

            extent = self._transform_extent(main_extent, main_crs, project_crs)

            for canvas in self.canvases:
                canvas.setExtent(extent)
                canvas.refresh()

        except Exception as e:
            QgsMessageLog.logMessage(
                f"Error syncing zoom: {e}", 'LayerGrid', Qgis.Warning
            )

    def _on_project_crs_changed(self):
        """Update destination CRS on all grid canvases when project CRS changes."""
        project_crs = QgsProject.instance().crs()
        self._transform_cache.clear()
        for canvas in self.canvases:
            canvas.setDestinationCrs(project_crs)
            canvas.refresh()

    def _get_selected_layers(self):
        """Return the layers currently selected in the Layers panel."""
        return self.iface.layerTreeView().selectedLayers()

    def _show_no_selection_dialog(self):
        """Show an informational dialog when no layers are selected."""
        dialog = QDialog(self.iface.mainWindow())
        dialog.setWindowTitle('Layer Grid View')
        dialog.setMinimumSize(520, 380)

        layout = QVBoxLayout(dialog)
        layout.setSpacing(16)
        layout.setContentsMargins(24, 24, 24, 20)

        # Icon + title row
        header = QHBoxLayout()
        icon_label = QLabel()
        icon_label.setPixmap(
            QIcon(':/plugins/layer_grid_view/icon.png').pixmap(40, 40)
        )
        title = QLabel('No layers selected')
        title.setStyleSheet('font-size: 18px; font-weight: bold;')
        header.addWidget(icon_label)
        header.addWidget(title)
        header.addStretch()
        layout.addLayout(header)

        # Instructions
        instructions = QLabel(
            'To use <b>Layer Grid View</b>, select one or more layers in the '
            '<b>Layers</b> panel before opening the plugin.<br><br>'
            '<b>How to select layers:</b>'
            '<ol style="margin-top: 6px;">'
            '<li style="margin-bottom: 8px;">'
            'In the <b>Layers</b> panel (usually on the left side), '
            'click on a layer name to select it.</li>'
            '<li style="margin-bottom: 8px;">'
            'To select <b>multiple layers</b>, hold <code>Ctrl</code> '
            '(<code>Cmd</code> on macOS) and click on each layer '
            'you want to include.</li>'
            '<li style="margin-bottom: 8px;">'
            'To select a <b>range</b> of consecutive layers, click the '
            'first layer, then hold <code>Shift</code> and click the last one.</li>'
            '<li style="margin-bottom: 8px;">'
            'Once your layers are highlighted, open the plugin again via '
            '<b>Plugins &gt; Layer Grid View</b>.</li>'
            '</ol>'
            'Each selected layer will be displayed in its own map canvas, '
            'arranged in a synchronized grid. Layers with different CRS '
            'are automatically reprojected to the project CRS.'
        )
        instructions.setWordWrap(True)
        instructions.setStyleSheet('font-size: 13px; line-height: 1.5;')
        layout.addWidget(instructions)

        layout.addStretch()

        # OK button
        buttons = QDialogButtonBox(QDialogButtonBox.Ok)
        buttons.accepted.connect(dialog.accept)
        layout.addWidget(buttons)

        dialog.exec_()

    def run(self):
        if self.pluginIsActive:
            self.dockwidget.show()
            return

        selected_layers = self._get_selected_layers()

        if not selected_layers:
            self._show_no_selection_dialog()
            return

        self.pluginIsActive = True
        self._transform_cache.clear()
        self.canvases = []
        self.layers = list(selected_layers)

        QgsMessageLog.logMessage(
            f"Building grid with {len(self.layers)} selected layer(s)",
            'LayerGrid', Qgis.Info
        )

        try:
            if self.dockwidget is None:
                self.dockwidget = LayerGridViewDockWidget()

            self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            self.scrollArea = QScrollArea()
            self.scrollArea.setWidgetResizable(True)

            contentWidget = QWidget()
            contentLayout = QVBoxLayout(contentWidget)
            self.layout = QGridLayout()

            contentLayout.addLayout(self.layout)
            self.scrollArea.setWidget(contentWidget)
            self.updateGrid(self.layers)

            self.dockwidget.setWidget(self.scrollArea)
            self.iface.mapCanvas().extentsChanged.connect(self._on_extent_changed)
            QgsProject.instance().crsChanged.connect(self._on_project_crs_changed)

            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()

        except Exception as e:
            if self.iface:
                msg = self.iface.messageBar().createMessage("run", f"Error -> {e}")
                self.iface.messageBar().pushWidget(msg, level=Qgis.Critical)
            else:
                QgsMessageLog.logMessage(
                    f"Error in run: {e}", 'LayerGrid', Qgis.Critical
                )
