from qgis.core import QgsMessageLog, Qgis, QgsProject
from PyQt5.QtWidgets import QColorDialog, QAction, QMenu, QStyledItemDelegate
from PyQt5.QtGui import QColor, QPainter
from PyQt5.QtCore import Qt


def log_message(message):
    """
    Envia mensagens para o painel de logs do QGIS e para a consola Python.
    """
    QgsMessageLog.logMessage(message, "LayerColorPlugin", Qgis.Info)
    print(message)  # Exibe na consola Python


class LayerColorPlugin:
    clipboard_color = None  # Variável para armazenar a cor copiada

    def copy_highlight_color(self):
        selected_nodes = self.layer_tree_view.selectedNodes()
        if not selected_nodes:
            log_message("Nenhuma camada ou grupo selecionado para copiar a cor.")
            return

        node = selected_nodes[0]
        node_name = node.name()
        color = self.layer_colors.get(node_name)
        if color:
            LayerColorPlugin.clipboard_color = color
            log_message(f"Cor '{color}' copiada para o clipboard.")
        else:
            log_message("Nenhuma cor atribuída ao item selecionado.")

    def paste_highlight_color(self):
        if not LayerColorPlugin.clipboard_color:
            log_message("Nenhuma cor disponível no clipboard para colar.")
            return
    
        selected_nodes = self.layer_tree_view.selectedNodes()
        if not selected_nodes:
            log_message("Nenhuma camada ou grupo selecionado para colar a cor.")
            return
    
        # Aplicar a cor a todos os nós selecionados
        for node in selected_nodes:
            node_name = node.name()
            self.layer_colors[node_name] = LayerColorPlugin.clipboard_color
            node.setCustomProperty("highlight_color", LayerColorPlugin.clipboard_color)  # Persistir a cor
            log_message(f"Cor '{LayerColorPlugin.clipboard_color}' colada no item '{node_name}'.")
    
        # Atualizar a visualização uma única vez após aplicar a todas as camadas
        self.layer_tree_view.viewport().update()
        
        
    def __init__(self, iface):
        self.iface = iface
        self.layer_colors = {}

    def initGui(self):
        try:
            self.layer_tree_view = self.iface.layerTreeView()
            if not self.layer_tree_view:
                raise Exception("Layer Tree View não disponível")
            
            # Delegate personalizado
            self.delegate = LayerColorDelegate(self.layer_colors, self.layer_tree_view)
            self.layer_tree_view.setItemDelegate(self.delegate)

            # Conectar sinal do menu de contexto
            self.layer_tree_view.setContextMenuPolicy(Qt.CustomContextMenu)
            self.layer_tree_view.customContextMenuRequested.connect(self.show_context_menu)

            # Conectar ao evento de salvamento do projeto
            QgsProject.instance().writeProject.connect(self.save_colors)

            # Conectar ao evento de carregamento do projeto
            self.iface.projectRead.connect(self.load_colors)

            # Carregar cores iniciais se o projeto já estiver carregado
            if self.iface.activeLayer():
                self.load_colors()

        except Exception as e:
            log_message(f"Erro na inicialização do plugin: {str(e)}")

    def unload(self):
        try:
            # Desconectar sinal de menu de contexto
            if hasattr(self, "layer_tree_view"):
                self.layer_tree_view.customContextMenuRequested.disconnect(self.show_context_menu)

            # Restaurar delegate original
            if hasattr(self, "delegate") and self.delegate is not None:
                original_delegate = QStyledItemDelegate(self.layer_tree_view)
                self.layer_tree_view.setItemDelegate(original_delegate)
                self.delegate = None

            # Desconectar sinais
            QgsProject.instance().writeProject.disconnect(self.save_colors)
            self.iface.projectRead.disconnect(self.load_colors)

            # Limpar dicionário de cores
            if hasattr(self, "layer_colors"):
                self.layer_colors.clear()

            # Atualizar interface
            if hasattr(self, "layer_tree_view"):
                self.layer_tree_view.viewport().update()

        except Exception as e:
            log_message(f"Erro durante unload: {str(e)}")

    def show_context_menu(self, point):
        # Desconectar temporariamente o sinal
        self.layer_tree_view.customContextMenuRequested.disconnect(self.show_context_menu)
    
        # Criar o menu de contexto personalizado
        context_menu = QMenu(self.layer_tree_view)
    
        # Obter as ações padrão do QGIS (se disponíveis)
        default_menu_provider = self.layer_tree_view.menuProvider()
        if default_menu_provider:
            default_menu = default_menu_provider.createContextMenu()
            if default_menu:
                context_menu.addActions(default_menu.actions())
    
        # Criar o submenu personalizado
        highlight_menu = QMenu("Layer Highlight Color", context_menu)
    
        set_color_action = QAction("Set Background Color", highlight_menu)
        set_color_action.triggered.connect(self.set_layer_color)
        highlight_menu.addAction(set_color_action)
    
        remove_color_action = QAction("Remove Background Color", highlight_menu)
        remove_color_action.triggered.connect(lambda: self.handle_remove_color(context_menu))
        highlight_menu.addAction(remove_color_action)
        
        copy_color_action = QAction("Copy Highlight Color", highlight_menu)
        copy_color_action.triggered.connect(lambda: self.handle_copy_color(context_menu))
        highlight_menu.addAction(copy_color_action)
        
        paste_color_action = QAction("Paste Highlight Color", highlight_menu)
        paste_color_action.triggered.connect(lambda: self.handle_paste_color(context_menu))
        highlight_menu.addAction(paste_color_action)
    
        # Inserir o submenu antes de "Properties"
        actions = context_menu.actions()
        if actions:
            context_menu.insertMenu(actions[-1], highlight_menu)
        else:
            context_menu.addMenu(highlight_menu)
    
        # Exibir o menu de contexto e reconectar o sinal depois
        context_menu.aboutToHide.connect(
            lambda: self.layer_tree_view.customContextMenuRequested.connect(self.show_context_menu)
        )
        context_menu.exec_(self.layer_tree_view.mapToGlobal(point))
        
    def handle_remove_color(self, context_menu):
        self.remove_layer_color()
        context_menu.close()
            
    def handle_copy_color(self, menu):
        self.copy_highlight_color()
        menu.close()
        
    def handle_paste_color(self, menu):
        self.paste_highlight_color()
        menu.close()

    def set_layer_color(self):
        selected_nodes = self.layer_tree_view.selectedNodes()
        if not selected_nodes:
            return
    
        # Abrir o diálogo de cor uma única vez
        color = QColorDialog.getColor()
        if color.isValid():
            # Aplicar a cor a todos os nós selecionados
            for layer in selected_nodes:
                layer_name = layer.name()
                self.layer_colors[layer_name] = color.name()
                layer.setCustomProperty("highlight_color", color.name())  # Persistir a cor
                log_message(f"Camada ou grupo: {layer_name}, Cor atribuída: {color.name()}")
            
            # Atualizar a visualização uma única vez após aplicar a todas as camadas
            self.layer_tree_view.viewport().update()

    def remove_layer_color(self):
        selected_nodes = self.layer_tree_view.selectedNodes()
        if not selected_nodes:
            return
    
        # Remover a cor de todos os nós selecionados
        for layer in selected_nodes:
            layer_name = layer.name()
            if layer_name in self.layer_colors:
                del self.layer_colors[layer_name]
                layer.removeCustomProperty("highlight_color")  # Remover a propriedade persistente
                log_message(f"Cor removida da camada ou grupo: {layer_name}")
        
        # Atualizar a visualização uma única vez após remover de todas as camadas
        self.layer_tree_view.viewport().update()

    def save_colors(self):
        """
        Salva as cores personalizadas no projeto para camadas e grupos.
        """
        try:
            root = self.iface.layerTreeView().layerTreeModel().rootGroup()
            for node in root.children():
                if hasattr(node, "customProperty"):  # Verificar se o nó suporta propriedades personalizadas
                    name = node.name()
                    color = self.layer_colors.get(name)
                    if color:
                        node.setCustomProperty("highlight_color", color)
                        log_message(f"Salvando cor '{color}' para '{name}'")
            log_message("Todas as cores foram salvas com sucesso no projeto.")
        except Exception as e:
            log_message(f"Erro ao salvar cores: {str(e)}")

    def load_colors(self):
        """
        Carrega as cores personalizadas do projeto para camadas e grupos.
        """
        self.layer_colors.clear()
        try:
            root = self.iface.layerTreeView().layerTreeModel().rootGroup()
            for node in root.children():
                if hasattr(node, "customProperty"):  # Verificar se o nó suporta propriedades personalizadas
                    color = node.customProperty("highlight_color")
                    if color:
                        self.layer_colors[node.name()] = color
                        log_message(f"Cor '{color}' carregada para '{node.name()}'")
            self.layer_tree_view.viewport().update()
            log_message("Todas as cores foram restauradas com sucesso.")
        except Exception as e:
            log_message(f"Erro ao carregar cores: {str(e)}")

    def get_layer_by_name(self, layer_name):
        root = self.iface.layerTreeView().layerTreeModel().rootGroup()
        for node in root.children():
            if node.name() == layer_name:
                return node
        return None


class LayerColorDelegate(QStyledItemDelegate):
    def __init__(self, layer_colors, parent=None):
        super().__init__(parent)
        self.layer_colors = layer_colors

    def paint(self, painter, option, index):
        layer_name = index.data(Qt.DisplayRole)
        if layer_name in self.layer_colors:
            painter.save()
            color = QColor(self.layer_colors[layer_name])
            painter.fillRect(option.rect, color)
            painter.restore()

        super().paint(painter, option, index)
