import time

from utilities.dcmgeometrysdk.dcmgeometry.points import PointGeom
from utilities.dcmgeometrysdk.dcmgeometry.admin import Admin
from utilities.dcmgeometrysdk.dcmgeometry.lines import LineGeom
from utilities.dcmgeometrysdk.dcmgeometry.polygons import PolygonGeom
from utilities.dcmgeometrysdk.dcmgeometry.arcs import ArcGeom
from utilities.dcmgeometrysdk.dcmgeometry.surveygraph import SurveyGraph
from utilities.dcmgeometrysdk.dcmgeometry.geometry import Geometries
from utilities.dcmgeometrysdk.geometryfunctions.conversionsfunctions import dd2hp, hp2dd
from domain.plan.model.point import Point
from domain.plan.model.admin import Admin as PlanAdmin
from domain.plan.model.line_detail import LineDetail
from domain.plan.model.parcel import Parcel
from domain.plan.model.parcel_line import ParcelLine
from domain.plan.model.point_detail import PointDetail
from domain.plan.model.type_lookup import TypeLookup

from sqlalchemy.orm import Session
import shapely.wkb as swkb
import shapely.wkt as swkt
import shapely.geometry as sg
from copy import deepcopy
from datetime import datetime


class PlanDb2DCMGeom:
    def __init__(self, engine, ids, split=True, polygons=True):
        self.geometries = {}
        self.ids = ids
        self.plan_lookup = {}
        self.engine = engine
        admin, points, lines, parcels = self.get_data(split)
        self.lookup = self.get_lookup()
        self.set_points(points)
        self.set_lines_arcs(lines)
        if polygons is True:
            self.set_parcels(parcels)
        self.set_admin(admin)

    def get_lookup(self):
        with Session(self.engine) as open_session:
            lookup = open_session.query(TypeLookup).all()
            return {l.type_id: l.label for l in lookup}

    def group_data(self, data, plan_lookup):
        grouped = {}
        for item in data:
            id_ = plan_lookup.get(item.admin_id)
            group = grouped.get(id_, [])
            group.append(item)
            grouped[id_] = group
        return grouped

    def get_data(self, split):
        with Session(self.engine) as open_session:
            query = PlanAdmin.id.in_(self.ids) & PlanAdmin.retire_event_id.is_(None)
            admin = open_session.query(PlanAdmin).where(query).all()
            if split is True:
                plan_lookup = {a.id: a.survey_number for a in admin}

                admin = {a.survey_number: a for a in admin}
            else:
                plan_lookup = {a.id: 'all' for a in admin}
                admin = {'all': a for a in admin}
            self.plan_lookup = plan_lookup

            query = LineDetail.admin_id.in_(self.ids) & LineDetail.retire_event_id.is_(None)
            lines = open_session.query(LineDetail).where(query).all()
            lines = self.group_data(lines, plan_lookup)

            query = (Parcel.admin_id.in_(self.ids) & Parcel.retire_event_id.is_(None) &
                     Parcel.parcel_geometry.isnot(None))
            parcels = open_session.query(Parcel).where(query).all()
            parcels = self.group_data(parcels, plan_lookup)

            on_ = Point.point_id == PointDetail.point_id
            query = PointDetail.admin_id.in_(self.ids) & PointDetail.retire_event_id.is_(None)
            points = open_session.query(*Point.__table__.columns,
                                        *PointDetail.__table__.columns).join(PointDetail, on_).where(query).all()
            points = self.group_data(points, plan_lookup)

        return admin, points, lines, parcels

    def set_points(self, points):
        for k, v in points.items():
            geometry = self.geometries.get(k, Geometries())
            geometry.points = {}
            for point in v:
                p = PointGeom()
                p.id = point.id
                if isinstance(point.id, int):
                    p.name = point.point_id
                else:
                    p.name = str(point.point_id)
                p.point_oid = point.point_id
                p.geometry = swkb.loads(point.draw_position.desc, hex=True)
                p.point_type = self.lookup.get(point.point_type, point.point_type)
                p.point_state = self.lookup.get(point.point_state, point.point_state)
                p.crs = point.draw_position.srid
                p.mon_type = self.lookup.get(point.mark_type, point.mark_type)
                p.mon_state = self.lookup.get(point.mark_state, point.mark_state)
                p.mon_desc = point.description
                p.mon_condition = point.mark_condition
                p.mon_origin_survey = point.ref_survey_number
                geometry.points[p.name] = p
            self.geometries[k] = geometry

    def set_lines_arcs(self, lines):
        for k, v in lines.items():
            geometry = self.geometries.get(k, Geometries())
            geometry.lines = {}
            line: LineDetail
            for line in v:
                if line.radius is not None:
                    l = ArcGeom()
                    l.is_arc = True
                    l.arc_length = line.arc_length
                    l.radius = line.radius
                    l.rot = line.rotation
                else:
                    l = LineGeom()
                l.name = line.name
                l.id = line.id
                l.setup_point = self.geometries[k].points.get(line.setup_point_id, PointGeom())
                l.target_point = self.geometries[k].points.get(line.target_point_id, PointGeom())
                if line.line_geometry is not None:
                    l.geometry = swkb.loads(line.line_geometry.desc, hex=True)
                l.hp_bearing = line.bearing_dms
                l.calc_dd_bearing()
                l.distance = float(line.distance.real)
                if line.ignore_event_id is not None:
                    l.distance_type = 'Ignored'
                elif line.pseudo is True:
                    l.line_type = 'pseudo'
                else:
                    l.azimuth_type = self.lookup.get(line.bearing_type, line.bearing_type)
                    l.distance_type = self.lookup.get(line.distance_type, line.distance_type)
                l.bearing_std = line.direction_std
                l.distance_std = line.distance_std
                l.az_adopt_fact = line.az_adopt_factor_dms
                l.crs = line.line_geometry.srid
                l.desc = line.description

                name = (l.setup_point.name, l.target_point.name)
                geometry.lines[name] = l
            self.geometries[k] = geometry

    def get_line_order(self, parcels):
        all_los = {}
        with Session(self.engine) as open_session:
            pids = set()
            for k, v in parcels.items():
                for p in v:
                    pids.add(p.id)

            query = ((ParcelLine.parcel_id.in_(pids)) & (ParcelLine.retire_event_id.is_(None))
                     & LineDetail.retire_event_id.is_(None))
            on_ = ParcelLine.line_id == LineDetail.line_id
            all_lines = open_session.query(*ParcelLine.__table__.columns,
                                           *LineDetail.__table__.columns).join(LineDetail, on_).where(
                query).order_by(ParcelLine.parcel_id,
                                ParcelLine.parcel_index,
                                ParcelLine.ring, ParcelLine.seq).all()
            al = {}
            for pl in all_lines:
                ls = al.get(pl.parcel_id, [])
                ls.append(pl)
                al[pl.parcel_id] = ls

            for k, v in parcels.items():
                p: Parcel
                los = {}
                for p in v:
                    lo = {}
                    lines = al.get(p.id, [])
                    for l in lines:
                        lr = lo.get(f'ring-{str(l.ring)}', [])
                        if l.reverse is True:
                            s = l.target_point_id
                            t = l.setup_point_id
                        else:
                            s = l.setup_point_id
                            t = l.target_point_id
                        line = deepcopy(self.geometries[k].lines.get((s, t)))

                        if l.reverse is True and line is not None:
                            line.flip_direction()
                        elif line is None:
                            line = deepcopy(self.geometries[k].lines.get((t, s)))
                        lr.append(line)
                        lo[f'ring-{str(l.ring)}'] = lr
                    if len(lo) > 0:
                        los[p.name] = lo
                all_los[k] = los
        return all_los

    def set_parcels(self, parcels):
        los = self.get_line_order(parcels)

        for k, v in parcels.items():
            geometry = self.geometries.get(k, Geometries())
            geometry.polygons = {}
            parcel: Parcel
            for parcel in v:
                p = PolygonGeom()
                p.name = parcel.name
                p.id = parcel.id
                p.line_order = los[k].get(p.name)
                p.parcel_state = self.lookup.get(parcel.parcel_state, parcel.parcel_state)
                p.parcel_state_id = parcel.parcel_state
                p.parcel_type = self.lookup.get(parcel.parcel_state, parcel.parcel_state)
                p.parcel_class = self.lookup.get(parcel.parcel_class, parcel.parcel_class)
                p.parcel_format = self.lookup.get(parcel.parcel_format, parcel.parcel_format)
                p.parcel_use = parcel.parcel_use
                if parcel.parcel_geometry is not None:
                    p.geometry = swkb.loads(parcel.parcel_geometry.desc, hex=True)
                    p.valid_geom = p.geometry.is_valid
                    p.crs = parcel.parcel_geometry.srid
                p.desc = parcel.description
                if parcel.pseudo is True:
                    p.parcel_type = 'pseudo'
                if p.parcel_class == 'Easement':
                    p.easement = True
                if p.parcel_type == 'Multipart':
                    p.multipart = True
                if p.parcel_type == 'Part':
                    p.part = True

                if isinstance(p.geometry, sg.MultiPolygon):
                    if len(p.geometry.geoms) == 1:
                        p.geometry = sg.Polygon(p.geometry.geoms[0])
                    else:
                        p.valid_geom = False
                if p.geometry is not None and p.valid_geom is True:
                    p.create_polygon(p.geometry, self.geometries[k].points, p.name, p.crs, coord_decimals=8)

                geometry.polygons[p.name] = p
            self.geometries[k] = geometry

    def set_admin(self, admin):
        a: PlanAdmin
        for k, a in admin.items():
            geometry = self.geometries.get(k, Geometries())
            ad = Admin()
            ad.plan_number = a.survey_number
            ad.registration_date = a.reg_date
            ad.lga_code = a.lga_code
            ad.date_of_survey = a.survey_date

            ad.geometry = ad.set_geometry(self.geometries[k].polygons)
            ad.set_crs(self.geometries[k].polygons)
            geometry.admin = ad
            geometry.survey_number = a.survey_number
            geometry.survey_year = a.survey_date.year if a.survey_date is not None else None
            geometry.crs = 7844
            self.geometries[k] = geometry

    def set_point_positions(self, records, smes=False):
        for k, v in self.geometries.items():
            for i, point in v.points.items():
                point: PointGeom
                i = int(i)
                if i in records:
                    item = records.get(i)
                    if smes is True:
                        point.geometry = swkb.loads(item.coordinate.desc, hex=True)
                        point.ccc = True
                    else:
                        point.geometry = swkb.loads(item.geom.desc, hex=True)

    def get_plan_from_admin_id(self, admin_id):
        plan_number = self.plan_lookup.get(admin_id)
        return self.geometries.get(plan_number)


