import os
from pathlib import Path
import ezdxf
from ezdxf.addons import Importer
from ezdxf.enums import TextEntityAlignment
import csv
from domain.helpers.db_operations import create_database_connection
from utilities.dcmgeometrysdk.dcmgeometry.geometry import Geometries
from utilities.dcmgeometrysdk.dcmgeometry.points import PointGeom
from utilities.dcmgeometrysdk.dcmgeometry.polygons import PolygonGeom
from utilities.dcmgeometrysdk.landxml.landxml import parse
import shapely.geometry as sg
import shapely.ops as so

class GeomToSCFF:
    def __init__(self, geometries, dxf_version='AC1024', template='../resources/scff_template.dxf',
                 layer_location='../resources/scff_layers.csv'):
        self.geometries = geometries
        self.doc = ezdxf.new(dxf_version)
        self.model_space = self.doc.modelspace()
        self.layers = {}
        self.read_template(template)
        self.map_points()
        self.map_lines()
        self.map_natural_boundaries()
        self.map_polygons()
        #self.read_layers(layer_location)
        self.blah = None

    def read_template(self, template):
        sdoc = ezdxf.readfile(template)
        importer = Importer(sdoc, self.doc)
        importer.import_modelspace()
        importer.import_entities(sdoc.entities)
        importer.import_tables()
        importer.finalize()

    def read_layers_from_csv(self, layer_location):
        with open(layer_location, 'r', encoding='utf-8-sig') as open_csv:
            self.layers = [i for i in csv.DictReader(open_csv)]

    def define_add_layers(self):
        pass

    def map_points(self):
        points = self.geometries.points
        point_layers = {'reference': 'POINT-REFERENCE',
                        'control': 'POINT-CONTROL',
                        'traverse': 'POINT-TRAVERSE'}
        point_text_layers = {'control': 'POINT-CONTROL-ID'}
        for k, point in points.items():
            point: PointGeom

            ptype = point_layers.get(point.point_type)
            ptext = point_text_layers.get(point.point_type)
            con_type = ''
            text = None
            if point.point_type == 'control':
                hoz_pos = point.horizontal_position
                text = hoz_pos.oID
                if point.mon_type == 'PM':
                    con_type = '-PM'
                else:
                    con_type = '-PCM'

            if ptype is not None:
                layer = ptype + con_type
                geometry = (point.geometry.x, point.geometry.y)
                self.model_space.add_point(geometry, dxfattribs={"layer": layer})
                if text is not None:
                    self.model_space.add_text(text, dxfattribs={"layer": ptext}).set_placement(
                        geometry, align=TextEntityAlignment.BOTTOM_LEFT)


    def cut(self, line, distance, lines):
        # Cuts a line in several segments at a distance from its starting point
        if distance <= 0.0 or distance >= line.length:
            return [sg.LineString(line)]
        coords = list(line.coords)
        for i, p in enumerate(coords):
            pd = line.project(sg.Point(p))
            if pd == distance:
                return [
                    sg.LineString(coords[:i + 1]),
                    sg.LineString(coords[i:])
                ]
            if pd > distance:
                cp = line.interpolate(distance)
                lines.append(sg.LineString(coords[:i] + [(cp.x, cp.y)]))
                line = sg.LineString([(cp.x, cp.y)] + coords[i:])
                if line.length > distance:
                    self.cut(line, distance, lines)
                else:
                    lines.append(sg.LineString([(cp.x, cp.y)] + coords[i:]))
                return lines

    @staticmethod
    def set_line_boxes(geometry, distance, hp_bearing, hp_string_as_dms):

        if geometry.length > 5:
            length = geometry.length / 2
            first_cut = length - (len(str(distance)) / 2)
            second_cut = length + (len(str(distance)) / 2)
            point1 = geometry.interpolate(first_cut)
            point2 = geometry.interpolate(second_cut)
            dline = sg.LineString([point1, point2])

            first_cut = length - (len(hp_string_as_dms) / 2)
            second_cut = length + (len(hp_string_as_dms) / 2)
            point1 = geometry.interpolate(first_cut)
            point2 = geometry.interpolate(second_cut)
            bline = sg.LineString([point1, point2])
        else:
            dline = geometry
            bline = geometry

        if hp_bearing > 180:
            dline = dline.reverse()
            bline = bline.reverse()

        distance_line = dline.parallel_offset(1, side='left').coords[:]
        bearing_line = bline.parallel_offset(2, side='right').coords[:]
        return distance_line, bearing_line

    def map_lines(self):
        lines = {k: v for k, v in self.geometries.lines.items() if v.line_type != 'natural boundary'}
        line_layers = {'sideshot': 'LINE-RADIATION',
                       'normal': 'LINE-CONNECTION',
                       'traverse': 'LINE-TRAVERSE',
                       'natural boundary': 'LINE-NATURAL-BOUNDARY'}

        for k, v in lines.items():
            layer = line_layers.get(v.line_type)
            if layer is not None:
                geometry = v.geometry.coords[:]
                distance_line, bearing_line = self.set_line_boxes(v.geometry, v.distance,
                                                                  v.hp_bearing, v.hp_string_as_dms())

                point = v.geometry.centroid
                centroid = (point.x, point.y)

                self.model_space.add_lwpolyline(geometry, dxfattribs={"layer": layer})

                distance = f'{v.distance:.3f}'

                self.model_space.add_text(distance,
                                          dxfattribs={"layer": 'TEXT-CROWN-PORTION-ID'}).set_placement(
                    distance_line[0], distance_line[-1], align=TextEntityAlignment.ALIGNED)


                self.model_space.add_text(str(v.hp_string_as_dms()),
                                          dxfattribs={"layer": 'TEXT-CROWN-PORTION-ID'}).set_placement(
                    bearing_line[0], bearing_line[-1], align=TextEntityAlignment.ALIGNED)

    def map_natural_boundaries(self):
        lines = [i for i in self.geometries.lines.values() if i.line_type == 'natural boundary']
        line_group = {}
        for i in lines:
            lg = line_group.get(i.desc, [])
            lg.append(i)
            line_group[i.desc] = lg

        for k, v in line_group.items():
            mid_line = v[int(len(v)/2)]
            line_union = so.linemerge([i.geometry for i in v])
            if isinstance(line_union, (sg.LineString)):
                distance_line, bearing_lines = self.set_line_boxes(line_union, line_union.length,
                                                                   mid_line.hp_bearing, mid_line.hp_string_as_dms())
                self.model_space.add_lwpolyline(line_union.coords[:], dxfattribs={"layer": 'LINE-NATURAL-BOUNDARY'})
                self.model_space.add_text(str(k),
                                          dxfattribs={"layer": 'IRR-BNDRY-DESC'}).set_placement(
                    distance_line[0], distance_line[-1], align=TextEntityAlignment.ALIGNED)
            if isinstance(line_union, (sg.MultiLineString)):
                for i in line_union.geoms:
                    distance_line, bearing_lines = self.set_line_boxes(i, i.length,
                                                                       mid_line.hp_bearing, mid_line.hp_string_as_dms())
                    if isinstance(i, (sg.LineString)):
                        self.model_space.add_lwpolyline(i.coords[:], dxfattribs={"layer": 'LINE-NATURAL-BOUNDARY'})

                        # self.model_space.add_text(str(k),
                        #                           dxfattribs={"layer": 'IRR-BNDRY-DESC'}).set_placement(
                        #     i.coords[0], i.coords[-1], align=TextEntityAlignment.ALIGNED)



    def map_parcel_unmapped_parcel_lines(self):
        pass



    def map_polygons(self):
        lot_mapping = {'Lot': 'LOT', 'Easement': 'EASEMENT', 'Stage Lot': 'STAGE-LOT', 'Road': 'ROAD',
                       'Reserve': 'RESERVE', 'Created': 'CREATED', 'Common Property': 'COMMON-PROPERTY',
                       'Crown Land Service': 'CROWN-LAND-SERVICE', 'Crown Allotment': 'CROWN-ALLOTMENT',
                       'Crown Portion': 'CROWN-PORTION', 'Restriction':'RESTRICTION'}
        state_mapping = {'existing': 'EXISTING', 'encroached': 'ENCROACHED', 'Cancelled': 'CANCELLED',
                         'Affected': 'AFFECTED', 'extinguished': 'CANCELLED'}
        use_mapping = {'County': 'BOUNDARY-COUNTY', 'LGA': 'BOUNDARY-LGA', 'Parish': 'BOUNDARY-PARISH',
                       'Township': 'BOUNDARY-TOWNSHIP', 'State': 'BOUNDARY-STATE'}
        lot_text_mapping = {'Lot': 'TEXT-LOT-ID', 'Easement': 'TEXT-EASEMENT-LOT-ID', 'Stage Lot': 'TEXT-STAGE-LOT-ID',
                            'Road': 'TEXT-ROAD-ID', 'Reserve': 'TEXT-RESERVE-LOT-ID',
                            'Common Property': 'TEXT-COMMON-PROPERTY-ID',
                            'Crown Land Service': 'TEXT-CROWN-LAND-SERVICE-ID',
                            'Crown Allotment': 'TEXT-CROWN-ALLOTMENT-ID', 'Crown Portion': 'TEXT-CROWN-PORTION-ID',
                            'Restriction': 'TEXT-RESTRICTION-LOT-ID', }
        for k, v in self.geometries.polygons.items():
            if isinstance(v.geometry, sg.Polygon):
                pclass = lot_mapping.get(v.parcel_class)
                pstate = state_mapping.get(v.parcel_state, '')
                ptext = lot_text_mapping.get(v.parcel_class)
                if v.parcel_class in ['Administrative Area']:
                    pclass = use_mapping.get(v.parcel_use)
                if v.parcel_class in ['Crown Land Service', 'Easement', 'Administrative Area']:
                    pstate = ''
                elif v.parcel_class in ['Restriction'] and v.parcel_state == 'extinguished':
                    pstate = 'REMOVED'

                if pclass is not None:
                    layer_name = pclass + '-' + pstate
                    geometry = v.geometry.exterior.coords[:]
                    self.model_space.add_lwpolyline(geometry, dxfattribs={"layer": layer_name}, close=v.closed)
                    name = v.name.replace('\\', '/')
                    text = f'{name}'
                    point = v.geometry.representative_point()
                    centroid = (point.x, point.y)
                    a = (point.x - 5, point.y)
                    b = (point.x + 5, point.y)
                    if ptext is not None:
                        self.model_space.add_text(text, dxfattribs={"layer": ptext}).set_placement(
                        a, b, align=TextEntityAlignment.ALIGNED)

    def map_other(self):
        pass

    def save_doc(self, location='/Users/jamesleversha/Downloads/demo/imported.dxf'):
        self.doc.saveas(location)

#fn = '/Users/jamesleversha/Downloads/Natural Boundary-PS912183E.xml'
fn = '/Users/jamesleversha/Downloads/PS908461B.xml'
#fn = '/Users/jamesleversha/Downloads/PS905879E.xml'
data = parse(fn)
g = Geometries(data)


out_location = Path('/Users/jamesleversha/Downloads/demo', g.survey_number)
out_location.mkdir(exist_ok=True, parents=True)
out_scff = os.path.join(out_location, f'{g.survey_number}.dxf')

scff = GeomToSCFF(g)
scff.save_doc(location=out_scff)

g.write_geom_to_file(location=out_location)
g.write_geom_to_file(location=out_location, file_type='ESRI Shapefile')
