import json

class StandardDeviations:
    def __init__(self, profile_name, profile_loc, year=2000):
        self.profile_name = profile_name
        self.profile_location = profile_loc
        self.stds, self.ufv = self.read_profile_json()
        if year is None:
            year = 2000
        self.year = year

    def get_stds_for_year(self):
        return self.stds.get(self.year)

    def set_profile_name(self, name='dcmvr'):
        """profile name from the apriori json file located in resources"""
        self.profile_name = name
        return name

    def set_profile_location(self, profile_loc=None):
        if profile_loc is None:
            profile_loc = 'resources/aprioris.json'
        """sets the location of the apriori file if not default"""
        self.profile_location = profile_loc
        return profile_loc

    def read_profile_json(self, profile_name=None, profile_loc=None):
        """reads the apriori file"""
        def handle_items(vals, expanded=None):
            if expanded is None:
                expanded = {}
            for item in vals:
                measure_types = item.get('measure_types')
                min_year = item.get('min_year')
                max_year = item.get('max_year')
                for year in range(min_year, max_year):
                    ex = expanded.get(year, {})
                    for measure in measure_types:
                        types = measure.get('types', [])
                        values = measure.get('values')
                        for typed in types:
                            ex[typed] = values
                    expanded[year] = ex
            return expanded
        if profile_name is None:
            profile_name = self.profile_name
        if profile_loc is None:
            profile_loc = self.profile_location

        with open(profile_loc, 'r') as openjson:
            d = json.load(openjson)
            profile = d.get(profile_name)
            if profile is None:
                profile = d.get('default')

        defaults = profile.get('default_values', [])
        expanded_dict = handle_items(defaults)
        expanded_dict = handle_items(profile.get('values', {}), expanded_dict)

        if profile.get('use_file_values', 'True') == 'True':
            ufv = True
        else:
            ufv = False
        return expanded_dict, ufv

    def set_line_stds(self, lines):
        def get_std(line_type, distance, year_std, bord='bearing'):
            line_std = year_stds.get(line_type, year_std.get('default')).get(bord)
            for i in line_std:
                if i['min_dist'] <= distance < i['max_dist']:
                    return i['value']

        year_stds = self.get_stds_for_year()
        for k, value in lines.items():
            dir_line_type = value.azimuth_type
            dist_line_type = value.distance_type
            if value.bearing_std is None or self.ufv is False:
                value.bearing_std = get_std(dir_line_type, value.distance, year_stds)
            if value.distance_std is None or self.ufv is False:
                dist_std = get_std(dist_line_type, value.distance, year_stds, 'distance')
                ppm = get_std(dist_line_type, value.distance, year_stds, 'ppm')
                if ppm is None:
                    ppm = 0
                ppm_value = float(value.distance) * float(ppm) / 1000000.
                value.distance_std = float(dist_std) + ppm_value

        return lines









    # def get_stds_lists(self, value, default_stds):
    #     """sets all the direction and distance aprioris in a list of dicts ready to be written to file"""
    #     dir_line_type = value.azimuth_type
    #     dist_line_type = value.distance_type
    #     use_file_values = self.ufv
    #
    #     if use_file_values is True:
    #
    #         dir_stds = value.bearing_std
    #         if dir_stds is None:
    #             dir_items = default_stds.get(dir_line_type)
    #             if dir_items is not None:
    #                 dir_stds = dir_items.get('bearing')
    #             else:
    #                 dir_stds = default_stds.get('default').get('bearing')
    #                 dir_line_type = 'default'
    #         if not isinstance(dir_stds, list):
    #             dir_stds = [{'min_dist': 0, 'max_dist': 9999999, 'value': dir_stds}]
    #
    #         dist_stds = value.distance_std
    #         if dist_stds is None:
    #             dist_items = default_stds.get(dist_line_type)
    #             if dist_items is not None:
    #                 dist_stds = dist_items.get('distance')
    #             else:
    #                 dist_stds = default_stds.get('default').get('distance')
    #                 dist_line_type = 'default'
    #
    #         if not isinstance(dist_stds, list):
    #             dist_stds = [{'min_dist': 0, 'max_dist': 9999999, 'value': dist_stds}]
    #
    #         ppms = default_stds.get(dist_line_type)
    #         if ppms is None:
    #             ppms = default_stds.get('default').get('ppm')
    #         else:
    #             ppms = ppms.get('ppm')
    #         if not isinstance(ppms, list):
    #             ppms = [{'min_dist': 0, 'max_dist': 9999999, 'value': ppms}]
    #     else:
    #         dir_stds = default_stds.get(dir_line_type).get('bearing')
    #         dist_stds = default_stds.get(dist_line_type).get('distance')
    #         ppms = default_stds.get(dist_line_type).get('ppm')
    #     return dir_stds, dist_stds, ppms