from utilities.dcmgeometrysdk.landxml.landxml import CgPoint
from utilities.dcmgeometrysdk.landxml.landxml import RedHorizontalPosition, RedVerticalObservation
from utilities.dcmgeometrysdk.geometryfunctions.bearingdistancefunctions import calc_distance
import shapely.geometry as sg
from copy import deepcopy

class PointGeom:

    def __init__(self, point=None, monument=None, reduced_hoz_pos=None, reduced_vert_obs=None):
        """Class to hold Point Geometry, will hold original geometry and any updated geometries
            :param point: CgPoint object from a parsed landxml
            :param monument: Monument object from a parsed landxml that has the same point ref
            :param reduced_hoz_pos: ReducedHorizontalPosition object from a parsed landxml that has the same point ref
            :param reduced_vert_obs: ReducedVerticalObservation object from a parsed landxml that has the same point ref
            :type point: CgPoint
            :type monument: Monument
            :type reduced_hoz_pos: RedHorizontalPosition
            :type reduced_vert_obs: RedVerticalObservation"""

        self.original_crs = None
        self.crs = None
        self.mon_state = None
        self.mon_type = None
        self.mon_condition = None
        self.mon_desc = None
        self.mon_origin_survey = None
        self.horizontal_position = RedHorizontalPosition()
        self.control_name = None
        self.vertical_position = RedVerticalObservation()
        self.translated = False
        self.translated_x = 0
        self.translated_y = 0
        self.translated_z = 0
        self.ccc = False
        self.p_constrained = -1
        self.p_latitude_dms = None
        self.q_constrained = -1
        self.q_longitude_dms = None
        self.height = None
        if isinstance(point, CgPoint):
            self.name = point.name
            self.point_type = point.pntSurv
            self.point_oid = point.oID
            self.point_desc = point.desc
            self.associated_point_oid = None
            self.associated_point_distance = None
            self.point_state = point.state
            self.latitude = point.latitude
            self.longitude = point.longitude
            self.neh = point.valueOf_.split()
            if len(self.neh) > 2:
                self.height = self.neh[2]
            self.original_geom = self.get_original_geom()
            self.geometry = deepcopy(self.original_geom)
            if monument is not None:
                self.set_monument(monument)
            if reduced_hoz_pos is not None:
                self.horizontal_position = reduced_hoz_pos
                self.control_name = reduced_hoz_pos.desc
            if reduced_vert_obs is not None:
                self.vertical_obs = reduced_vert_obs
            
        else:
            self.name = None
            self.point_type = None
            self.point_oid = None
            self.point_desc = None
            self.associated_point_oid = None
            self.associated_point_distance = None
            self.point_state = None
            self.latitude = None
            self.longitude = None
            self.neh = None
            self.original_geom = None
            self.geometry = None

        if not isinstance(self.name, str):
            self.name = str(self.name)

    def set_monument(self, monument):
        """if available this will set the monument details to the point object, needs the monument to be supplied
        :param monument: pre linked monument object
        :type monument: Monument"""
        self.mon_state = monument.state
        self.mon_type = monument.type_
        self.mon_condition = monument.condition
        self.mon_desc = monument.desc
        self.mon_origin_survey = monument.originSurvey

    def get_original_geom(self):
        """gets the original geometry that was used when the class was created and returns a shapely geometry for it
        :returns : sg.Point"""
        neh = self.neh
        if None in neh:
            neh = [self.latitude, self.longitude]
            if len(self.neh) == 3:
                neh += [self.neh[2]]
        if len(neh) == 3:
            enh = [neh[1]] + [neh[0] + neh[2]]
        else:
            enh = [neh[1]] + [neh[0]]
        return sg.Point((float(x) for x in enh))

    def set_new_geometry(self, geometry):
        """sets a new geometry for the class, raises type error if not a shapely point object
        :param geometry: shapely point object that will replace the existing geometry
        :type geometry: sg.Point"""
        if isinstance(geometry, sg.Point):
            if self.original_geom is None:
                if self.geometry is not None:
                    self.original_geom = deepcopy(self.geometry)
                else:
                    self.original_geom = geometry
            self.geometry = geometry
        else:
            raise TypeError()

    def set_associated_id(self, oid, distance, set_ccc=False):
        """sets an associated id to this point, used when joining plans together with different identifiers
        :param oid: the oid of associated point
        :param distance: the distance between the associated points
        :type oid: int, float, string, tuple
        :type distance: float"""
        self.associated_point_oid = oid
        self.associated_point_distance = distance
        if set_ccc is True:
            self.ccc = True

    def get_distance_between_orig_new_coords(self):
        """return the distance between the orginal position and new position
        :returns : distance between the 2 objects"""
        return calc_distance(self.original_geom, self.geometry)

    def round_geometry(self, decimals=100):
        """rounds the geometry to a certain number of decimal points, useful for lats and longs with a large amount of
            decimal places
            :param decimals: number of decimal places to round each of the geometries to
            :type decimals: int"""
        geom = sg.Point((round(self.geometry.x, decimals), round(self.geometry.y, decimals)))
        self.set_new_geometry(geom)

    def get_long_lat(self):
        """returns the individual components of x and y in the geometry
        :returns : x, or long, and y, or long"""
        x = self.geometry.x
        y = self.geometry.y
        return x, y
