from utilities.dcmgeometrysdk.dcmgeometry.lines import LineGeom
from utilities.dcmgeometrysdk.dcmgeometry.arcs import ArcGeom
import shapely.geometry as sg
import numpy as np
from utilities.dcmgeometrysdk.geometryfunctions.bearingdistancefunctions import calc_bearing, calc_distance
from utilities.dcmgeometrysdk.geometryfunctions.arcfunctions import calc_arc_length_size_using_centre, GenerateArc
from utilities.dcmgeometrysdk.geometryfunctions.conversionsfunctions import chord2arc
from utilities.dcmgeometrysdk.landxml.landxml import Curve
from utilities.dcmgeometrysdk.landxml.landxml import LandXML


class ArcLineGeomFactory:

    def __init__(self):
        """"class to generate lines and arcs from a landxml, importantly creating the geometric shapes"""

        self.points = None
        self.is2point = None
        self.lines = None

    def get_line_geom(self, lxml_data, crs, swing=0):
        """ creates a LineGeom object from the landxml data
        :param lxml_data: parsed landxml file containing ReducedObeservations and ReducedArcObservations
        :param crs: coordinate reference system of the lines
        :param swing: swing to apply to the bearings
        :type lxml_data: LandXML
        :tyoe crs: int
        :type swing: float
        :returns : List of LineGeom objects created from the parsed landxml file"""
        lines = lxml_data.Survey[0].ObservationGroup[0].ReducedObservation
        return [LineGeom(line, self.points, self.is2point, crs, swing=swing) for line in lines]

    def get_arc_geom(self, lxml_data, crs, swing=0):
        """ creates a ArcGeom object from the landxml data
                :param lxml_data: parsed landxml file containing ReducedObeservations and ReducedArcObservations
                :param crs: coordinate reference system of the lines
                :param swing: swing to apply to the bearings
                :type lxml_data: LandXML
                :tyoe crs: int
                :type swing: float
                :returns : List of ArcGeom objects created from the parsed landxml file"""
        arcs = lxml_data.Survey[0].ObservationGroup[0].ReducedArcObservation
        return [ArcGeom(arc, self.points, self.is2point, crs, swing=swing) for arc in arcs]

    def find_missing_lines(self, lxml_data, crs=None, swing=0, plan_swing=0):
        """ finds missing lines that are within the parcel structure of a landxml but aren't necessarily captured as
            a reduced observation, this method generates a generated observation from the coordinates of the points
            that are used within the parcel structure, it is then stored in the line dictionary as an observation.
                :param lxml_data: parsed landxml file containing ReducedObeservations and ReducedArcObservations
                :param crs: coordinate reference system of the lines
                :param swing: swing to apply to the bearings
                :param plan_swing: apply overall existing plan_swing to file
                :type lxml_data: LandXML
                :tyoe crs: int
                :type swing: float
                """
        count = 0

        max_line_distance = 0
        pswing = 0
        for k, v in self.lines.items():
            v: LineGeom
            if v.line_type.lower() == 'boundary' and v.distance > max_line_distance:
                max_line_distance = v.distance
                pswing = v.difference_geom_v_actual()


        for polygon in lxml_data.Parcels[0].Parcel:
            if polygon is not None:
                if polygon.parcelType != 'Multipart':
                    if (polygon.class_ == 'Easement' and polygon.parcelFormat == 'Standard') is False and len(
                            polygon.CoordGeom) > 0:
                        if len(polygon.CoordGeom) > 0:
                            coord_geom = polygon.CoordGeom[0]
                            # lo = sorted([(x.polygon_index, x) for x in coord_geom.Line +
                            #       coord_geom.Curve + coord_geom.IrregularLine])
                            for x in coord_geom.Line + coord_geom.Curve + coord_geom.IrregularLine:
                                line = self.lines.get((x.Start.pntRef, x.End.pntRef))
                                if line is None:
                                    line = self.lines.get((x.End.pntRef, x.Start.pntRef))
                                    # handle reversed generated line could have multiple descriptions.....
                                    if line is not None:
                                        if line.distance_type == 'Generated':
                                            line = None
                                    if line is None:
                                        count += 1
                                        # pswing = l_points.get(x.Start.pntRef)
                                        # if pswing is None:
                                        #     pswing = l_points.get(x.End.pntRef)
                                        # if pswing is None:
                                        #     pswing = plan_swing
                                        # l_points[x.Start.pntRef] = pswing
                                        # l_points[x.End.pntRef] = pswing
                                        self.lines[(x.Start.pntRef,
                                                    x.End.pntRef)] = self.generate_line(x, count, crs,
                                                                                        swing, pswing)

    def generate_line(self, line, count, crs=None, swing=0, plan_swing=0):
        """:param line: ReducedArcObservation or ReducedObservation from the landxml
                :param count: arbitrary count of objects using this process for generated lines, used to assign an id
                                to generated lines, such as GENOBS-1
                :param crs: coordinate reference system of the lines
                :param swing: swing to apply to the bearings
                :type line: ReducedArcObservation, ReducedObservation
                :type count: int
                :tyoe crs: int
                :type swing: float
                :returns nl: a LineGeom, or ArcGeom object created from the landxml object"""
        sp = line.Start.pntRef
        ep = line.End.pntRef

        start = self.points.get(sp)
        end = self.points.get(ep)

        if isinstance(line, Curve):
            cp = line.Center.pntRef
            centre = self.points.get(cp)
            radius = line.radius
            rotation = line.rot
            chord = calc_distance(end.geometry, start.geometry)
            # work out big arc or small arc based on original values
            # and the centre point of the arc and rotation
            # build the arc using ArcGeom class
            nl = ArcGeom()
            nl.rot = rotation
            nl.radius = radius
            nl.setup_point = start
            nl.target_point = end
            nl.centre_point = centre
            nl.distance = chord
            nl.is_arc = True

            large = calc_arc_length_size_using_centre(start, end, centre, nl.rot)

            # add the calculated arc length here
            nl.arc_length = chord2arc(nl.distance, nl.radius, large)
            nl.geometry = GenerateArc(radius=nl.radius, arc_length=nl.arc_length, rot=nl.rot, distance=nl.distance,
                                      setup=start, target=end, centre=centre).geometry


        else:
            geom = [start.geometry.coords[:][0], end.geometry.coords[:][0]]
            nl = LineGeom()
            nl.geometry = sg.LineString(geom)
            nl.setup_point = start
            nl.target_point = end
            nl.is_arc = False

        if line.desc is not None:
            nl.line_type = line.desc
        else:
            nl.line_type = 'Generated'
        nl.crs = crs
        nl.is_nb = False
        nl.name = f'GENOBS-{count}'
        nl.azimuth_type = 'Generated'
        nl.distance_type = 'Generated'
        nl.create_bearing_distance_from_geometry()
        if swing != 0:
            nl.apply_swing(swing)
        if plan_swing != 0:
            nl.dd_bearing -= plan_swing
            nl.calc_hp_bearing()
        return nl

    def estimate_line_swing(self):
        differences = []
        for line in self.lines.values():
            line: LineGeom
            if line.line_type.lower() in ['boundary']:
                dist, dd = line.difference_geom_v_actual()
                differences.append(dd)
        return np.median(differences)


    def set_plan_features(self, plan_features, name='', pf_desc=''):
        pfs = []
        for pf in plan_features:
            polygon = LineGeom()
            polygon.name = pf.name
            polygon.class_ = name
            polygon.parcelFormat = 'Standard'
            polygon.state = 'Occupation'
            polygon.Center = []
            if isinstance(pf.CoordGeom, list):
                polygon.CoordGeom = pf.CoordGeom
            else:
                polygon.CoordGeom = [pf.CoordGeom]
            if len(pf.desc.strip()) > 0:
                polygon.desc = pf.desc
            else:
                polygon.desc = pf_desc
            polygon.parcelType = "Single"
            pfs.append(polygon)
        return pfs

    def build(self, lxml_data, point_geom, is2point, crs, swing=0) -> [LineGeom]:
        """"Creates a line dictionary with LineGeom objects from a parsed xml, it will store in order of entry
            it also stores arcs in here as they are esentially lines, only stores 1 line per setup and target
            direction as the lookup is a dict with (setup, target) as the key.
            :param lxml_data: parsed landxml file containing ReducedObeservations and ReducedArcObservations
            :param point_geom: already created dictionary containing PointGeom objects for building of line geometry
            :param is2point: lookup dictionary to convert instrument setups to their associated points
            :param crs: coordinate reference system of the lines
            :param swing: swing to apply to the bearings
            :type lxml_data: LandXML
            :type point_geom: dict
            :type is2point: dict
            :tyoe crs: int
            :type swing: float
            :returns self.lines: dictionary with tuples as key with the LineGeom objects as values, stored in self
                                 as well"""

        self.points = point_geom
        self.is2point = is2point
        lines = self.get_line_geom(lxml_data, crs, swing)
        lines += self.get_arc_geom(lxml_data, crs, swing)

        line_dict = {}
        for line in lines:
            s = line.setup_point.name
            e = line.target_point.name
            key = (s, e)
            exist = line_dict.get(key)
            if exist is not None:
                if (e, s) in line_dict.keys():
                    print('Line/Arc already stored in both directions, not storing line')
                    key = None
                else:
                    line.flip_direction()
                    key = (e, s)
            if key is not None:
                line_dict[key] = line
        self.lines = line_dict
        plan_swing = self.estimate_line_swing()
        self.find_missing_lines(lxml_data, crs, swing, plan_swing)
        return self.lines
