import math
from decimal import Decimal
import shapely.geometry as sg

from utilities.dcmgeometrysdk.landxml.landxml import ReducedArcObservation
from utilities.dcmgeometrysdk.dcmgeometry.lines import LineGeom
from utilities.dcmgeometrysdk.geometryfunctions.conversionsfunctions import hp2dd, arc2chord
from utilities.dcmgeometrysdk.geometryfunctions.bearingdistancefunctions import calc_bearing, calc_distance, \
    calc_inside_360, calc_new_point, metres2feet, metres2links
from utilities.dcmgeometrysdk.geometryfunctions.arcfunctions import GenerateArc

class ArcGeom(LineGeom, GenerateArc):
    def __init__(self, line=None, point_geom=None, is2point=None, crs=None, swing=0):
        """Class to store arc/curve details containing functions that allow generation of arc and proper conversions,
        inherits all items from LineGeom and GenerateArc classes
        :param line: ReducedArcObservation from a parsed landxml file or other source
        :param point_geom: dictionary of PointGeoms i.e. {id1: PointGeom, id2: PointGeom}
        :param is2point: lookup table to match instrument setups with point ids
        :param crs: coordinate reference system for the class
        :param swing: swing to apply to the line
        :type line: ReducedArcObservation or None
        :type point_geom: dict, None
        :type is2point: dict, None
        :type crs: int
        :type swing: float
        """

        super().__init__()
        self.is_arc = True
        self.reversed = False
        self.crs = crs
        self.centre_point = None
        self.swing_dd_bearing = None
        self.swing_hp_bearing = None
        self.field_notes = []
        self.transformer = None

        if isinstance(line, (ReducedArcObservation,)):
            self.get_field_notes(line)
            self.name = line.name
            self.setup_point = point_geom.get(is2point.get(line.setupID, {}))
            self.target_point = point_geom.get(is2point.get(line.targetSetupID, {}))
            self.hp_bearing = line.chordAzimuth
            self.dd_bearing = hp2dd(self.hp_bearing)
            self.apply_swing(swing)
            self.radius = line.radius
            self.arc_length = line.length
            self.rot = line.rot
            self.distance = arc2chord(self.arc_length, self.radius)
            self.geometry = self.calc_arc_values()
            self.distance_std = line.arcLengthAccuracy
            self.bearing_std = line.arcAzimuthAccuracy
            self.desc = line.desc
            self.line_type = line.purpose
            self.distance_type = line.arcType
            self.azimuth_type = line.arcType
            self.az_adopt_fact = line.azimuthAdoptionFactor
            self.dist_adopt_fact = line.lengthAdoptionFactor
            self.adopted_az_survey = line.adoptedSurvey
            self.adopted_dist_survey = line.adoptedSurvey
            self.is_nb = self.check_nb()

        else:
            self.radius = None
            self.arc_length = None
            self.rot = None

    def radius2feet(self):
        """converts the radius of the arc to feet from metres
        :returns : converted value"""
        if isinstance(self.radius, (float, int, Decimal)):
            return metres2feet(self.radius)

    def radius2links(self):
        """converts the radius of the arc to links from metres
        :returns : converted value"""
        if isinstance(self.radius, (float, int, Decimal)):
            return metres2links(self.radius)

    def arclength2feet(self):
        """convers the arclength to feet from metres
        :returns : converted value"""
        if isinstance(self.arc_length, (float, int, Decimal)):
            return metres2feet(self.arc_length)

    def arclength2links(self):
        """convers the arclength to links from metres
        :returns : converted value"""
        if isinstance(self.arc_length, (float, int, Decimal)):
            return metres2links(self.arc_length)

    def calc_arc_geometry_from_start_end(self):
        """creates an arc geometry from the start and end points using the stored values in the class for radius etc."""
        self.geometry = self.calc_arc_values()


