from pathlib import Path
from datetime import datetime
import boto3
import os
import json
from copy import deepcopy
from itertools import combinations

from utilities.dcmgeometrysdk.geometryfunctions.otherfunctions import sort_by_tuple
from utilities.dcmgeometrysdk.geometryfunctions.conversionsfunctions import dd2hp
from utilities.dcmgeometrysdk.geometryfunctions.transformationfunctions import transform_geoms
from utilities.dcmgeometrysdk.geometryfunctions.bearingdistancefunctions import calc_inside_360
from utilities.dcmgeometrysdk.dna.standard_deviations import StandardDeviations


def get_dms(b1):
    d, ms = str(b1).split('.')
    ms = ms.ljust(4, '0')
    m = ms[:2]
    s = ms[2:4]
    if len(ms) > 4:
        millis = ms[4:8]
    else:
        millis = '0'
    millis = millis.ljust(2, '0')

    return d, m, s, millis


class AprioriValues:
    def __init__(self, name, b, d, ppm):
        self.name = name
        self.bearing_std = b
        self.distance_std = d
        self.ppm = ppm


class DNADirectionSets:
    def __init__(self, ip=None, to_stations=None, lines=None, correlate=False, points=None, angles=False, angle_std=1,
                 db_id=''):
        self.ip = ip
        self.stations = to_stations
        self.correlate = correlate
        self.lines = lines
        self.angle_std = angle_std
        self.db_id = db_id
        if points is not None:
            self.points = points
        else:
            self.points = {}
        self.line_sets = {}

        if self.ip is None or self.lines is None or self.stations is None:
            self.direction_set = []
        else:
            self.order_stations()
            self.direction_set = self.generate_direction_set()
            if angles is True:
                self.direction_set += self.generate_angles_from_stations()

    def order_stations(self):
        stations = []
        for item in self.stations:
            line1 = self.lines.get((self.ip, item))
            if line1 is not None:
                self.line_sets[(self.ip, item)] = line1
            else:
                line1 = deepcopy(self.lines.get((item, self.ip)))
                line1.flip_direction()
                self.line_sets[(self.ip, item)] = line1
            stations.append((line1, line1.dd_bearing))
        self.stations = [i[0] for i in sorted(stations, key=lambda x: x[1])]

    def generate_direction_set(self):
        directions = []
        c2 = None
        if self.correlate is False:
            dset_length = 1
        else:
            dset_length = len(self.stations) - 1
        for item in self.stations:
            if item.setup_point.name == self.ip:
                target = item.target_point.name
            else:
                target = item.setup_point.name
            order = self.stations.index(item)
            if order == 0:
                c2 = self.points.get(target).ccc
            direction = DNADirection(self.ip, target, dset_length, item.bearing_std, item.hp_bearing, order=order,
                                     db_id=self.db_id)

            if c2 is False or self.points.get(self.ip).ccc is False or self.points.get(target).ccc is False:
                if self.correlate is False:
                    if len(directions) > 0 and order != 1:
                        directions.append(directions[0])
                    directions.append(direction)
                else:
                    directions.append(direction)
        if len(directions) < 2:
            directions = []
        return directions

    def generate_angles_from_stations(self, important_angles=(0, 90, 180, 270)):
        angles = []
        for a, b in combinations(self.line_sets, 2):
            if a[1] != self.ip and b[1] != self.ip:
                l1 = self.line_sets[a]
                l2 = self.line_sets[b]
                if l1.dd_bearing > l2.dd_bearing:
                    angle = calc_inside_360(l2.dd_bearing - l1.dd_bearing)
                else:
                    angle = l2.dd_bearing - l1.dd_bearing
                # print(round(angle, 2))
                if round(angle, 2) in important_angles or important_angles is None:
                    if self.points.get(a[1]).ccc is False or self.points.get(self.ip).ccc is False or \
                            self.points.get(b[1]).ccc is False:
                        angled = DNAAngle(self.ip, a[1], b[1], round(angle, 2), std=self.angle_std)
                        angles.append(angled)
        return angles

    def add_direction_to_dir_set(self, direction):
        self.direction_set.append(direction)


class DNAPQMeasure:
    def __init__(self, station_name=None, coordinate=None, std=None, measure='P', ignore='', db_id=''):
        self.station_name = station_name
        self.coordinate = coordinate
        self.std = std
        self.measure = measure
        self.ignore = ignore
        self.db_id = db_id

    def create_line_text(self):
        line = (self.measure.rjust(1) + self.ignore.ljust(1) + str(self.station_name)[:20].ljust(20) + ''.ljust(40) +
                str(self.coordinate)[:14].ljust(14) + ''.ljust(14) + str(self.std)[:9].rjust(9) + ''.rjust(43) +
                str(self.db_id).rjust(10))
        return line


class DNAGXYMeasure:
    def __init__(self, station=None, x=None, y=None, z=None, xcovs=None, ycovs=None, zcovs=None,
                 second_station_name='', ignore=''):

        if xcovs is None:
            xcovs = []
        if ycovs is None:
            ycovs = []
        if zcovs is None:
            zcovs = []

        self.station = station
        self.xcovs = self.ensure_list(xcovs)
        self.xyz = self.pad_list([x, y, z], left=False)
        self.ycovs = self.pad_list(self.ensure_list(ycovs))
        self.zcovs = self.pad_list(self.ensure_list(zcovs))
        self.second_station_name = second_station_name

        self.ignore = ignore

    def pad_list(self, value, left=True):
        if left is True:
            return ([''] * (len(self.xcovs) - len(value))) + value
        else:
            return value + ([''] * (len(self.xcovs) - len(value)))

    def ensure_list(self, item):
        if not isinstance(item, list):
            item = [item]
        return item


class DNAGXYMeasureSet:
    def __init__(self, stations=None, measure_type='Y', reference_frame='ITRF2014', v_scale=1., p_scale=1., l_scale=1.,
                 h_scale=1., coordsys='XYZ', epoch=datetime.now().date(), db_id=''):
        if stations is None:
            stations = []
        self.measure_type = measure_type
        self.stations = stations
        self.v_scale = v_scale
        self.p_scale = p_scale
        self.l_scale = l_scale
        self.h_scale = h_scale
        self.cluster_count = len(self.stations)
        self.epoch = epoch.strftime('%d.%m.%Y')
        self.db_id = db_id
        self.reference_frame = reference_frame
        self.coordsys = coordsys
        self.lines = []

    def add_gxy_measure(self, measure):
        self.stations.append(measure)

    def create_line_text(self):
        lines = []
        for c, gyxmeasure in enumerate(self.stations):
            gyxmeasure: DNAGXYMeasure
            if c == 0:
                if self.measure_type == 'Y':
                    columns2342 = str(self.coordsys)[:20].ljust(20)
                else:
                    columns2342 = str(gyxmeasure.second_station_name)[:20].ljust(20)
                line = (self.measure_type.rjust(1) +
                        gyxmeasure.ignore.rjust(1) +
                        str(gyxmeasure.station)[:20].ljust(20) +
                        columns2342 +
                        str(len(self.stations)).ljust(20) +
                        '{:.4f}'.format(float(self.v_scale))[:10].rjust(10) +
                        '{:.4f}'.format(float(self.p_scale))[:10].rjust(10) +
                        '{:.4f}'.format(float(self.l_scale))[:10].rjust(10) +
                        '{:.4f}'.format(float(self.h_scale))[:10].rjust(10) +
                        str(self.reference_frame).rjust(20) +
                        self.epoch.rjust(20) +
                        str(self.db_id).rjust(20))
            else:
                line = (self.measure_type.rjust(1) +
                        gyxmeasure.ignore.rjust(1) +
                        str(self.stations[c])[:20].ljust(20))

            lines.append(line)

            for count, value in enumerate(gyxmeasure.xcovs):
                x_cov = value
                xyz_value = gyxmeasure.xyz[count]
                y_cov = gyxmeasure.ycovs[count]
                z_cov = gyxmeasure.zcovs[count]
                line = (''.ljust(62) +
                        str(xyz_value).rjust(20) +
                        str(x_cov)[:20].rjust(20) +
                        str(y_cov)[:20].rjust(20) +
                        str(z_cov)[:20].rjust(20))
                lines.append(line)
        return lines


class DNAAngle:
    def __init__(self, ip, stn1, stn2, angle, std):
        self.ip = ip
        self.stn2 = stn1
        self.stn3 = stn2
        self.angle = dd2hp(angle, rnd=True)
        self.std = std

    def write_line(self):
        d, m, s, millis = get_dms(self.angle)
        row = ('A'.rjust(1) +
               ''.rjust(1) +
               str(self.ip)[:20].ljust(20) +
               str(self.stn2)[:20].ljust(20) +
               str(self.stn3)[:20].ljust(20) +
               ''.rjust(14) +
               d.rjust(3) + ' ' +
               m.ljust(2) +
               (' ' + s + '.' + millis).ljust(8) +
               str(self.std).rjust(9))
        return row


class DNADirection:
    def __init__(self, ip, to_station, dset_length, std, hp_bearing, order, ignore='', db_id=''):
        self.ip = ip

        self.to_station = to_station
        self.hp_bearing = hp_bearing
        self.db_id = db_id
        if not isinstance(std, (float, int)):
            try:
                std = float(std)
            except ValueError as err:
                std = 60
        self.std = '{:.7f}'.format(float(std))
        if len(self.std) > 9:
            self.std = self.std[:9]
        self.order = order
        self.dset_length = dset_length
        self.ignore = ignore

    def write_line(self):
        d, m, s, millis = get_dms(self.hp_bearing)
        if self.order == 0:
            c1 = self.ip
            c2 = self.to_station
            c3 = self.dset_length
        else:
            c1 = ''
            c2 = ''
            c3 = self.to_station

        row = ('D'.rjust(1) +
               self.ignore.rjust(1) +
               str(c1)[:20].ljust(20) +
               str(c2)[:20].ljust(20) +
               str(c3)[:20].ljust(20) +
               ''.rjust(14) +
               d.rjust(3) + ' ' +
               m.ljust(2) +
               (' ' + s + '.' + millis).ljust(8) +
               self.std.rjust(9) +
               ''.rjust(43) +
               str(self.db_id).rjust(10))

        return row


class DNADistances:
    def __init__(self, line=None, ins_height=0, targ_height=0, db_id='', ignore=''):
        if line is not None:
            self.distance = line.distance
            self.dist_std = line.distance_std
            self.setup = line.setup_point.name
            self.target = line.target_point.name
        else:
            self.distance = None
            self.dist_std = None
            self.setup = None
            self.target = None
        self.ignore = ignore
        self.ins_height = ins_height
        self.targ_height = targ_height
        self.db_id = db_id

    def set_distance_from_values(self, distance, dist_std, setup, target, db_id=''):
        self.distance = distance
        self.dist_std = dist_std
        self.setup = setup
        self.target = target
        if len(str(db_id)) > 0:
            self.db_id = db_id

    def write_line(self):
        line = ('S'.rjust(1)
                + self.ignore.ljust(1)
                + str(self.setup)[:20].ljust(20)
                + str(self.target)[:20].ljust(20)
                + ''.ljust(20)
                + ('{:.4f}'.format(float(self.distance))).rjust(14)
                + ''.ljust(14)
                + ('{:.4f}'.format(float(self.dist_std)).rjust(9))
                + ('{:.4f}'.format(float(self.ins_height)).rjust(7))
                + ('{:.4f}'.format(float(self.targ_height)).rjust(7))
                + ''.rjust(29)
                + str(self.db_id).rjust(10))
        return line


class DNABearing:
    def __init__(self, line, std):
        self.setup = line.setup_point.name
        self.target = line.target_point.name
        self.bearing = line.hp_bearing
        self.std = std

    def write_line(self):
        d, m, s, millis = get_dms(self.bearing)
        line = ('B'.rjust(1)
                + ''.ljust(1)
                + str(self.setup)[:20].ljust(20)
                + str(self.target)[:20].ljust(20)
                + str('')[:20].ljust(20) +
                ''.rjust(14) +
                d.rjust(3) + ' ' +
                m.ljust(2) +
                (' ' + s + '.' + millis).ljust(8) +
                str(self.std).rjust(9))
        return line


class DNAWriters:
    def __init__(self, geometries=None, output_dir=Path(), filename=None, in_datum=None,
                 out_datum=7844, survey_year=None, survey_type='surveyed', constrained_marks=None, ignored_stns=None,
                 profile_location=None, profile='dcmvr', stds=None,
                 ignore_types=('Ignored', 'BranchConnection', 'GraphGenerated', 'calculated'),
                 use_ignored_for_angle=False, ignore_ccc=False):
        geometries = deepcopy(geometries)

        if in_datum is None:
            if geometries is not None:
                self.in_datum = geometries.crs
            else:
                self.in_datum = 7844
        else:
            self.in_datum = in_datum

        self.bearing_constraints = []
        self.angle_constraints = []

        if geometries is not None:
            self.points = geometries.points  # dict of PointGeom items taken from the dcmgeometry class
            self.lines = geometries.lines
            self.polygons = geometries.polygons# dict of bearings and distances from LineGeom dcmgeometry class
        else:
            self.points = {}
            self.lines = {}
            self.polygons = {}

        if filename is None and geometries is not None:
            filename = geometries.survey_number
        self.filename = str(filename)
        self.out_path = output_dir

        self.out_datum = out_datum
        self.exclude_lines = set()
        if survey_year is not None:
            self.survey_year = survey_year
        else:
            if geometries is not None:
                self.survey_year = geometries.survey_year
            if self.survey_year is None:
                self.survey_year = 2000
        self.use_ignored_for_angle = use_ignored_for_angle
        self.survey_type = survey_type
        self.ignore_types = ignore_types
        self.ignored_obs = self.get_ignored_obs(ignored_stns)
        self.constrained_marks = None
        self.ignore_ccc = ignore_ccc
        self.set_constrained(constrained_marks)
        self.min_distance = None
        self.bearing_std = None
        self.include_bearings = self.set_bearing_constraints()
        self.correlate = self.set_correlate()
        self.angle_std = None
        self.include_angles = self.set_include_angles()
        if stds is None:
            stds = StandardDeviations(profile, profile_location, self.survey_year)
        self.lines = stds.set_line_stds(self.lines)
        self.obs, self.dist_obs = self.group_obs()
        self.distances = []
        self.create_msr_dist_lines()
        self.direction_sets = []
        self.create_msr_dir_lines()
        self.pq_obs = []
        self.gxy_obs = []
        self.create_msr_p_q_constraints()
        self.block_id = None
        self.zone_id = None
        self.most_connected_point = None

    def set_correlate(self, correlate=False):
        """this sets the dynadjust grouping of directions to correlate so that when the file is written you will only
        get one set of directions per set rather than all directions available for that station in a group"""
        self.correlate = correlate
        return correlate

    def set_include_angles(self, include_angles=False, angle_std=1):
        self.include_angles = include_angles
        self.angle_std = angle_std
        return include_angles

    def set_bearing_constraints(self, include_bearings=False, distance=.2, std=324000):
        self.include_bearings = include_bearings
        self.min_distance = distance
        self.bearing_std = std
        return include_bearings

    def get_ignored_obs(self, ignored_obs=None):
        # TODO this needs testing
        """this is used if there are ignored stations, these will not be used in
        the adjustment """
        new_ignored = set()

        if ignored_obs is not None:
            new_ignored = {i for i in ignored_obs}
        else:
            ignored_obs = set()
        used = set()
        for k, v in self.lines.items():
            if v.azimuth_type in self.ignore_types or v.distance_type in self.ignore_types:
                if k[0] not in used:
                    new_ignored.add(k[0])
                if k[1] not in used:
                    new_ignored.add(k[1])
            else:
                used.add(k[0])
                used.add(k[1])
                if k[0] in new_ignored and k[0] not in ignored_obs:
                    new_ignored.remove(k[0])
                if k[1] in new_ignored and k[1] not in ignored_obs:
                    new_ignored.remove(k[1])

        return new_ignored

    def set_constrained(self, constrained_marks=None):
        """sets the constrained mark names that the user want to make constrained (CCC)"""
        if constrained_marks is None:
            constrained_marks = set()
            for k, v in self.points.items():
                if v.ccc is True and self.ignore_ccc is False:
                    constrained_marks.add(k)
            self.constrained_marks = constrained_marks
        elif isinstance(constrained_marks, set):
            self.constrained_marks = constrained_marks
        return constrained_marks

    def add_angle_constraint(self, stn1, stn2, stn3, angle, std, refresh_angles=False):
        if refresh_angles is True:
            self.angle_constraints = []
        angle = DNAAngle(stn1, stn2, stn3, angle, std)
        self.angle_constraints.append(angle)

    def create_bearing_constraints(self, refresh_bearings=True):
        if refresh_bearings is True:
            self.bearing_constraints = []
        lines = []
        for line in self.lines.values():
            if line.distance < self.min_distance:  # and line.setup_point.ccc is False and line.target_point.ccc is False:
                b = DNABearing(line, self.bearing_std)
                self.bearing_constraints.append(b)
                lines.append(b.write_line())
        return lines

    def set_exclude(self, line_names=None):
        """manually exclude some names from the adjustment, these are set here with a set of line names"""
        if line_names is None:
            line_names = set()
        self.exclude_lines = line_names

    def create_stn_header(self):
        """
        creates the dna station header line, this contains the header information required by dynadjust to
        run a minimally constrained adjustment
        :param number_of_stations: the number of stations that are going to be written into the file
        :return: stn: the station header
        """
        # dt1 = '01.01.1994'
        if self.out_datum == 7844:
            g = 'GDA'
        else:
            g = 'GDA'

        dt = datetime.strftime(datetime.now(), '%d.%m.%Y')
        header = ('!#=DNA'.rjust(6) +
                  '3.01'.rjust(6) +
                  'STN'.rjust(3) +
                  dt.rjust(14) +
                  f'{g}2020'.rjust(14) +
                  '01.01.2020'.rjust(14) +
                  (str(len(self.points))).rjust(10))
        return [header]

    def create_stn_lines(self):
        """
        creates the dna station file data lines, is a list that contains pre-formatted strings that will be written to
        the stn file, setting the elevation at zero for all items
        :return: a list of items to be writen as lines in the dynadjust stn file.
        """
        stn = []
        elevation = '0'
        projection = 'UTM'
        zone = str(self.in_datum)[-2:]
        if self.in_datum is None:
            self.in_datum = 7855
        if self.out_datum == 7844:
            projection = 'LLH'
            zone = ''

        if self.in_datum != self.out_datum:
            self.points = transform_geoms(self.points, self.in_datum, self.out_datum)
            self.in_datum = self.out_datum

        if self.constrained_marks is None:
            pnts = {pt: v for pt, v in self.points.items() if v.point_type == 'sideshot' and pt not in self.ignored_obs}
            constrained = set(sorted(pnts.keys())[0])
        else:
            constrained = self.constrained_marks

        # pnts = {pt: v for pt, v in self.points.items() if pt not in self.ignored_obs}
        for pt, val in self.points.items():
            # if val.associated_point_oid is not None:
            #     pt = val.associated_point_oid

            if pt in constrained and self.ignore_ccc is False:
                const = 'CCC'
            else:
                const = 'FFC'

            x = val.geometry.x
            y = val.geometry.y
            # set to vic
            if x < 100000 and projection != 'LLH':
                x += 250000
                y += 5900000

            if projection == 'LLH':
                y = dd2hp(float(y))
                x = dd2hp(float(x))
                e = "{0:.14f}".format(y).rjust(20)
                n = "{0:.14f}".format(x).rjust(20)
            else:
                e = "{0:.3f}".format(float(x)).rjust(20)
                n = "{0:.3f}".format(float(y)).rjust(20)

            line = (str(pt)[:20].ljust(20)  # Pt ID
                    + const.ljust(3)
                    + ''.ljust(1)  # Constraint
                    + projection.ljust(3)  # Projection
                    + e  # Easting
                    + n  # Northing
                    + elevation.rjust(20)  # Elevation
                    + zone.rjust(3))  # Hemisphere/Zone input
            if const == 'CCC':
                stn = [line] + stn
            else:
                stn.append(line)
        return stn

    def write_stn_file_encode(self, return_lines=False):
        """
        creates a binary linestring for dna stn format groups the measurements and adds the estimates that can be written
        to a file in AWS
        :return:
        """
        lines = self.create_stn_header()
        lines += self.create_stn_lines()

        line_string = ''
        for line in lines:
            line_string += line + "\n"

        line_string = line_string.encode('UTF-8')
        if return_lines is False:
            return line_string
        else:
            return lines

    def write_stn_file(self, filename=None, out_path=None, lines=None):
        """Writes the station file for Dynadjust"""
        if out_path is None:
            out_path = Path(self.out_path)
        else:
            out_path = Path(out_path)
        if filename is None:
            filename = self.filename
            if '\\' in self.filename:
                filename = filename.replace("\\", "_")
            if '/' in self.filename:
                filename = filename.replace("/", "_")
        out_path.mkdir(parents=True, exist_ok=True)
        out_path = Path(out_path, str(filename) + '.stn')
        if lines is None:
            lines = self.write_stn_file_encode()
            lines = lines.decode('UTF-8')
        elif isinstance(lines, list):
            line_string = ''
            for line in lines:
                line_string += line + "\n"
            lines = line_string
        with open(out_path, 'w') as stn_file:
            stn_file.write(lines)
        return out_path

    def group_obs(self):
        """groups all the observations in the file to the setup points"""
        obs = {}
        dist_obs = {}
        k_alt = {}
        all_obs = {}
        for k, v in self.lines.items():
            all_obs[k[0], k[1]] = v
            x = deepcopy(v)
            x.flip_direction()
            all_obs[k[1], k[0]] = x

        for k, v in all_obs.items():
            if v.line_type not in self.exclude_lines:
                if v.azimuth_type not in self.ignore_types:
                    exist_obs = obs.get(k[0], [])
                    exist_obs.append((k[1]))
                    obs[k[0]] = exist_obs
                else:
                    k_alt[k[0]] = k[1]
                if v.distance != 0:
                    if v.distance_type not in self.ignore_types and v.reversed is False:
                        dist_obs[k] = v

        if self.use_ignored_for_angle is True:
            dir_obs = {}
            for k, v in obs.items():
                if len(v) == 1:
                    extra = k_alt.get(k[0])
                    if extra is not None:
                        v.append(extra)
                        dir_obs[k] = v
                elif len(v) > 1:
                    dir_obs[k] = v
        else:
            dir_obs = {k: v for k, v in obs.items() if len(v) > 1}
        return dir_obs, dist_obs

    def add_directions_to_direction_sets(self, direction: DNADirection):
        if self.direction_sets is None:
            self.direction_sets = DNADirectionSets()
        self.direction_sets.add_direction_to_dir_set(direction)

    def create_msr_dir_lines(self, refresh_dir_set=True):
        if refresh_dir_set is True:
            self.direction_sets = []
        """creates the list of measurement direction lines to write into the measurement Dynadjust file"""
        # line_lookups = {k:v for k,v in line_lookups.items() if v.get('bearing_type' != 'Ignored')}
        lines = []

        for k, v in self.obs.items():
            # sort to get clockwise angle order
            d = DNADirectionSets(k, to_stations=v, lines=self.lines, correlate=self.correlate, points=self.points,
                                 angle_std=self.angle_std)
            self.direction_sets.append(d)

    def add_distance_to_distances(self, distance: DNADistances):
        self.distances.append(distance)

    def add_angle_to_angles(self, angle: DNAAngle):
        self.angle_constraints.append(angle)

    def add_gyx_measures_to_writer(self, measures: [DNAGXYMeasureSet]):
        self.gxy_obs.extend(measures)

    def create_msr_dist_lines(self, ins_height=0, targ_height=0, refresh_distances=True):
        """creates a list of measurement distance lines for the dynadjust file"""
        if refresh_distances is True:
            self.distances = []
        lines = []
        for k, v in self.dist_obs.items():
            line = DNADistances(v, ins_height, targ_height)
            self.distances.append(line)

    def create_msr_p_q_constraints(self, refresh_pqs=True):
        if refresh_pqs is True:
            self.pq_obs = []
        for k, v in self.points.items():

            if v.p_constrained >= 0:
                latitude = v.p_latitude_dms
                if latitude is None:
                    latitude = dd2hp(v.latitude)
                constraint = DNAPQMeasure(v.name, latitude, v.p_constrained)
                self.add_p_q_constraint(constraint)
            if v.q_constrained >= 0:
                longitude = v.q_longitude_dms
                if longitude is None:
                    longitude = dd2hp(v.longitude)
                constraint = DNAPQMeasure(v.name, longitude, v.q_constrained, 'Q')
                self.add_p_q_constraint(constraint)

    def add_p_q_constraint(self, constraint):
        self.pq_obs.append(constraint)

    def create_msr_header(self):
        """Creates the measurement file header needed for a dynadjust file"""
        obscount = len(self.direction_sets)
        obscount += len(self.distances)
        obscount += len(self.angle_constraints)
        obscount += len(self.pq_obs)
        obscount += len(self.gxy_obs)
        now = datetime.now()
        date = now.strftime('%d.%m.%Y')
        date2 = '01.01.2020'
        g = 'GDA'
        header = ('!#=DNA'.rjust(6) +
                  '3.01'.rjust(6) +
                  'MSR'.rjust(3) +
                  date.rjust(14) +
                  f'{g}2020'.rjust(14) +
                  date2.rjust(14) +
                  str(obscount).rjust(10))

        return [header]

    def write_msr_file_encode(self, return_lines=False):
        """
        creates a binary linestring for dna msr format groups the measurements and adds the estimates
        that can be written to a file in AWS
        """
        lines = self.create_msr_header()
        for d in self.direction_sets:
            if len(d.direction_set) > 1:
                for direction in d.direction_set:
                    lines.append(direction.write_line())
        for line in self.distances:
            lines.append(line.write_line())
        if self.include_angles is True:
            for f in self.direction_sets:
                angles = f.generate_angles_from_stations()
                for a in angles:
                    lines.append(a.write_line())

        if len(self.angle_constraints) > 0:
            for a in self.angle_constraints:
                lines.append(a.write_line())

        if self.include_bearings is True:
            lines += self.create_bearing_constraints()

        lines += [c.create_line_text() for c in self.pq_obs]
        for c in self.gxy_obs:
            for line in c.create_line_text():
                lines.append(line)
        line_string = ''
        for line in lines:
            line_string += line + "\n"

        line_string = line_string.encode('UTF-8')
        if return_lines is False:
            return line_string
        else:
            return lines

    def write_msr_file(self, filename=None, out_path=None, lines=None):
        if out_path is None:
            out_path = Path(self.out_path)
        else:
            out_path = Path(out_path)

        if filename is None:
            filename = self.filename
            if '\\' in self.filename:
                filename = filename.replace("\\", "_")
            if '/' in self.filename:
                filename = filename.replace("/", "_")
        out_path.mkdir(parents=True, exist_ok=True)
        out_path = Path(out_path, str(filename) + '.msr')
        if lines is None:
            lines = self.write_msr_file_encode()
            lines = lines.decode('UTF-8')
        elif isinstance(lines, list):
            line_string = ''
            for line in lines:
                line_string += line + "\n"
            lines = line_string
        with open(out_path, 'w') as msr_file:
            msr_file.write(lines)
        return out_path

    def write_stn_msr_encode(self, return_lines=False):
        """writes out both station and measurement files to a variable rather than file"""
        stn_lines = self.write_stn_file_encode(return_lines)
        msr_lines = self.write_msr_file_encode(return_lines)
        return stn_lines, msr_lines

    def write_stn_msr_file(self, filename=None, out_path=None, stn_lines=None, msr_lines=None):
        """writes out both station and measurement files"""
        stn_file = self.write_stn_file(filename, out_path, lines=stn_lines)
        msr_file = self.write_msr_file(filename, out_path, lines=msr_lines)
        return stn_file, msr_file

    def write_stn_msr_to_bucket(self, bucket_name=None, key_loc=''):
        if bucket_name is not None:
            stn_lines, msr_lines = self.write_stn_msr_encode()
            s3r = boto3.resource('s3')

            slog = os.path.join(key_loc, self.filename + '.stn')
            obj = s3r.Object(bucket_name, slog)
            obj.put(Body=stn_lines)

            mlog = os.path.join(key_loc, self.filename + '.msr')
            obj = s3r.Object(bucket_name, mlog)
            obj.put(Body=msr_lines)

            return self.filename + '.stn', self.filename + '.msr'

    def write_stn_geopackage(self, geopackage_name=None):
        pass

    def write_msr_geopackage(self, geopackage_name=None):
        pass

    def list_pq_stations(self, p_and_q_only=True):
        pq_stations = {}
        for item in self.pq_obs:
            item: DNAPQMeasure
            pq = pq_stations.get(item.station_name, set())
            pq.add(item.measure)
            pq_stations[item.station_name] = pq

        if p_and_q_only is True:
            return {k: len(v) for k, v in pq_stations.items() if len(v) > 1}
        else:
            return {k: len(v) for k, v in pq_stations.items()}

    def list_y_stations(self):
        y_stations = set()
        for item in self.gxy_obs:
            if item.measure_type == 'Y':
                for i in item.stations:
                    y_stations.add(i)

        return y_stations

    def keep_measures(self, kept_stations=None, remove_measures=None):

        if kept_stations is None:
            kept_stations = set()
        if remove_measures is None:
            remove_measures = set()
        mks = set()
        for item in kept_stations.keys():
            try:
                item = int(item)
            except ValueError:
                pass
            mks.add(item)

        if len(mks) > 0:
            new_distances = [i for i in self.distances if (i.setup in mks and i.target in mks)
                             and ((i.setup, i.target) not in remove_measures or
                                  (i.target, i.setup) not in remove_measures)]
            self.distances = new_distances

            new_pqs = [i for i in self.pq_obs if i.station_name in mks]
            self.pq_obs = new_pqs

            self.angle_constraints = [i for i in self.angle_constraints if (i.ip in mks and i.stn2 in mks and
                                                                            i.stn3 in mks)]

            new_dirs = []
            for i in self.direction_sets:
                d_set = []
                for d in i.direction_set:
                    if d.ip in mks and d.to_station in mks:
                        if (d.ip, d.to_station) not in remove_measures and (d.to_station, d.ip) not in remove_measures:
                            d_set.append(d)

                i.direction_set = d_set

                if len(i.direction_set) > 1:
                    count = 0
                    for item in i.direction_set:
                        item.order = count
                        item.dset_length = len(i.direction_set) - 1
                        count += 1
                    i.ip = i.direction_set[0].ip
                    new_dirs.append(i)
            self.direction_sets = new_dirs

            new_y_vals = []
            for i in self.gxy_obs:
                kept = []
                for stn in i.stations:
                    if stn in mks:
                        kept.append(stn)
                if len(kept) == len(i.stations):
                    new_y_vals.append(i)
            self.gxy_obs = new_y_vals
