import logging
import os
import platform as pf
from inspect import getsourcefile
import subprocess
from subprocess import PIPE
from pathlib import Path
import warnings

from utilities.dcmgeometrysdk.dna.dnawriters import DNAWriters


class DNARunner:
    def __init__(self, dna_dir, geoid_fp=None, multi_thread=False, max_iter=10,
                 iter_thresh=.0005, output_dir=None, filename=None, cwd=None, docker=False, mount_dir=None,
                 host_directory=None, container_name='DNA', apu=False, convert_station_heights=True,
                 outputmsr_to_stn=False, export_dna_files=False, seach_nearby_stn=False,
                 nearby_stn_buffer=-1, search_similar_msr=False, ignore_similar_msr=False,
                 remove_similar_msr=False, include_stns_assoc_msrs=tuple(),
                 exclude_stns_assoc_msrs=tuple(), confidence_int=95, inner_stations=150, outer_stations=150,
                 output_db_ids=False, connected_segment=None, number_of_networks=None, number_of_blocks=None,
                 exclude_insufficient_measures=False):
        if cwd is None:
            self.cwd = Path(os.path.split(os.path.abspath(getsourcefile(lambda: 0)))[0])
        else:
            self.cwd = cwd
        self.system_type = pf.system()
        if docker is True:
            self.system_type = 'docker'

        self.dna_dir = dna_dir
        self.max_iterations = max_iter
        self.iter_threshold = iter_thresh
        self.inner_stations = inner_stations
        self.outer_stations = outer_stations
        self.connected_segment = connected_segment
        self.number_of_networks = number_of_networks
        self.number_of_blocks = number_of_blocks
        self.output_dir = self.set_output_dir(output_dir)
        if mount_dir is not None:
            self.mount_dir = Path(str(output_dir).replace(str(host_directory), mount_dir))
        else:
            self.mount_dir = output_dir
        self.host_directory = host_directory
        self.docker = docker
        if docker is True:
            self.docker_commands = ['/usr/local/bin/docker', 'exec', '-w',
                                    f'{self.mount_dir}', '-i', f'{container_name}']
            self.docker_prefix = f'/usr/local/bin/docker exec -w {self.mount_dir} -it {container_name} '
        else:
            self.docker_commands = []
            self.docker_prefix = ''

        if filename.endswith('.xml'):
            filename = filename.replace('.xml', '')
        self.filename = filename
        self.prefix = ''
        self.suffix = ''
        self.ext = ''
        self.convert_stn_heights = convert_station_heights
        self.geoid_fp = self.set_geoid_location(geoid_fp)
        self.dms_msr_format = None
        self.sep = ''
        self.multi_thread = multi_thread
        self.set_run_info()

        self.dna_import = self.set_dna_fps('import')
        self.dna_reftran = self.set_dna_fps('reftran')
        self.dna_adjust = self.set_dna_fps('adjust')
        self.dna_plot = self.set_dna_fps('plot')
        self.dna_segment = self.set_dna_fps('segment')
        self.dna_geoid = self.set_dna_fps('geoid')
        self.execution_lines = []
        self.dms_msr_format = 2
        self.adj_type = '.simult.adj'
        self.stn_coord_types = 'ENzPLHh'
        self.confidence_int = str(confidence_int)
        if apu is False:
            self.output_type = '--output-adj-msr'
            self.apu = None
        else:
            self.output_type = '--output-adj-msr'
            self.apu = '--output-pos-uncertainty'
        self.cmd_output = None
        self.results_file = None

        # add more dna settings here.
        if outputmsr_to_stn is False:
            self.outputmsr_to_stn = ''
        else:
            self.outputmsr_to_stn = '--output-msr-to-stn'
        if export_dna_files is False:
            self.export_dna_files = ''
        else:
            self.export_dna_files = '--export-dna-files'
        if seach_nearby_stn is False:
            self.search_nearby_stn = ''
        else:
            self.search_nearby_stn = '--search-nearby-stn'
        if nearby_stn_buffer < 0:
            self.nearby_stn_buffer = ''
        else:
            self.nearby_stn_buffer = f'--nearby-stn-buffer {nearby_stn_buffer}'

        if search_similar_msr is False:
            self.search_similar_msr = ''
        else:
            self.search_similar_msr = '--search-similar-msr'

        if ignore_similar_msr is False:
            self.ignore_similar_msr = ''
        else:
            self.ignore_similar_msr = f'--ignore-similar-msr'

        if remove_similar_msr is False:
            self.remove_similar_msr = ''
        else:
            self.remove_similar_msr = f'--remove-similar-msr'

        if len(include_stns_assoc_msrs) == 0:
            self.include_stns_assoc_msrs = ''
        else:
            self.include_stns_assoc_msrs = f"--include-stns-assoc-msrs {','.join(include_stns_assoc_msrs)}"

        if len(exclude_stns_assoc_msrs) == 0:
            self.exclude_stns_assoc_msrs = ''
        else:
            self.exclude_stns_assoc_msrs = f"--exclude-stns-assoc-msrs {','.join(exclude_stns_assoc_msrs)}"
        if output_db_ids is True:
            self.db_ids = '--output-database-ids'
        else:
            self.db_ids = ''
        if exclude_insufficient_measures is True:
            self.exclude_insufficent_measures = '--ignore-insufficient-msrs'
        else:
            self.exclude_insufficent_measures = ''
    def set_dna_fps(self, module):
        fp = Path(self.dna_dir, self.prefix + module + self.suffix)
        if fp.exists():
            return fp
        elif self.docker is True:
            return fp
        else:
            print(f'{module} is not present at {fp}')
            return fp


    def set_output_dir(self, output_dir=None):
        if output_dir is None:
            # warnings.showwarning(f'No output directory set, setting to {self.cwd}')
            return self.cwd
        else:
            return Path(output_dir)

    def set_stns_msrs(self, items):
        if isinstance(items, list):
            if self.docker is True:
                n_items = [Path(str(stn).replace(str(self.output_dir), str(self.mount_dir))) for stn in items]
            else:
                n_items = []
                for stn in items:
                    if str(stn).startswith(str(self.output_dir)):
                        stn = Path(str(stn).replace(str(self.output_dir), ''))
                        n_items.append(stn)
                    else:
                        n_items.append(Path(stn))
            fn = n_items[0]
            n_items = [str(i) for i in n_items]
            n_item = ' '.join(n_items)
        else:
            if self.docker is True:
                n_items = Path(str(items).replace(str(self.output_dir), str(self.mount_dir)))
            fn = n_items
            n_item = n_items


        return fn, n_item, n_items

    def build_execution_cmd(self, msr_locations, stn_locations, only_import=False):
        def gen_import(fn, msr_location_, stn_location_, insuff=False):
            linea = f'''{self.docker_prefix}"{str(self.dna_import)}" -n {fn} {str(stn_location_)} '''
            linea += f'''{str(msr_location_)} -r GDA2020 {self.outputmsr_to_stn} {self.export_dna_files} '''
            linea += f'''{self.search_nearby_stn} {self.nearby_stn_buffer} {self.search_similar_msr} '''
            linea += f'''{self.ignore_similar_msr} {self.remove_similar_msr} '''
            linea += f'''{self.include_stns_assoc_msrs} {self.exclude_stns_assoc_msrs} '''
            if insuff:
                linea += f'''{self.exclude_insufficent_measures}'''
            linea = linea.strip()
            return linea

        fn, msr_location, msr_locations = self.set_stns_msrs(msr_locations)
        fn, stn_location, stn_locations = self.set_stns_msrs(stn_locations)

        if self.filename is None:
            self.filename = fn.parts[-1].replace('.stn', '')
            # warnings.showwarning(f'No filename set, setting to {self.filename}')
        if self.docker is False:
            lines = [f'cd "{self.output_dir}"']

        else:
            lines = []
        temp_msr = []
        temp_stn = []
        if len(self.connected_segment) > 0:

            temp_name = self.filename + '_temp'
            line = gen_import(temp_name, msr_location, stn_location)
            lines.append(line)
            segments = [str(i) for i in self.connected_segment]
            starting_stns = ','.join(segments)
            line = (f'{self.docker_prefix}"{str(self.dna_segment)}" -n "{temp_name}" --min-inner-stns ' +
                    f'{150} --max-block-stns {65535} --starting-stns {starting_stns}, --contiguous-blocks 0')
            lines.append(line)

            if self.number_of_networks is None and self.number_of_blocks is None:
                print('Setting number of networks to 1 as we need to process something')
                self.number_of_networks = 1

            if self.number_of_networks is not None:
                for i in range(self.number_of_networks):
                    line = f'{self.docker_prefix}"{self.dna_import}" -n "{temp_name}" ' \
                           f'--import-contiguous-stn-msr {i} --export-dna-files'
                    lines.append(line)
                    msr = os.path.join(self.output_dir, temp_name + f'.network-{i}.msr')
                    stn = msr[:-4] + '.stn'
                    temp_msr.append(msr)
                    temp_stn.append(stn)

            elif self.number_of_blocks is not None:
                for i in range(self.number_of_blocks):
                    line = f'{self.docker_prefix}"{self.dna_import}" -n "{temp_name}" ' \
                           f'--import-block-stn-msr {i + 1} --export-dna-files'
                    lines.append(line)
                    msr = os.path.join(self.output_dir, temp_name + f'.block-{i + 1}.msr')
                    stn = msr[:-4] + '.stn'
                    temp_msr.append(msr)
                    temp_stn.append(stn)

        insuf = False
        if len(str(self.exclude_insufficent_measures)):
            insuf = True

        if len(temp_msr) > 0:
            fn, temp_msr_location, temp_msr_locations = self.set_stns_msrs(temp_msr)
            fn, temp_stn_location, temp_stn_locations = self.set_stns_msrs(temp_stn)
            line = gen_import(self.filename, temp_msr_location, temp_stn_location, insuff=insuf)

        else:
            line = gen_import(self.filename, msr_location, stn_location, insuff=insuf)
        lines.append(line)
        if only_import is False:
            if self.geoid_fp is not None:
                if self.convert_stn_heights is True:
                    lines.append(f'{self.docker_prefix}'
                                 f'"{str(self.dna_geoid)}" -n "{self.filename}" '
                                 f'-g "{str(self.geoid_fp)}"')
                else:
                    lines.append(f'{self.docker_prefix}"{str(self.dna_geoid)}" '
                                 f'-n "{self.filename}" -g "{str(self.geoid_fp)}"')
            else:
                pass
                # warnings.showwarning('No geoid file is set')

            if self.multi_thread is True:
                mt = '--multi-thread'
                lines.append(f'{self.docker_prefix}"{str(self.dna_segment)}" -n "{self.filename}" --min-inner-stns '
                             f'{self.inner_stations} --max-block-stns {self.outer_stations}')
                self.adj_type = '.phased-mt.adj'
            else:
                mt = ''
                self.adj_type = '.simult.adj'
            apu = self.apu
            if apu is None:
                apu = ''
            lines.extend([
                f'{self.docker_prefix}"{str(self.dna_adjust)}" -n "{self.filename}" {self.output_type} {apu} '
                f'--dms-msr-format {str(self.dms_msr_format)} --stn-coord-types {self.stn_coord_types} '
                f'--stn-corrections '
                f'--max-iterations {self.max_iterations} --conf-interval {self.confidence_int} '
                f'--iteration-threshold {self.iter_threshold} {self.db_ids} {mt}'])
        self.execution_lines = lines
        return lines

    def determine_output(self):
        if self.docker is True:
            output_dir = self.mount_dir
        else:
            output_dir = self.output_dir
        return output_dir

    def write_execution_cmd_to_file(self, filename_suffix=''):

        if len(filename_suffix) > 0:
            filename_suffix = '_' + filename_suffix
        self.cmd_output = Path(self.output_dir, self.filename + filename_suffix + self.ext)
        if len(self.execution_lines) > 0:
            with open(self.cmd_output, 'w') as open_txt:
                for line in self.execution_lines:
                    open_txt.write(line + '\n')

    def execute_cmd(self):
        if self.cmd_output is not None:
            if self.cmd_output.exists():
                subprocess.call([self.cmd_output], cwd=self.output_dir)
                self.set_results()

    def check_non_empty_string(self, item, some_list):
        if len(item) > 0:
            some_list.append(item)
        return some_list

    def run_dna_via_subprocesses(self, msr_locations, stn_locations, only_import=False):

        lines = self.build_execution_cmd(msr_locations, stn_locations, only_import)
        cmd_list = []
        for line in lines:
            cmds = line.split(' ')
            cmds = [i.strip(' ').replace('"', '') for i in cmds
                    if len(i.strip(' ').replace('"', '')) > 0 and i != '-it']

            cmd_list.append(cmds)
        cmd_list = [cmd for cmd in cmd_list if cmd[0] != 'cd']
        if only_import is True:
            if self.connected_segment:
                cmd_list = cmd_list[:4]
            else:
                cmd_list = cmd_list[:2]
        for cmd in cmd_list:
            print(str(cmd))
            p = subprocess.run(cmd, cwd=self.output_dir, stdin=PIPE, stdout=PIPE, stderr=PIPE)
            print(p.stdout.decode('utf-8').strip())
            print(p.stderr.decode('utf-8').strip())
            # cmd

        if only_import is False:
            results = self.set_results()
        else:
            results = None
        return results

    def set_results(self):
        results = Path(str(self.output_dir), self.filename + self.adj_type)
        if results.exists is False:
            results = None
            # warnings.showwarning("No results file, adjustment hasn't worked for some reason")
        self.results_file = results
        return results

    def set_geoid_location(self, geoid_fp):
        if geoid_fp is None:
            geoid_fp = Path(self.dna_dir, 'geoid_file', 'AUSGeoid2020_20180201.gsb')
        if geoid_fp.exists() is False:
            if self.docker is True:
                self.geoid_fp = geoid_fp
            else:
                geoid_fp = Path(self.dna_dir, 'geoid_file', 'AUSGeoid2020_20180201.gsb')
                if self.docker is False:
                    if geoid_fp.exists() is False:
                        pass
                    # warnings.showwarning('Could not find geooid file')
        return geoid_fp

    def set_run_info(self):
        if self.system_type.lower() == 'windows':
            self.ext = '.bat'
            self.sep = '&'
            self.suffix = '.exe'
            # self.directory = 'windows1064'
        elif self.system_type.lower() == 'darwin':
            self.ext = '.sh'
            self.prefix = 'dna'
            # self.directory = 'darwin'
        else:
            self.ext = '.sh'
            self.prefix = 'dna'
            # self.directory = 'linux'
