from utilities.dcmgeometrysdk.geometryfunctions.misclosefunctions import loop_checker, get_likely_candy
from utilities.dcmgeometrysdk.geometryfunctions.conversionsfunctions import dd2hp, hp2dd
from utilities.dcmgeometrysdk.geometryfunctions.bearingdistancefunctions import calc_inside_360
from shapely.geometry import MultiLineString
from shapely.ops import linemerge
import statistics
import math

class Loop:
    def __init__(self, loop, lines, likely=False):
        """class to store a single loop object will hold the geometry of the loop and if its a likely candidate for
            error
            :param loop: dictionary of loop items
            :param lines: dictionary of LineGeom objects
            :param likely: set the loop if its a likely candidate object
            :type loop: dict
            :type lines: dict
            :type likely: bool"""
        self.loop = loop
        self.geometry = self.set_geometry(lines)
        self.likely_candidate = likely

    def set_geometry(self, lines):
        """sets the geometry of the object, uses the geometries in lines to generate a multi-line object for all
            lines in a loop.
          :param lines: dictionary of LineGeom objects
          :type lines: dict"""
        loop_lines = []
        for line in lines.values():
            if line.name in self.loop:
                if line.geometry is not None:
                    loop_lines.append(line.geometry)

        return MultiLineString(loop_lines)


class Loops:
    def __init__(self, key, loop, lines):

        self.loop = loop.get('loop')
        self.likely_names = loop.get('likely')
        self.distances = loop.get('distances')
        self.average_distance = self.set_average_distance()
        self.angles = loop.get('angles')
        self.average_angle = self.set_average_angle()
        self.misclose_tolerance = loop.get('mis_tol')
        self.loops = self.set_individual_loops(lines)
        self.likely = self.set_likely(lines)
        self.geometry = self.set_geometry(lines)
        self.crs = self.set_crs_from_first_line(lines)
        # group for loop error distance estimated based on the tolerance set to a factor of 10
        self.group_value = key
        self.misclose_category = self.set_misclose_category()


    def set_average_distance(self):
        if len(self.distances) > 0:
            return statistics.mean(self.distances)

    def set_average_angle(self):
        angle_sin_sum = 0
        angle_cos_sum = 0
        for angle in self.angles:
            angle = hp2dd(angle)
            angle_sin_sum += math.sin(math.radians(angle))
            angle_cos_sum += math.cos(math.radians(angle))

        angle_sum = math.degrees(math.atan2((angle_sin_sum/len(self.angles)), (angle_cos_sum/len(self.angles))))
        return round(dd2hp(calc_inside_360(round(angle_sum, 4))), 4)

    def set_misclose_category(self):
        low_tol = self.misclose_tolerance
        mean_dist = statistics.mean(self.distances)
        high_tol = low_tol + .5
        if low_tol <= mean_dist < high_tol:
            return 'WARNING'
        if mean_dist >= high_tol:
            return 'FAIL'
        else:
            return 'PASS'

    def set_crs_from_first_line(self, lines):
        crs = None
        for k, v in lines.items():
            crs = v.crs
            break
        return crs

    def set_individual_loops(self, lines):
        loops = []
        for l in self.loop:
            loops.append(Loop(l, lines))
        return loops

    # just set the geometry and turn it into a multiline
    def set_geometry(self, lines):
        loop_lines = []
        loops = []
        for l in self.loop:
            for i in l:
                loops.append(i)

        for k, v in lines.items():
            if v.name in loops:
                loop_lines.append(v.geometry)
        return MultiLineString(loop_lines)

    # likely cadidates dict of lines
    def set_likely(self, lines):
        likely_lines = []
        for item in self.likely_names:
            likely_lines.append(Loop([item], lines, likely=True))
        return likely_lines

    # return a merged geometry of the loop
    def get_merged_loop_geometry(self):
        return linemerge(self.geometry)

