import math
import shapely.geometry as sg
from copy import deepcopy
from utilities.dcmgeometrysdk.landxml.landxml import ReducedObservation
from utilities.dcmgeometrysdk.geometryfunctions.conversionsfunctions import hp2dd, dd2hp
from utilities.dcmgeometrysdk.geometryfunctions.bearingdistancefunctions import flip_bearing, calc_bearing, \
                                                                                calc_distance, metres2feet, \
                                                                                metres2links, calc_inside_360
from utilities.dcmgeometrysdk.geometryfunctions.transformationfunctions import transform_coordinates, estimate_zone


class LineGeom:
    def __init__(self, line=None, point_geom=None, is2point=None, crs=None, swing=0):
        """Class to store line details containing functions conversions and swinging and change of direction of lines,
                :param line: ReducedArcObservation from a parsed landxml file or other source
                :param point_geom: dictionary of PointGeoms i.e. {id1: PointGeom, id2: PointGeom}
                :param is2point: lookup table to match instrument setups with point ids
                :param crs: coordinate reference system for the class
                :param swing: swing to apply to the line
                :type line: ReducedObservation or None
                :type point_geom: dict, None
                :type is2point: dict, None
                :type crs: int
                :type swing: float
                """
        self.is_arc = False
        self.reversed = False
        self.crs = crs
        self.likely_candidate = False
        self.overlapping = False
        self.swing_dd_bearing = None
        self.swing_hp_bearing = None
        self.rot = None
        self.field_notes = []

        if isinstance(line, ReducedObservation):
            self.get_field_notes(line)
            self.name = line.name
            self.distance = line.horizDistance
            self.hp_bearing = line.azimuth
            self.dd_bearing = hp2dd(self.hp_bearing)
            self.apply_swing(swing)
            self.distance_std = line.distanceAccuracy
            self.orig_distance_std = line.distanceAccuracy
            self.bearing_std = line.azimuthAccuracy
            self.orig_bearing_std = line.azimuthAccuracy
            self.equipment_used = line.equipmentUsed
            self.setup_point = point_geom.get(is2point.get(line.setupID, {}))
            self.target_point = point_geom.get(is2point.get(line.targetSetupID, {}))
            self.geometry = self.generate_transformed_geom()
            self.line_type = line.purpose
            self.distance_type = line.distanceType
            self.azimuth_type = line.azimuthType
            self.az_adopt_fact = line.azimuthAdoptionFactor
            self.dist_adopt_fact = line.distanceAdoptionFactor
            self.adopted_az_survey = line.adoptedAzimuthSurvey
            self.adopted_dist_survey = line.adoptedDistanceSurvey
            self.is_nb = self.check_nb()
            self.desc = line.desc


        else:
            self.name = None
            self.distance = None
            self.hp_bearing = None
            self.dd_bearing = None
            self.distance_std = None
            self.orig_distance_std = None
            self.bearing_std = None
            self.orig_bearing_std = None
            self.setup_point = None
            self.target_point = None
            self.geometry = None
            self.line_type = None
            self.distance_type = None
            self.azimuth_type = None
            self.az_adopt_fact = None
            self.dist_adopt_fact = None
            self.adopted_az_survey = None
            self.adopted_dist_survey = None
            self.equipment_used = None
            self.is_nb = None
            self.desc = None

    def get_field_notes(self, line):
        "sets any field notes that are in the Reduced observation object into a list"
        for item in line.FieldNote:
            self.field_notes.append(item.valueOf_)

    def apply_swing(self, swing=0):
        "applies swing to the decimal degrees bearing and the dms bearing"
        self.swing_dd_bearing = calc_inside_360(self.dd_bearing + swing)
        self.swing_hp_bearing = dd2hp(self.swing_dd_bearing, rnd=True)

    def check_nb(self):
        """applied a natural boundary flag if the line start or finish point object is set to natural boundary
        :returns is_nb: A true or false value if the line is a natural boundary"""
        if self.setup_point.point_type == 'natural_boundary' and \
                self.target_point.point_type == 'natural_boundary':
            is_nb = True
        else:
            is_nb = False
        return is_nb

    # bearing types must be one of 'hp, dd'
    def update_bearings(self, bearing=None, bearing_type=None):
        """updates the bearing based on a value supplied, user must supply a string value of either hp, or dd
        to indicate which bearing should be updated.
        :param bearing: the bearing value to which the user wants updated
        :param bearing_type: string value of either hp, or dd
        :type bearing: float, None
        :type bearing_type: str"""
        if bearing is not None:
            if bearing_type == 'hp':
                self.hp_bearing = bearing
                self.dd_bearing = hp2dd(self.hp_bearing)
            elif bearing_type == 'dd':
                self.dd_bearing = bearing
                self.hp_bearing = dd2hp(self.dd_bearing, rnd=True)

    def generate_transformed_geom(self):
        """:returns : a sg.Linestring from start and end points"""
        return sg.LineString([self.setup_point.geometry, self.target_point.geometry])

    def flip_direction(self):
        """flips the direction of the line in place, updates all relevant information to completely flip the line,
        including start and end points"""
        t = deepcopy(self.target_point)
        self.target_point = self.setup_point
        self.setup_point = t
        self.hp_bearing = flip_bearing(self.hp_bearing)
        self.calc_dd_bearing()
        self.geometry = sg.LineString(reversed(self.geometry.coords[:]))
        if self.rot == 'ccw':
            self.rot = 'cw'
        elif self.rot == 'cw':
            self.rot = 'ccw'
        else:
            self.rot = None
        self.reversed = True

    def create_bearing_distance_from_geometry(self):
        """if the geometry is set calculate the bearing and distance values from the geometry itself other than
            specified values does this on VicGrid datum if input crs is Geographic GDA2020 or 94"""
        distance, dd_bearing = self.bearing_distance_from_geometry()
        self.distance = distance
        self.dd_bearing = dd_bearing
        self.calc_hp_bearing()

    def bearing_distance_from_geometry(self):
        geom = deepcopy(self.geometry)
        if self.crs in [7844, 4283]:
            out_datum = estimate_zone(self.setup_point.geometry)
            geom = transform_coordinates(geom, in_datum=self.crs, out_datum=out_datum)
        coords = geom.coords
        distance = calc_distance(sg.Point(coords[0]), sg.Point(coords[-1]))
        dd_bearing = math.degrees(calc_bearing(sg.Point(coords[-1]), sg.Point(coords[0])))
        dd_bearing = calc_inside_360(dd_bearing)
        return distance, dd_bearing

    def difference_geom_v_actual(self):
        dist_geom, dd_geom = self.bearing_distance_from_geometry()
        dif_dd = dd_geom - self.dd_bearing
        dif_dist = dist_geom - self.distance
        return dif_dist, dif_dd

    def create_line_from_coords(self, setup_point=None, target_point=None, coords=None, name=None,
                                line_type=None, crs=None):
        """generates a line from a line string or a set of coordinates, it will then populate the other fields from
            these values, if a setup point and target point are supply then will populate these as well, this line will
            have its distance and azi type values always set to calculated.
            :param setup_point: PointGeom object that represents the start of the line
            :param target_point: PointGeom object that reporsents the end of the line
            :param coords: coords can be shapely LineString or a list of coordinates
            :param name: the name of the line
            :param line_type: specified line type
            :param crs: epsg integer of coordinate reference system of the line
            :type setup_point: PointGeom, None
            :type target_point: PointGeom, None
            :type coords: sg.LineString, list, None
            :type name: str, None
            :line_type: str, None
            :crs: int, None"""
        if name is not None:
            self.name = name
        self.line_type = line_type
        if setup_point is not None:
            self.setup_point = setup_point
        if target_point is not None:
            self.target_point = target_point
        if crs is None:
            crs = self.setup_point.crs
        self.crs = crs
        self.calc_geometry_from_start_end(coords=coords)
        self.create_bearing_distance_from_geometry()
        self.distance_type = 'calculated'
        self.azimuth_type = 'calculated'

    def calc_hp_bearing(self):
        """calculates the hp notation (dms) bearing from the decimal bearing roundaed to DDD.MMSS"""
        self.hp_bearing = dd2hp(self.dd_bearing, rnd=True)

    def calc_dd_bearing(self):
        """caclulates the decimal degrees bearing from hp"""
        self.dd_bearing = hp2dd(self.hp_bearing)

    def calc_geometry_from_start_end(self, coords=None):
        """calculates the geometry of the line from the setup and target points if no coords input is included,
        otherwise will use supplied values to generate the line geometry
        :param coords: either a list of coordinates or a sg.LineString which is used to generate a line, if a linestring
                        then will just store that supplied value
        :type coords: list, sg.LineString, None"""
        if coords is None:
            coords = sg.LineString([self.setup_point.geometry, self.target_point.geometry])
        if isinstance(coords, sg.LineString):
            self.geometry = coords
        else:
            self.geometry = sg.LineString(coords)

    def distance2feet(self):
        """converts the distance value to feet from metres
        :returns : distance value in feet"""
        return metres2feet(self.distance)

    def distance2links(self):
        """converts the distance value to links from metres
               :returns : distance value in links"""
        return metres2links(self.distance)

    def transform_geom(self, out_crs):
        """transforms geometry to a new coordinate system
        :param out_crs: epsg value of the new crs
        :type out_crs: int"""
        self.geometry = transform_coordinates(self.geometry, in_datum=self.crs, out_datum=out_crs)
        self.crs = out_crs

    def hp_string_as_dms(self):
        d, ms = str(self.hp_bearing).split('.')
        m, s = ms[:2], ms[2:].ljust(2, '0')
        return d + u'\N{DEGREE SIGN}' + m + "'" + s + '"'



    # def generate_original_geom(self, point_geom):
    #     s = point_geom.get(self.setup_point).original_geom
    #     t = point_geom.get(self.target_point).original_geom
    #     if t is not None and s is not None:
    #         return sg.LineString([s, t])





