import base64
import inspect
import types

import networkx as nx
import statistics
from datetime import date
from datetime import datetime
import shapely.affinity as sa
import shapely.ops as so
from shapely.prepared import prep
import os
from itertools import combinations
from io import StringIO, BytesIO
from copy import deepcopy

from pathlib import Path
from utilities.dcmgeometrysdk.geometryfunctions.transformationfunctions import transform_geoms, build_transformer, \
    transform_coordinates, estimate_zone
from utilities.dcmgeometrysdk.geometryfunctions.transformationfunctions import helmert_transformation_with_ids
from utilities.dcmgeometrysdk.geometryfunctions.bearingdistancefunctions import *
from utilities.dcmgeometrysdk.geometryfunctions.conversionsfunctions import dd2hp
from utilities.dcmgeometrysdk.geometryfunctions.otherfunctions import previous_and_next, chunker
from utilities.dcmgeometrysdk.dcmgeometry.factories.polygonfactory import PolygonGeomFactory
from utilities.dcmgeometrysdk.dcmgeometry.surveygraph import SurveyGraph
from utilities.dcmgeometrysdk.dcmgeometry.factories.arclinefactory import ArcLineGeomFactory
from utilities.dcmgeometrysdk.dcmgeometry.factories.pointfactory import PointGeomFactory
from utilities.dcmgeometrysdk.dcmgeometry.lines import LineGeom
from utilities.dcmgeometrysdk.dcmgeometry.arcs import ArcGeom
from utilities.dcmgeometrysdk.dcmgeometry.admin import Admin
from utilities.dcmgeometrysdk.dcmgeometry.loops import Loops
from utilities.dcmgeometrysdk.dcmgeometry.polygons import PolygonGeom
from utilities.dcmgeometrysdk.landxml.landxml import LandXML, CgPoint, InstrumentSetup, RedVerticalObservation, \
    InstrumentPoint, ReducedObservation, ParcelType, \
    RedHorizontalPosition, Parcels, SurveyHeader, ObservationGroup, CoordinateSystem
from utilities.dcmgeometrysdk.geometryfunctions.misclosefunctions import loop_checker


class Geometries:
    """main class that handles survey and plan information, generally populated by a landxml
        but can be populated other ways.
        :param landxml: a parsed landxml file
        :param mis_tol: non-mandatory misclose tolerance that is applied throughout the class
        :type landxml: LandXML
        :type mis_tol: int
        """
    def __init__(self, landxml=None, mis_tol=.1):
        self.landxml = landxml
        self.points = {} #None
        self.lines = {} #None
        self.polygons = {} #None
        self.loops = None
        self.dataframes = {}
        self.target_points = None
        self.crs = None
        self.swing_value = 0
        self.transform_crs = None
        self.target_points = None
        self.mis_tol = mis_tol
        self.survey_graph = None
        self.survey_number = None
        self.survey_year = None
        self.admin = None
        self.ccc = None
        self.polygon_irregular_lines = {}

        if landxml is not None:

            self.landxml = self.add_plan_features_to_parcels(self.landxml)

            self.add_missing_points_lines()
            is2point = self.get_is_2_point()
            start = datetime.now()
            self.points = PointGeomFactory().build(self.landxml, is2point)
            #print(datetime.now() - start, 'points')
            start = datetime.now()
            self.crs = self.set_crs()
            self.lines = ArcLineGeomFactory().build(self.landxml, self.points, is2point, self.crs)
            #print(datetime.now() - start, 'lines')
            start = datetime.now()
            self.survey_graph = SurveyGraph(self.lines, self.points)
            #print(datetime.now() - start, 'graph')
            start = datetime.now()
            self.polygons = PolygonGeomFactory().build(self.landxml, self.lines, self.points, self.crs,
                                                       self.polygon_irregular_lines)
            #print(datetime.now() - start, 'polygons')
            start = datetime.now()
            self.survey_number = self.get_survey_number()
            self.survey_year = self.get_survey_year()
            self.admin = Admin(self.landxml.Survey[0].SurveyHeader, self.polygons, self.landxml.CoordinateSystem)
            #print(datetime.now() - start, 'admin')
            start = datetime.now()
            self.set_null_geoms_to_admin_geom()
            self.ccc = self.set_ccc()
            self.loops = self.set_loop_errors()
            self.original_crs = self.crs
            #print(datetime.now() - start, 'rest')
            start = datetime.now()

    def set_null_geoms_to_admin_geom(self):
        """sets all polygons that have null geometry to the same shape as the overall admin polygon"""
        for k, v in self.polygons.items():
            if v.null_geometry is True:
                if len(v.children) == 0:
                    v.geometry = self.admin.geometry


    def add_plan_features_to_parcels(self, landxml=None):
        """adds plan features to the parcels dictionary
            :param landxml: parsed landxml file
            :type landxml: LandXML, None"""
        if landxml is None:
            landxml = self.landxml
        if landxml is not None:
            for item in landxml.PlanFeatures:
                name = item.name
                pf_desc = item.desc
                for pf in item.PlanFeature:
                    polygon = ParcelType()
                    polygon.name = pf.name
                    polygon.class_ = name
                    polygon.parcelFormat = 'Geometry'
                    polygon.state = name
                    polygon.Center = []
                    if isinstance(pf.CoordGeom, list):
                        polygon.CoordGeom = pf.CoordGeom
                    else:
                        polygon.CoordGeom = [pf.CoordGeom]
                    polygon.desc = pf_desc
                    if pf.desc is not None:
                        if len(pf.desc.strip()) > 0:
                            polygon.desc = pf.desc
                    polygon.parcelType = "Single"
                    landxml.Parcels[0].Parcel.append(polygon)
        return landxml


    # instrument setup id to point id reference dictionary
    def get_is_2_point(self):
        """the following 2 methods link the lines to the points, and the lines to the polygons, vice versa"""
        if self.landxml is not None:
            return {i.id: i.InstrumentPoint[0].pntRef for i in self.landxml.Survey[0].InstrumentSetup}

    # point id to instrument setup id reference dictionary
    def get_point_2_is(self):
        """the following 2 methods link the lines to the points, and the lines to the polygons, vice versa"""
        is2point = self.get_is_2_point()
        if is2point is not None:
            return {v: k for k, v in is2point}

    def add_irregular_lines(self, temp_lines):
        """adds irregular lines to the line dictionary
            :param temp_lines: dictionary of lines to add the to main line dictionary
            :type temp_lines: dict"""
        for k, v in temp_lines.items():
            if k not in self.lines:
                self.lines[k] = v

    def get_survey_number(self):
        """get survey number from the landxml"""
        survey_header = self.landxml.Survey[0].SurveyHeader
        survey_number = survey_header.name
        if survey_number is None:
            survey_number = ''
        return survey_number

    def set_loop_errors(self, mis_tol=None):
        """get loops errors from self
            :param mis_tol: misclose tolerance for loop errors to be displayed
            :type mis_tol: in, None"""
        if mis_tol is None:
            mis_tol = self.mis_tol
        if mis_tol is None:
            mis_tol = .1


        loops = loop_checker(self, self.ccc, mis_tol=mis_tol)

        start = datetime.now()
        f_loops = {}
        all_likely = []
        for k, v in loops.items():
            f_loops[k] = Loops(k, v, self.lines)
            all_likely.extend(v.get('likely', []))
        all_likely = set(all_likely)
        for k, v in self.lines.items():
            if v.name in all_likely:
                v.likely_candidate = True

        return f_loops

    def set_ccc(self):
        """sets the constrained mark for the file, all recalculations, transformations
            will be based around this point"""
        d = dict(self.survey_graph.graph.degree())
        id_name = dict()
        for key, value in d.items():
            id_name.setdefault(value, list()).append(key)
        ccc = (sorted(id_name[max(id_name)]))[0]
        self.points.get(ccc).ccc = True
        return ccc

    def get_survey_year(self):
        """gets the survey year from the landxml"""
        survey_header = self.landxml.Survey[0].SurveyHeader
        dates = survey_header.AdministrativeDate
        year = 1900

        if self.survey_number.startswith('TP'):
            year = 1900
        else:
            if not isinstance(dates, list):
                dates = [dates]
            for dated in dates:
                if dated.adminDateType == 'Date of Survey':
                    survey_date = dated.adminDate
                    if survey_date is None:
                        survey_date = date(year=1900, month=1, day=1)
                    try:
                        year = survey_date.year
                    except ValueError:
                        year = 1900
        return year

    def estimate_swing(self):
        """sets the estimated swing value from the file
            uses the original values of the coordinates in the file versus the stated bearings"""
        swings = []
        for key, value in self.lines.items():
            calcd = math.degrees(calc_bearing(self.points.get(key[1]).geometry,
                                              self.points.get(key[0]).geometry))
            dif = calcd - value.dd_bearing
            if dif < -180:
                dif += 360
            swings.append(dif)

        if len(swings) > 0:
            std = statistics.pstdev(swings)
            mean = statistics.mean(swings)
            swings = [x for x in swings if (x > mean - 2 * std) and (x < mean + 2 * std)]
            self.swing_value = statistics.mean(swings)
        else:
            self.swing_value = 0

    def recalc_geometries(self, ref_point=None, swing=False, sf=1, swing_value=None, xml=True, leave_constrained=False,
                          only_connections=False, use_branches=False, loops=True, update_geoms=True,
                          ignore_line_types=None):
        """recalculates the coordinates in the file based on the bearing and distances in the lines dictionary
            uses the landxml if its available otherwise just used what is available
            can set the swing to rotated the values accordingly, as well as a scale factor to scale the plan
            :param ref_point: the point to start the recalculation from, and will rotate and scale from this point
            :param swing: set to true or false depending if you want a swing to be applied during recalculation
            :param sf: scale factor set to 1 as default, will be a multiple of the distance value, as in if scale
                        factor is set to 2 and distance is 1m then distance will be set to 2m
            :param swing_value: value in degrees to rotate the measurements by, positive is clockwise
            :param xml: if available use the xml file to make the calculations quicker
            :type ref_point: str or None
            :type swing: bool
            :type sf: float
            :type swing_value: float or None
            :type xml: bool"""
        # need to make sure geometries are on a plane here.
        ocrs = None
        if self.crs in (7844, 4383):
            ocrs = self.crs
            zone = self.estimate_zone()
            self.transform_geometries(zone, xml)

        self.swing_value = swing_value
        if swing is True and swing_value is None:
            self.estimate_swing()
        elif swing is False:
            self.swing_value = 0

        if use_branches is True:
            if ignore_line_types is not None:
                gs = self.survey_graph.graph_from_branches(line_types=ignore_line_types)
            else:
                gs = self.survey_graph.graph_from_branches(line_types=('GraphGenerated', 'BranchConnection'))
        else:
            if ignore_line_types is not None:
                gs = [self.survey_graph.ignore_line_type(ignore_line_types)]
            else:
                gs = [self.survey_graph.graph]
        constrained = self.get_constrained_points()
        for g in gs:
            if self.points.get(ref_point) is None or ref_point not in g.nodes:
                connected = nx.is_k_edge_connected(g, 1)
                ref_points = sorted(list(g.nodes))
                if connected is False:
                    branches = list(nx.k_edge_components(g, 1))
                    ccc = [i for i in constrained if i in ref_points]
                    if len(ccc) > 0:
                        ref_point = ccc[0]
                    else:
                        max_b = max(branches, key=len)
                        ref_point = sorted([item for item in max_b])[-1]

                else:

                    ccc = [i for i in constrained if i in ref_points]
                    if len(ccc) > 0:
                        ref_point = ccc[0]
                    else:
                        ref_point = ref_points[0]

            short_paths = nx.single_source_shortest_path(g, ref_point)
            ref_point_coords = self.points.get(ref_point).geometry.coords[:]
            new_geom = set()
            for key, value in short_paths.items():
                if self.points.get(ref_point).geometry.has_z is False:
                    ref_e, ref_n = ref_point_coords[0]
                else:
                    ref_e, ref_n, ref_z = ref_point_coords[0]

                for prev, item, nxt in previous_and_next(value):
                    if nxt is not None:
                        b_d = g.get_edge_data(item, nxt)
                        bearing = b_d['bearing']
                        if (item, nxt) != b_d['st']:
                            bearing += 180
                            bearing = calc_inside_360(bearing)

                        delta_e, delta_n = calc_new_point(bearing, b_d['distance'] * sf)
                        ref_e += delta_e
                        ref_n += delta_n

                        if nxt not in new_geom or leave_constrained is True:
                            new_geom.add(nxt)
                            if leave_constrained is True and self.points[nxt].ccc is True:
                                ref_e = self.points[nxt].geometry.x
                                ref_n = self.points[nxt].geometry.y
                            elif only_connections is True:
                                if b_d['distance_type'] == 'GraphGenerated':
                                    self.points[nxt].geometry = sg.Point(ref_e, ref_n)
                                else:
                                    ref_e = self.points[nxt].geometry.x
                                    ref_n = self.points[nxt].geometry.y
                            else:
                                self.points[nxt].geometry = sg.Point(ref_e, ref_n)

        if swing is True and self.swing_value != 0:
            self.apply_swing(self.swing_value, origin=self.points.get(ref_point).geometry, xml=xml)
        elif update_geoms is True:
            self.update_geometries(self.swing_value, xml=xml, loops=loops)

        if ocrs is not None:
            self.transform_geometries(ocrs, use_xml_data=xml)

    def transform_geometries(self, out_proj, use_xml_data=True, update_line_coords=False, only_geom=False):
        """transforms the geometry onto a different projection system
            :param out_proj: epsg projection number
            :param use_xml_data: use the data within the xml file or just use existing information
            :type out_proj: int
            :type use_xml_data: bool"""
        self.points = transform_geoms(self.points, self.crs, out_proj)

        if use_xml_data is True and self.landxml is not None:
            self.crs = out_proj
            self.update_geometries()
        else:
            if self.lines is not None:
                self.lines = transform_geoms(self.lines, self.crs, out_proj)
                if update_line_coords is True:
                    for k, v in self.lines.items():
                        v.create_line_from_coords(setup_point=self.points.get(k[0]),
                                                  target_point=self.points.get(k[1], crs=out_proj))
                        self.lines[k] = v
            if self.polygons is not None:
                self.polygons = transform_geoms(self.polygons, self.crs, out_proj)
                if only_geom is False:
                    for k, v in self.polygons.items():
                        v.crs = out_proj
                        v.set_coord_lookup(self.points)
                        if len(self.lines) > 0:
                            for ring, lo in v.generated_line_order.items():
                                nlo = []
                                for value in lo:
                                    nl = self.lines.get((value.setup_point.name, value.target_point.name))
                                    nlo.append(nl)
                                v.generated_line_order[ring] = nlo
                                v.line_order[ring] = nlo
                        else:
                            los = v.generated_line_order
                            for ring, lo in los.items():
                                items = {(value.setup_point.name, value.target_point.name): value for value in lo}

                                los[ring] = [value for value in items.values()]
                            v.generated_line_order = los
                            v.line_order = los
                        self.polygons[k] = v
            if self.loops is not None:
                if len(self.loops) > 0:
                    self.loops = transform_geoms(self.loops, self.crs, out_proj)
            if self.admin is not None:
                if self.admin.geometry is not None:
                    ag = transform_geoms({'i': self.admin}, self.crs, out_proj)
                    self.admin = ag['i']
            self.crs = out_proj

    def reset_geometries_to_original_crs(self):
        """resets the geometry to the original crs"""
        self.transform_geometries(out_proj=self.original_crs)

    def set_crs(self):
        """set the crs from the first point assuming only one crs for the entire plan
        :returns crs: value or crs"""
        for p in self.points.values():
            crs = p.crs
            break
        return crs

    def set_target_points(self, target_points):
        """sets the target points to try and match geometry to
            :param target_points: dict of shapely geometry point objects
            :type target_points: dict"""
        self.target_points = target_points

    def transform_onto_points_with_ids(self, target_points=None):
        """if ids are known transform the points from this class to those matched points, as in if id 1 has x,y in this
            file move it to id 1 x,y in the target points, updates points inplace and returns values as well
            :param target_points: dict of shapely geometry point objects uses self if target points is None
            :type target_points: None or dict
            :returns self.points: dictionary of PointGeom objects"""

        if target_points is None:
            target_points = self.target_points
        self.points = helmert_transformation_with_ids(self.points, target_points, self.lines)
        return self.points

    def add_missing_points_lines(self):
        import numpy as np
        """some points exist in polygons and other parts of the file such as irregular lines
            this will add some generated lines needs self.landxml to exist"""
        existing_points = {point.valueOf_.strip(): point for point in self.landxml.CgPoints[0].get_CgPoint()}
        lxml_parcels = self.landxml.Parcels
        lxml_polygons = []
        if len(lxml_parcels) > 0:
            for x in lxml_parcels:
                lxml_polygons += x.Parcel
        for polygon in lxml_polygons:
            if polygon.parcelType != 'Multipart':
                if (polygon.class_ == 'Easement' and polygon.parcelFormat == 'Standard') is False:
                    if len(polygon.CoordGeom) > 0:
                        coord_geom = polygon.CoordGeom[0]
                        irls = coord_geom.IrregularLine
                        ir_count = 0
                        c = 90000
                        for count, irregular_line in enumerate(irls):
                            pnt_list = irregular_line.PntList2D
                            chunk_list = [(float(i[0]), float(i[1])) for i in chunker(pnt_list.split(), 2, tuple_=True)]
                            non_rolled = np.array([sg.Point(i[1], i[0]) for i in chunk_list])
                            rolled_list = np.roll(non_rolled, shift=-1)
                            lines = np.vstack([non_rolled, rolled_list]).transpose()[:-1].tolist()

                            for line in lines:
                                points = []
                                for p in line:
                                    n = str(p.y)
                                    e = str(p.x)
                                    ne = f'{n} {e}'
                                    exist = existing_points.get(ne)
                                    if exist is None:
                                        point = CgPoint()
                                        point.valueOf_ = ne
                                        point.name = f'CGPNT-{ne}'.replace(' ', '-')
                                        point.pntSurv = 'natural boundary'
                                        point.state = 'existing'
                                        points.append(point)
                                        # add point
                                        existing_points[ne] = point
                                        self.landxml.CgPoints[0].add_CgPoint(value=point)

                                        # add IS station
                                        insetup = InstrumentSetup()
                                        insetup.id = point.name.replace('CGPNT', 'IS')
                                        insetup.stationName = insetup.id
                                        insetup.instrumentHeight = 0
                                        insetup.add_InstrumentPoint(value=InstrumentPoint(point.name))
                                        self.landxml.Survey[0].add_InstrumentSetup(value=insetup)
                                    else:
                                        points.append(exist)

                                ro = ReducedObservation()
                                default = c + ir_count
                                ro.name = f'OBS-{default}'
                                if irregular_line.desc is not None:
                                    ro.desc = irregular_line.desc
                                else:
                                    ro.desc = 'Irregular Line'
                                ro.azimuthType = 'Generated'
                                ro.distanceType = 'Generated'
                                ro.purpose = 'natural boundary'
                                ro.setupID = points[0].name.replace('CGPNT', 'IS')
                                ro.targetSetupID = points[1].name.replace('CGPNT', 'IS')
                                ro.horizDistance = calc_distance(line[0], line[1])
                                ro.azimuth = dd2hp(calc_bearing(line[1], line[0]))
                                self.landxml.Survey[0].ObservationGroup[0].add_ReducedObservation(value=ro)
                                ir = self.polygon_irregular_lines.get(polygon.name, {})
                                iro = ir.get(count, [])
                                iro.append(ro)
                                ir[count] = iro
                                self.polygon_irregular_lines[polygon.name] = ir
                                ir_count += 1

    def apply_translation(self, x=0, y=0):
        """apply a translation in xy to the geometry class
            :param x: easting or longitude value
            :param y: northing or latitude value
            :type x: float
            :type y: float"""
        if x != 0 and y != 0:
            for k, v in self.points.items():
                self.points[k].geometry = sa.translate(v.geometry, x, y)
            self.update_geometries()

    def add_lines_from_polygons(self, duplicates=False):
        if self.lines is None:
            self.lines = {}
        for polygon in self.polygons.values():
            polygon: PolygonGeom
            for lines in polygon.line_order.values():
                for l in lines:
                    if (l.target_point.name, l.setup_point.name) not in self.lines and duplicates is False:
                        self.lines[(l.setup_point.name, l.target_point.name)] = l


    def apply_swing(self, angle=0, origin='center', xml=False):
        """origin here should be the point you want to swing the plan around.
        defaulted to the centre of a point, so you won't get any rotation of a point feature if it's not set,
        this will generally be the point that you translate the survey onto
        angle is in decimal degrees
        The point of origin can be a keyword 'center' for the bounding box
        center (default), 'centroid' for the geometry's centroid, a Point object
        or a coordinate tuple (x0, y0).
        :param angle: angle in decimal degrees
        :param origin: origin to rotate around such as a point coordinate, center of all points is default
        :param xml: updated the values using the xml data if available
        :type angle: float
        :type origin: str, tuple, sg.Point
        :type xml: bool"""

        if angle != 0:
            for k, v in self.points.items():
                self.points[k].geometry = sa.rotate(v.geometry, -1 * angle, origin)
            self.update_geometries(swing=angle, xml=xml)

    def apply_affine_transformation(self, mat=None, points_list=None):
        """apply and affine transformation to the geometry
        :param mat: matrix for affine transformation
            For 2D affine transformations, the 6 parameter matrix is::
            [a, b, d, e, xoff, yoff]
            which represents the augmented matrix:
            [x']   / a  b xoff \ [x]
            [y'] = | d  e yoff | [y]
            [1 ]   \ 0  0   1  / [1]
        :type mat: list"""
        if points_list is not None:
            points = {k: v for k, v in self.points.items() if k in points_list}
        else:
            points = self.points
        if mat is not None:
            for k, v in points.items():
                if v.associated_point_oid is None:
                    self.points[k].geometry = sa.affine_transform(v.geometry, mat)
            self.update_geometries()

    def update_geometries(self, swing=0., xml=True, loops=True):
        """updates the geometries based on the point coordinates can apply a swing in here as well
            uses the landxml file if its available otherwise it will use the line values to then recreate
            the polygons
            :param swing: swing value to apply if needed
            :param xml: use xml values if availabl
            :type swing: float
            :type xml: bool"""
        if xml is True and self.landxml is not None:
            self.lines = ArcLineGeomFactory().build(self.landxml, self.points, self.get_is_2_point(), self.crs, swing)
            self.polygons = PolygonGeomFactory().build(self.landxml, self.lines, self.points, self.crs,
                                                       self.polygon_irregular_lines)
            self.admin.geometry = self.admin.set_geometry(self.polygons)
            self.admin.crs = self.crs
        else:
            line: LineGeom
            for line in self.lines.values():
                line.crs = self.crs
                line.setup_point = self.points.get(line.setup_point.name)
                line.target_point = self.points.get(line.target_point.name)
                if line.is_arc is False:
                    line.calc_geometry_from_start_end()
                else:
                    line: ArcGeom
                    line.calc_arc_geometry_from_start_end()

            for polygon in self.polygons.values():

                polygon.update_line_order(self.lines)
                polygon.update_polygon_geom_from_lines()
                polygon.crs = self.crs

        self.survey_graph = SurveyGraph(self.lines, self.points)
        if loops is True:
            self.loops = self.set_loop_errors()

    def write_geom_to_file(self, points=True, lines=True, arcs=True, polygons=True, location=None, loops=False,
                           file_type='GPKG', same_file=True, df_only=False, filename_suffix='', remove=None):
        """uses geopanadas and fiona to export the geometries to file cleans columns
            choose what files you want to output and if you jsut want to build and store the dataframes in the
            class rather than output to file.
            :param points: output points
            :param lines: output lines
            :param arcs: output arcs
            :param polygons: output polygons
            :param location: output location
            :param loops: output loops
            :param file_type: the type of file to output defaulted to geopackage
            :param same_file: output all layers to the same file if available
            :param df_only: dont output to file but just store the dataframes
            :param filename_suffix: and a suffix to the output name for identification
            :param remove: columns to remove
            :type points: bool
            :type lines: bool
            :type arcs: bool
            :type polygons: bool
            :type location: str or Path obejct
            :type loops: bool
            :type file_type: str
            :type same_file: bool
            :type df_only: bool
            :type filename_suffix: str
            :type remove: None or list
            """
        def clean_columns(df):
            for column in df.columns:
                if column == 'setup_point' or column == 'target_point':
                    try:
                        df[column] = df[column].apply(lambda x: x.point_oid)
                    except Exception as err:
                        print(err)
            return df

        def drop_columns(df, columns):
            for column in columns:
                if column in df.columns:
                    df.drop(columns=[column], inplace=True)
            return df

        def make_gdf(vals, cols=None):
            if cols is None:
                cols = []

            df = pd.DataFrame(vals)
            df = drop_columns(df, cols)
            df = clean_columns(df)
            gdf = gpd.GeoDataFrame(df.drop(columns='geometry'), geometry=df['geometry'], crs=self.crs)
            gdf['filename'] = self.survey_number
            return gdf

        def write_file(gdf, layer, location, suffix, file_type, same_file, filename_suffix):
            out_path = Path(location, self.survey_number)
            out_path.mkdir(exist_ok=True, parents=True)
            if same_file is False or file_type != 'GPKG':
                out_path = Path(out_path, layer + filename_suffix + suffix)
                gdf.to_file(str(out_path), driver=file_type)
            else:
                out_path = Path(out_path, self.survey_number + filename_suffix + suffix)
                gdf.to_file(str(out_path), driver=file_type, layer=layer)

        extension_lookup = {'GPKG': '.gpkg', 'ESRI Shapefile': '.shp'}

        try:
            import pandas as pd
            import geopandas as gpd
        except ImportError as e:
            print(e)
            raise
        if file_type not in {'GPKG', 'ESRI Shapefile'}:
            file_type = 'ESRI Shapefile'
            print('File type wasnt recognised, setting to shapefile')

        suffix = extension_lookup.get(file_type)

        if location is None and df_only is False:
            print('Location cannot be None')
            raise NotADirectoryError
        elif location is not None:
            if os.path.exists(location) is False and df_only is False:
                out_path = Path(location)
                out_path.mkdir(exist_ok=True, parents=True)
                # print('Location must exist')
                # raise NotADirectoryError

        if points is True:
            vals = [v.__dict__ for v in self.points.values()]
            if remove is None:
                remove = ['neh', 'original_geom']
            nd = {}
            for i in vals:
                for k, v in i.items():
                    if isinstance(v, (RedVerticalObservation, RedHorizontalPosition)) is True:
                        for key, value in v.__dict__.items():
                            if isinstance(value, (list, set, dict)):
                                value = str(value)
                            nd[str(k) + '_' + str(key)] = value
                        remove.append(k)
            i = {**i, **nd}
            gdf = make_gdf(vals, remove)
            self.dataframes['points'] = gdf
            if df_only is False:
                write_file(gdf, 'points', location, suffix, file_type, same_file, filename_suffix)

        if loops is True:
            vals = [v.__dict__ for v in self.loops.values()]
            gdf = make_gdf(vals)
            self.dataframes['loops'] = gdf
            if df_only is False:
                write_file(gdf, 'loops', location, suffix, file_type, same_file, filename_suffix)

        if lines is True or arcs is True:
            lines_vals = []
            arcs_vals = []

            for v in self.lines.values():
                nd = {}
                for key, value in v.__dict__.items():
                    if isinstance(value, (list, set, dict)):
                        value = str(value)
                    nd[key] = value
                if isinstance(v, LineGeom):

                    lines_vals.append(nd)
                else:
                    arcs_vals.append(nd)



            if len(lines_vals) > 0 and lines is True:
                gdf = make_gdf(lines_vals)
                self.dataframes['lines'] = gdf
                if df_only is False:
                    write_file(gdf, 'lines', location, suffix, file_type, same_file, filename_suffix)
            if len(arcs_vals) > 0 and arcs is True:
                gdf = make_gdf(arcs_vals)
                self.dataframes['arcs'] = gdf
                if df_only is False:
                    write_file(gdf, 'arcs', location, suffix, file_type, same_file, filename_suffix)

        if polygons is True:

            vals = [v.__dict__ for v in self.polygons.values()]
            if remove is None:
                remove = []
            gdf = make_gdf(vals, ['generated_line_order', 'line_order', 'polygon_points', 'inner_angles',
                                  'coord_lookup',
                                  'point_lookup', 'original_geom', 'misclose', 'parcel_arcs', 'arc_rot_errors',
                                  'arc_rad_errors', 'children', 'irregular_lines', 'all_children'] + remove)

            gdf['polygon_notations'] = gdf['polygon_notations'].apply(lambda x: ','.join(map(str, x)))

            self.dataframes['polygons'] = gdf
            if df_only is False:
                write_file(gdf, 'polygons', location, suffix, file_type, same_file, filename_suffix)

    def set_likely_candiates(self):
        """set line attribute if they are a likely candidate in the loops"""
        for k, v in self.loops:
            for name in v.likely_names:
                line = self.lines.get(name)
                if line is None:
                    line = self.lines.get(line[1], line[0])
                if line is not None:
                    line.likely_candidate = True

    def set_overlapping_lines(self):
        """sets overlapping line attribute if the lines are overlapping/crossing one another"""
        lines_nn = {k: v for k, v in self.lines.items() if v.line_type == 'normal'}
        geoms = []
        rev_lookup = {}
        for k, v in lines_nn.items():
            geoms.append(v.geometry)
            rev_lookup[v.geometry.wkb] = k

        for item, line in lines_nn.items():
            line_geom = line.geometry
            geoms.remove(line_geom)
            if len(geoms) > 0:
                prepped_geom = prep(line_geom)
                hits = list(filter(prepped_geom.crosses, geoms))
                if len(hits) > 0:
                    self.lines[item].overlapping = True
                    for hit in hits:
                        line_hit = rev_lookup[hit.wkb]
                        self.lines[line_hit].overlapping = True

    def gen_graph_from_points_lines(self, add_unconnected=True, add_gen_lines=False,
                                    remove=tuple(), join_branches=False):
        """generate a graph from the points and lines in the file this is used with other methods for spatial analysis
        :param add_unconnected: include unconnected lines in the graph by adding a joining edge
        :param add_gen_lines: add the generated lines to the graph, these lines arent in the original survey, but are
                                created for joining unconnected points etc.
        :param remove: tuple of line types not to include in graph
        :param join_branches: specifies if you want to connect the branches together with a generated edge
        :type add_unconnected: bool
        :type add_gen_lines: bool
        :type remove: tuple
        :type join_branches: bool"""
        if self.points is not None and self.lines is not None:
            self.survey_graph = SurveyGraph(self.lines, self.points,
                                            add_unconnected, add_gen_lines, remove, join_branches)

    def produce_dissolved_polygon(self, types=('created',), include_easements=False, specific=None):
        """create a dissolved polygon from the selected polygon types, defaulted at just created types, choose to
        include easements in the result
        :param types: tuple of types that should be included in the result
        :param include_easements: include easements in the resulting polygon
        :param specific: specific polygons to included
        :type types: tuple of strings
        :type include_easements: bool

        :return p: PolygonGeom which includes dissolved geometry from all input polygons
        """
        if specific is not None:
            created = so.unary_union([v.geometry for k, v in self.polygons.items()
                                      if v.geometry is not None and k in specific])
        else:
            if include_easements is True:
                created = so.unary_union([v.geometry for k, v in self.polygons.items() if v.parcel_state in types
                                          and v.geometry is not None])
            else:
                created = so.unary_union([v.geometry for k, v in self.polygons.items()
                                          if v.parcel_state in types and v.easement is False
                                          and v.geometry is not None])

        p = PolygonGeom()
        crs = None
        name = None
        coord_dec = 100
        if len(list(self.polygons.values())) > 0:
            crs = list(self.polygons.values())[0].crs
            if crs in [7844, 4283]:
                coord_dec = 10
            else:
                coord_dec = 3
            name = list(self.polygons.values())[0].name
        p.create_polygon(created, self.points, name, crs, coord_decimals=coord_dec)
        return p

    def update_xml_coordinates(self):
        """update the coordinates within the parsed xml file for reexporting with new data, uses the self.points dict
        to get the updated coordinates"""
        points = self.landxml.CgPoints[0].CgPoint
        for point in points:
            name = point.get_name()
            p = self.points.get(name)
            if p is not None:
                coords = p.geometry.coords[:][0]
                n = str(coords[1])
                e = str(coords[0])
                point.latitude = float(n)
                point.longitude = float(e)
                if len(coords) > 2:
                    h = str(coords[2])
                    point.valueOf_ = f'{n} {e} {h}'

                else:
                    point.valueOf_ = f'{n} {e}'
        coord_system: CoordinateSystem
        coord_system = self.landxml.CoordinateSystem[0]
        coord_system.datum = f'MGA2020_Zone{str(self.crs)[2:]}'
        coord_system.horizontalDatum = f'MGA2020_Zone{str(self.crs)[2:]}'

    def update_eplan_compliance_value(self, eplan_compliance=True):
        from utilities.dcmgeometrysdk.landxml.landxml import GeneratedsSuper
        """hacky method to set all values in the stored land xml file to have an attribute of eplan compliance"""
        def update_eplan_compliance(item, updated_value=True):

            if isinstance(item, list):
                for value in item:
                    update_eplan_compliance(value, updated_value)


            elif hasattr(item, 'eplan_value'):
                item.eplan_value = updated_value

                for key, value in item.__dict__.items():
                    if isinstance(value, list):
                        for i in value:
                            update_eplan_compliance(i, updated_value)
                    elif hasattr(value, 'eplan_value') and key not in ['parent_object_', 'Tag_strip_pattern_',
                    'gds_collector_', 'gds_elementtree_node_', 'tzoff_pattern', '_FixedOffsetTZ']:
                        update_eplan_compliance(value, True)

        if self.landxml is not None:
            update_eplan_compliance(self.landxml, updated_value=eplan_compliance)

    def write_to_eplan_compliant_file(self, outfile=None, encode=False):
        self.landxml: LandXML
        output = None
        if self.landxml is not None:
            self.update_eplan_compliance_value()

            if self.crs in {7844}:
                point = [x.geometry for x in self.points.values()]
                if len(point) > 0:
                    zone = estimate_zone(point[0].geometry)
                    self.transform_geometries(zone)
            self.update_xml_coordinates()
            if outfile is None:
                outfile = StringIO()
                self.landxml.export(outfile=outfile, level=0)
                outfile.seek(0)
                if encode is True:
                    output = base64.b64encode(outfile.read().encode('utf-8'))
                else:
                    output = outfile.read()
            else:
                self.write_xml_to_file(outfile)
                output = outfile
            return output

    def write_xml_to_file(self, outfile):
        """output the xml file within this class, often only used after the coordinates have been updated"""
        with open(outfile, 'w') as open_file:
            self.landxml.export(outfile=open_file, level=0)


    def associated_point_look_up(self):
        """when the class is matched to a different dataset associated points are matched and added to the PointGeom
        class, this creates a look up for those associations."""
        return {int(p.associated_point_oid): p.name for p in self.points.values()
                     if p.associated_point_oid is not None}

    def match_to_plan(self, plan_to_match_geom, close_points):
        """matches geometry to a common azimuth with another plan. Close points is a list of points that are
            common between the 2 plans
            :param plan_to_match_geom: geometry class of another file to match to
            :param close_points: an iterable of points ids
            :type plan_to_match_geom: Geometries
            :type close_points: iterable"""
        associated_lookup = plan_to_match_geom.associated_point_look_up()
        self.transform_geometries(out_proj=plan_to_match_geom.crs, use_xml_data=False)
        self.gen_graph_from_points_lines()
        self.recalc_geometries(xml=False)
        # get longest line in main plan
        max_line = None
        max_dist = 0
        max_a = None
        max_b = None
        max_a_lookup = None
        max_b_lookup = None
        for a, b in combinations(close_points, 2):
            a_lookup = associated_lookup.get(a, plan_to_match_geom.points.get(a))
            b_lookup = associated_lookup.get(b, plan_to_match_geom.points.get(b))

            line = plan_to_match_geom.lines.get((a_lookup, b_lookup))
            if line is None:
                line = plan_to_match_geom.lines.get((b_lookup, a_lookup))
                if line is not None:
                    line = deepcopy(line)
                    line.flip_direction()
                else:
                    line = LineGeom()
                    line.create_line_from_coords(plan_to_match_geom.points.get(a_lookup),
                                                 plan_to_match_geom.points.get(b_lookup),
                                                 name='gen1', crs=plan_to_match_geom.crs)

            if line.distance > max_dist:
                max_dist = line.distance
                max_line = line
                max_a = a
                max_b = b
                max_a_lookup = a_lookup

        a_point = self.points.get(str(max_a))
        b_point = self.points.get(str(max_b))
        bearing = calc_inside_360(math.degrees(calc_bearing(b_point.geometry, a_point.geometry)))
        self.points.get(str(max_a)).set_new_geometry(plan_to_match_geom.points.get(max_a_lookup).geometry)
        swing = max_line.dd_bearing - bearing
        self.recalc_geometries(ref_point=str(max_a), swing=True, swing_value=swing, xml=False)

    def update_coords_from_dna_reader(self, dna_results, use_xml=True, loops=True):
        dna_coords = dna_results.coordinates
        transformer = None
        current_crs = None
        for k, v in self.points.items():
            point = dna_coords.get(k)
            if point is None:
                point = dna_coords.get(str(k))
            if point is not None:
                v.set_new_geometry(point.geometry)
                o_crs = v.crs
                if o_crs != dna_results.crs:
                    if transformer is None or current_crs != o_crs:
                        current_crs = o_crs
                        transformer, osr_value = build_transformer(dna_results.crs, o_crs)
                    v.geometry = transform_coordinates(v.geometry, project=transformer)
                v.crs = o_crs
                self.points[k] = v

        self.update_geometries(xml=use_xml, loops=loops)


    def get_polygon_points(self):
        poly_points = set()
        for k, v in self.polygons.items():
            v: PolygonGeom
            pp = set(v.polygon_points.get('all'))
            poly_points = poly_points.union(pp)
        return poly_points

    def insignificant_points(self, distance=.2, angle=180):
        i_points = set()
        s_points = set()
        for k, v in self.polygons.items():
            v: PolygonGeom
            for point, item in v.inner_angles.items():
                a = item.get('angle')
                if (angle - 2) < a < (angle + 2):
                    to_distance = item.get('to_distance')
                    from_distance = item.get('from_distance')
                    if to_distance < distance or from_distance < distance:
                        i_points.add(point)
                    else:
                        s_points.add(point)
                else:
                    s_points.add(point)
        return i_points.difference(s_points)

    def get_constrained_points(self):
        return [k for k, v in self.points.items() if v.ccc is True]

    def estimate_zone(self):
        point = [x.geometry for x in self.points.values()]
        return estimate_zone(point[0])










