from geoalchemy2.shape import to_shape
import shapely.ops as so
from domain.adjustment.model.cadastral_block import CadastralBlock
from domain.adjustment.model.cadastralblockplan import CadastralBlockPlan
from domain.adjustment.model.cadastralblockcontrol import CadastralBlockControl
from domain.adjustment.model.dynadjustmeasure import DynadjustMeasure
from domain.adjustment.model.constraints import Constraints
from domain.adjustment.model.constraints_point_geometry import ConstraintsPointGeom
from domain.adjustment.model.constraints_line_geometry import ConstraintsLineGeom
from utilities.dcmgeometrysdk.dna.dnarunner import DNAWriters
from utilities.dcmgeometrysdk.geometryfunctions.otherfunctions import read_pickle_from_file
from copy import deepcopy
from domain.helpers.db_operations import create_database_connection

import logging


log = logging.getLogger()
log.setLevel(logging.INFO)


class MapDNA:
    def __init__(self, writer=None, event_id=None, json_message=None, pickle_file=None):
        self.event_id = event_id,
        self.json_object = json_message
        self.adjustment_db = create_database_connection(json_message["adjustment"])
        self.writer = None
        if writer is not None:
            self.writer = writer
        elif pickle_file is not None:
            self.writer = read_pickle_from_file(pickle_file)
        if self.writer is not None:
            self.writer: DNAWriters
            self.cad_block = self.writer.block_id
            self.plan_number = self.writer.filename
            self.plan_id = int(self.json_object["plan_admin_id"])
            with self.adjustment_db as open_session:
                f = open_session.query(CadastralBlock).where(
                    CadastralBlock.source_block_id == self.cad_block).all()
                if len(f) > 0:
                    self.zone_oid = f[0].cadastral_zone_id
                block_geom = [to_shape(i.geom) for i in f]
                self.block_geom = so.unary_union(block_geom)

            self.block_plan = self.map_cadastral_block_plan()
            self.direction_measures = self.map_direction_measures()
            self.distance_measures = self.map_distance_measures()
            self.block_control = self.map_cadastral_block_control()
        self.load_all()

    def load_all(self):
        logging.info('Load pickle file data to adjustment DB')
        with self.adjustment_db as open_session:
            open_session.add_all(self.direction_measures)
            open_session.add_all(self.distance_measures)
            open_session.add_all(self.block_control)
            open_session.add(self.block_plan)
            open_session.commit()
        logging.info('Loaded pickle file data to adjustment DB')

    def map_direction_measures(self):
        direction_measures = []
        for ds in self.writer.direction_sets:
            directions = ds.direction_set
            model = DynadjustMeasure()
            model.cadastralzoneid = self.zone_oid
            if len(ds.direction_set) > 1:
                model.first = directions[0].ip
                model.value1 = directions[0].hp_bearing
                model.second = directions[0].to_station
                if len(directions[0].ignore) > 0:
                    model.ignore = True
                else:
                    model.ignore = False
                model.stddev1 = directions[0].std
                for direction in directions[1:]:
                    new_model = deepcopy(model)
                    new_model.third = direction.to_station
                    new_model.stddev2 = direction.std
                    new_model.value2 = direction.hp_bearing
                    new_model.plannumber = self.plan_number
                    new_model.type = 'D'
                    new_model.source = 'DCAS'
                    new_model.active = True
                    new_model.instrumentheight = 0
                    new_model.stationheight = 0
                    new_model.planid = self.plan_id
                    new_model.create_event_id = self.event_id
                    direction_measures.append(new_model)

        return direction_measures

    def map_distance_measures(self):
        distance_measures = []
        for ds in self.writer.distances:
            model = DynadjustMeasure()
            model.cadastralzoneid = self.zone_oid
            model.first = ds.setup
            model.second = ds.target
            model.value1 = ds.distance
            model.stddev1 = ds.dist_std
            model.instrumentheight = ds.ins_height
            model.stationheight = ds.targ_height
            model.ignore = False
            model.active = True
            model.value2 = None
            model.stddev2 = None
            model.third = None
            model.plannumber = self.plan_number
            model.type = 'S'
            model.planid = self.plan_id
            model.create_event_id = self.event_id
            distance_measures.append(model)
        return distance_measures

    def map_pq_measure(self):
        pass

    def map_cadastral_block_control(self):
        block_control = []
        for control in self.writer.points.values():
            if control.ccc is True:
                bc = CadastralBlockControl()
                bc.active = True
                bc.type = 'SMES'
                bc.cadastralblockid = self.cad_block
                bc.pointid = control.point_oid
                bc.create_event_id = self.event_id
                block_control.append(bc)
        return block_control

    def map_cadastral_block_plan(self):
        block_plan = CadastralBlockPlan()
        block_plan.cadastralblockid = self.cad_block
        block_plan.active = True
        block_plan.plannumber = self.plan_number
        block_plan.planid = self.plan_id
        block_plan.create_event_id = self.event_id
        return block_plan

    def map_constraints(self):
        pass

    # def get_plan_id(self, plan_engine):
    #     sql = f"""select a.id from plan."admin" a
    #               where a.survey_number = '{self.plan_number}' and a.retire_event_id is Null"""
    #     f = plan_engine.execute(sql).first()
    #     return f[0]

# if __name__ == '__main__':
#     ADJUSTMENTDB_CONNECTION = "postgresql://dcmadjustmentdb:5QkK#jLjZvf(s1@localhost:9001/adjustmentdb"
#     message_body_json={}
#     message_body_json["adjustment"] = ADJUSTMENTDB_CONNECTION
#     message_body_json["plan_admin_id"] = "2345"
#     mapper = MapDNA(None, json_message=message_body_json, pickle_file='C:\\Users\\4893\\Downloads\\CP108808.pkl')
#     mapper.load_all()
