import os
import sys
from typing import List, Tuple, Dict, Optional, Type

from PyQt5.QtCore import Qt
from PyQt5.QtWidgets import QListWidgetItem, QWidget, QSizePolicy, QHeaderView
from qgis.PyQt import uic
from qgis.PyQt import QtWidgets, QtCore
from qgis._core import QgsField
from qgis.core import QgsMapLayerProxyModel, QgsRasterLayer, QgsVectorLayer

from landsklim.lk.utils import LandsklimUtils
from landsklim.lk.landsklim_analysis import LandsklimAnalysisMode
from landsklim.lk.phase import IPhase
from landsklim.lk.regressor import Regressor
from landsklim.lk.regressor_factory import RegressorDefinition
from landsklim.ui.utils import LandsklimTableModelRegressors, LandsklimTableColumnType, QListWidgetItemDataComparable
from landsklim.ui.landsklim_dialog import LandsklimDialog
from landsklim import landsklim
from landsklim.lk.map_layer import MapLayerCollection, VectorLayer

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'view_new_analysis.ui'))


class ViewNewAnalysis(LandsklimDialog, FORM_CLASS):
    """
    Represents the new analysis dialog

    :ivar Dict[str, List[int]] edited_regressors: Store regressors edited by user when click on min/max button
    """
    def __init__(self, default_regressors: List[Regressor], parent=None):
        """Constructor."""
        self.__regressors = default_regressors
        super(ViewNewAnalysis, self).__init__(parent)
        self.setModal(True)
        self.edited_regressors: Dict[str, List[int]] = {}

    def init_ui(self):
        """
        Init UI components
        """
        for configuration in self.get_configurations():
            self.cb_configuration.addItem(str(configuration), configuration)

        available_phases: List[Type[IPhase]] = self._instance.get_available_phases()
        for phase in available_phases:
            self.cb_phase1.addItem(phase.name(), phase.class_name())
            self.cb_phase2.addItem(phase.name(), phase.class_name())

        self.cb_phase2.setCurrentIndex(1)

        dem_shape = LandsklimUtils.raster_to_array(self._instance.get_landsklim_project().get_dem().qgis_layer()).shape
        max_window_allowed = int(min(dem_shape)/2)

        stations: MapLayerCollection = self._instance.get_landsklim_project().get_stations_sources()
        self.mlcb_stations.clear()
        self.mlcb_stations.setAdditionalLayers(stations.get_qgis_layers())
        self.mlcb_stations.currentIndexChanged.connect(self.on_stations_changed)
        if self.mlcb_stations.count() > 0:
            self.on_stations_changed(0)

        regressors_definition: List[RegressorDefinition] = self.regressors_to_regressor_definitions(self.__regressors)
        table_model = LandsklimTableModelRegressors(data=regressors_definition, max_window_size=max_window_allowed)
        self.table_variables.setModel(table_model)
        self.table_variables.horizontalHeader().setSectionResizeMode(0, QHeaderView.ResizeToContents)

        self.button_min_window.clicked.connect(self.click_button_min_windows)
        self.button_max_window.clicked.connect(self.click_button_max_windows)
        self.button_add.clicked.connect(self.click_button_add)
        self.button_remove.clicked.connect(self.click_button_remove)
        self.button_add_all.clicked.connect(self.click_button_add_all)
        self.button_remove_all.clicked.connect(self.click_button_remove_all)

        size_policy = self.label_neighborhood.sizePolicy()
        size_policy.setRetainSizeWhenHidden(True)
        self.label_neighborhood.setSizePolicy(size_policy)
        size_policy = self.spinbox_neighborhood.sizePolicy()
        size_policy.setRetainSizeWhenHidden(True)
        self.spinbox_neighborhood.setSizePolicy(size_policy)

        self.rb_local.clicked.connect(self.click_analysis_mode)
        self.rb_global.clicked.connect(self.click_analysis_mode)
        self.update_interpolation_type()

        self.checkbox_partial_correlations.setEnabled(False)

    def get_maximum_neighborhood_value(self, vector_layer: QgsVectorLayer):
        """
        Get features from selected shapefile
        """
        return vector_layer.featureCount() - 1

    def get_available_situations(self, vector_layer: QgsVectorLayer):
        """
        Get available situations from selected shapefile
        """
        situation_names = []
        for field in vector_layer.fields():  # type: QgsField
            situation_names.append(field.name())
        return situation_names

    def click_button_add(self):
        """
        Button action to add a new situation to the analysis
        """
        item = self.list_situations_available.selectedItems()[0] if len(self.list_situations_available.selectedItems()) > 0 else None
        if item is not None:
            item_removed = self.list_situations_available.takeItem(self.list_situations_available.row(item))
            self.list_situations_selected.addItem(item_removed)
        self.sort_lists()

    def click_button_remove(self):
        """
        Button action to remove a new situation to the analysis
        """
        item = self.list_situations_selected.selectedItems()[0] if len(
            self.list_situations_selected.selectedItems()) > 0 else None
        if item is not None:
            item_removed = self.list_situations_selected.takeItem(self.list_situations_selected.row(item))
            self.list_situations_available.addItem(item_removed)
        self.sort_lists()

    def click_button_add_all(self):
        """
        Button action to add all situations available to the analysis
        """
        for i in range(self.list_situations_available.count()):
            self.list_situations_selected.addItem(self.list_situations_available.takeItem(0))
        self.sort_lists()

    def click_button_remove_all(self):
        """
        Button action to remove every situation to the analysis
        """
        for i in range(self.list_situations_selected.count()):
            self.list_situations_available.addItem(self.list_situations_selected.takeItem(0))
        self.sort_lists()

    def click_analysis_mode(self):
        """
        Click on 'global' ou 'local' analysis radio buttons
        """
        self.update_interpolation_type()


    def update_interpolation_type(self):
        if self.get_interpolation_mode() == LandsklimAnalysisMode.Local:
            self.label_neighborhood.show()
            self.spinbox_neighborhood.show()
        else:
            self.label_neighborhood.hide()
            self.spinbox_neighborhood.hide()

    def store_edited_regressors(self):
        self.edited_regressors = {}
        table: LandsklimTableModelRegressors = self.table_variables.model()
        for i in range(table.rowCount()):
            regressor_name: str = table.data(table.index(i, 0))
            row_windows: str = table.data(table.index(i, 1))
            windows = [int(i) for i in row_windows.replace(' ', '').split(',')] if len(row_windows.replace(' ', '')) > 0 else []
            self.edited_regressors[regressor_name] = windows

    def set_table_window_data(self, data: Dict[str, List[int]]):
        table: LandsklimTableModelRegressors = self.table_variables.model()
        for i in range(table.rowCount()):
            regressor_name: str = table.data(table.index(i, 0))
            row_windows: str = table.data(table.index(i, 1))
            table.setData(table.index(i, 1), ", ".join(map(str, data[regressor_name])), Qt.EditRole)
        table.refresh()
        self.table_variables.repaint()

    def click_button_min_windows(self, checked: bool):
        """
        Click on min window button : display currently edited min window for each regressor on the table
        """
        if self.button_max_window.isChecked():
            self.set_table_window_data(self.edited_regressors)  # Reset to avoid storing min/max regressors
            self.button_max_window.setChecked(False)
        if checked:
            self.store_edited_regressors()  # Store currently edited regressors
            table: LandsklimTableModelRegressors = self.table_variables.model()
            min_regressors = {}
            for regressor_name, regressor_windows in self.edited_regressors.items():
                min_regressors[regressor_name] = [sorted(regressor_windows)[0]] if len(regressor_windows) > 0 else []
            self.set_table_window_data(min_regressors)
        else:
            # Button unchecked, restore windows as before
            self.set_table_window_data(self.edited_regressors)

    def click_button_max_windows(self, checked: bool):
        """
        Click on max window button : display currently edited max window for each regressor on the table
        """
        if self.button_min_window.isChecked():
            self.set_table_window_data(self.edited_regressors)  # Reset to avoid storing min/max regressors
            self.button_min_window.setChecked(False)
        if checked:
            self.store_edited_regressors()  # Store currently edited regressors
            table: LandsklimTableModelRegressors = self.table_variables.model()
            max_regressors = {}
            for regressor_name, regressor_windows in self.edited_regressors.items():
                max_regressors[regressor_name] = [sorted(regressor_windows, reverse=True)[0]] if len(regressor_windows) > 0 else []
            self.set_table_window_data(max_regressors)
        else:
            # Button unchecked, restore windows as before
            self.set_table_window_data(self.edited_regressors)

    def sort_lists(self):
        """
        Sort situations list
        """
        self.list_situations_available.sortItems()
        self.list_situations_selected.sortItems()

    def refresh_ui(self):
        """
        Refresh UI components after a specific action was triggered
        """
        pass

    def input_are_valid(self) -> Tuple[bool, List[str]]:
        """
        Check if user inputs are valid
        """
        errors = []
        is_ok = True

        if self.get_analysis_name() == "" or "[" in self.get_analysis_name() or "]" in self.get_analysis_name():
            is_ok = False
            errors.append("ERROR_ANALYSIS_NAME")
        if self.get_configuration() is None:
            is_ok = False
            errors.append("ERROR_NO_CONFIGURATION")
        phases = self.get_phases()
        if phases[0] == phases[1]:
            is_ok = False
            errors.append("ERROR_SAME_PHASE")
        if phases[0] is None:
            is_ok = False
            errors.append("ERROR_PHASE_1_NOT_DEFINED")
        if phases[1] is None:
            is_ok = False
            errors.append("ERROR_PHASE_2_NOT_DEFINED")
        if self.get_data_to_interpolate() is None:
            is_ok = False
            errors.append("ERROR_NO_STATIONS")
        if len(self.get_situations()) == 0:
            is_ok = False
            errors.append("ERROR_NO_SITUATIONS_SELECTED")
        return is_ok, errors

    def get_analysis_name(self) -> str:
        return self.le_name.text()

    def get_configuration(self):
        return self.cb_configuration.currentData()

    def get_phases(self) -> Tuple[str, str]:
        # phases definitions could become dict if parameters are required (if PhasePolynomial is implemented one day)
        phase_1 = self.cb_phase1.currentData()
        phase_2 = self.cb_phase2.currentData()
        return (phase_1, phase_2)

    def get_phase_degree(self) -> int:
        return self.spinbox_degree.value()

    def get_interpolation_mode(self) -> LandsklimAnalysisMode:
        """
        returns: True if interpolation is local, False otherwise
        :rtype: bool
        """
        is_local: bool = self.rb_local.isChecked()
        return LandsklimAnalysisMode.Local if is_local else LandsklimAnalysisMode.Global

    def get_neighborhood_size(self) -> int:
        """
        :returns: Neighborhood size when analysis is local
        :rtype: int
        """
        return self.spinbox_neighborhood.value()

    def get_use_partial_correlations(self) -> bool:
        return self.checkbox_partial_correlations.isChecked()

    def get_data_to_interpolate(self) -> QgsRasterLayer:
        return self.mlcb_stations.currentLayer()

    def get_situations(self) -> List[int]:
        situations = []
        for i in range(self.list_situations_selected.count()):
            item_index = self.list_situations_selected.item(i).data(QtCore.Qt.UserRole)
            situations.append(item_index)
        return situations

    def get_explicatives_variables(self) -> List[RegressorDefinition]:
        return self.table_variables.model().get_explicative_variables()

    def on_stations_changed(self, idx):
        if 0 <= idx < len(self._instance.get_landsklim_project().get_stations_sources()):
            layer = self.mlcb_stations.currentLayer()
            situations = self.get_available_situations(layer)
            self.list_situations_available.clear()
            self.list_situations_selected.clear()
            for i, situation in enumerate(situations):
                widget_item = QListWidgetItemDataComparable()
                widget_item.setText(situation)
                widget_item.setData(QtCore.Qt.UserRole, i)
                self.list_situations_available.addItem(widget_item)

            self.spinbox_neighborhood.setValue(30)
            maximum_neighorhood_value: int = self.get_maximum_neighborhood_value(layer)
            self.spinbox_neighborhood.setMaximum(maximum_neighorhood_value)
            if self.spinbox_neighborhood.value() > maximum_neighorhood_value:
                self.spinbox_neighborhood.setValue(maximum_neighorhood_value)
