import base64
import json
from json import JSONDecoder
from typing import Dict, Any

import numpy as np
import pandas as pd
from pykrige import OrdinaryKriging
from sklearn.linear_model import LinearRegression

from landsklim.lk.landsklim_constants import LandsklimLayerType
from landsklim.lk.landsklim_analysis import LandsklimAnalysis, LandsklimAnalysisMode
from landsklim.lk.landsklim_configuration import LandsklimConfiguration
from landsklim.lk.landsklim_interpolation import LandsklimInterpolation, LandsklimInterpolationType, ExtrapolationMode, \
    LandsklimRectangle
from landsklim.lk.landsklim_project import LandsklimProject
from landsklim.lk.map_layer import MapLayer, MapLayerCollection
from landsklim.lk.phase import IPhase
from landsklim.lk.polygons_definition import PolygonsDefinition
from landsklim.lk.regression_model import MultipleRegressionModel, SmoothingMode
from landsklim.lk.regressor import Regressor
from landsklim.serialization.landsklim_unpickler import PearsonRResultDummyClass

PUBLIC_ENUMS = {
    'LISDQSAnalysisMode': LandsklimAnalysisMode,  # Old name for LandsklimAnalysisMode
    'LandsklimAnalysisMode': LandsklimAnalysisMode,
    'SmoothingMode': SmoothingMode,
    'LISDQSInterpolationType': LandsklimInterpolationType,  # Old name of LandsklimInterpolationType
    'LandsklimInterpolationType': LandsklimInterpolationType,
    'ExtrapolationMode': ExtrapolationMode,
    'LandsklimLayerType': LandsklimLayerType
}

class LandsklimEncoder(json.JSONEncoder):
    """
    Serialize Landsklim classes to JSON

    Warning: JSON serialization does not distinguish between tuples and lists,
     and it is impossible to encode all tuples in the desired format.
     During deserialization, tuples are generally converted into list.
     Tuple attributes must be considered as lists in the Landsklim code.
     Tuple attributes in Landsklim are immutable, so this shouldn't be a problem when running the project
     once they've been converted to lists.
    """

    def landsklim_object_to_json(self, obj: Any) -> Dict:
        """
        Convert an object to a JSON dictionary

        :param obj: Object to convert
        :type obj: Any

        :returns: JSON dictionary
        :rtype: Dict
        """
        obj_state = obj.to_json()
        dct = {'__class__': obj.__class__.__name__}
        dct.update(obj_state)
        return dct

    def dict_to_json(self, obj: object, cls_name: str) -> Dict:
        """
        Convert the current state of an object to a JSON dictionary

        :param obj: Object to convert
        :type obj: Any

        :param cls_name: JSON attribute to identify the type
        :type cls_name: str

        :returns: JSON dictionary
        :rtype: Dict
        """
        dct = {'__class__': cls_name}
        state = obj.__dict__.copy()
        dct.update(state)
        return dct

    def default(self, obj):
        landsklim_types = (LandsklimRectangle, LandsklimProject, LandsklimConfiguration, LandsklimAnalysis,
                           MultipleRegressionModel, LandsklimInterpolation, IPhase,
                           Regressor, MapLayer, PolygonsDefinition)
        if isinstance(obj, landsklim_types):
            js = self.landsklim_object_to_json(obj)
            return js

        if isinstance(obj, tuple):
            return {'__class__': 'tuple', 'array': obj}
        if isinstance(obj, np.integer):
            return int(obj)
        if isinstance(obj, np.floating):
            return float(obj)
        if isinstance(obj, PearsonRResultDummyClass):  # In 0.5.1, LISDQSAnalysis.self._pearson_correlation[...][...] can be pickled as PearsonRResult
            return self.default(obj.to_json())

        if type(obj) in PUBLIC_ENUMS.values():
            return {"__enum__": str(obj)}

        if isinstance(obj, pd.DataFrame):
            state_str = obj.to_csv()
            # state_str = obj.to_json(orient='split')
            dct = {'__class__': 'pd.DataFrame', 'data': state_str}
            return dct
        if isinstance(obj, np.ndarray):
            return {"__class__": "np.array", "type": str(obj.dtype), "array": obj.tolist()}
        if isinstance(obj, pd.Series):
            dct = {'__class__': "pd.Series", "type": str(obj.dtype)}
            state = obj.to_dict()
            dct.update(state)
            return dct
        if isinstance(obj, LinearRegression):
            return self.dict_to_json(obj, 'LinearRegression')
        if isinstance(obj, OrdinaryKriging):
            dct = self.dict_to_json(obj, 'OrdinaryKriging')
            dct.pop('variogram_function')
            return dct

        return super().default(obj)
