from typing import Tuple, List

from PyQt5.QtWidgets import QComboBox

from processing.gui.wrappers import WidgetWrapper
from qgis.PyQt import QtCore
from qgis._core import QgsProcessingParameterDefinition
from landsklim.landsklim import Landsklim

"""
Analysis parameter
"""


class AnalysisWidgetWrapper(WidgetWrapper):
    """
    Represents the widget wrapper to select an Analysis from Processing parameters panel
    """

    def __init__(self, *args, **kwargs):
        self.panel: QComboBox = None
        super(AnalysisWidgetWrapper, self).__init__(*args, **kwargs)

    def on_combo_box_value_changed(self, index):
        """
        Emit signal specifying parameter value has changed
        """
        self.widgetValueHasChanged.emit(self)

    def createWidget(self):
        self.panel = QComboBox()
        self.panel.currentIndexChanged.connect(self.on_combo_box_value_changed)
        if Landsklim.instance().get_landsklim_project() is not None:
            for analysis in Landsklim.instance().get_landsklim_project().get_analysis():  # type: "LandsklimAnalysis"
                self.panel.addItem(analysis.to_string(), analysis)
        return self.panel

    def setValue(self, value: "LandsklimAnalysis"):
        self.panel.setCurrentIndex(self.panel.findData(value, QtCore.Qt.UserRole))

    def value(self) -> "LandsklimAnalysis":
        return self.panel.currentData(QtCore.Qt.UserRole)


class QgsProcessingParameterAnalysis(QgsProcessingParameterDefinition):
    """
    Represents the Processing Parameter Definition to represents an Analysis through Processing
    """

    def __init__(self, name='', description='', default=None, optional=False):
        QgsProcessingParameterDefinition.__init__(self, name, description, default, optional)
        base_package_name = __name__.split('.')[0]

        self.setMetadata({
            'widget_wrapper': {
                'class': '{0}.processing.processing_parameter_analysis.AnalysisWidgetWrapper'.format(base_package_name)}})

    def valueAsString(self, value: "LandsklimAnalysis", context) -> Tuple[str, bool]:
        return value.to_string(), True

    def valueAsPythonComment(self, value: "LandsklimAnalysis", context) -> str:
        return value.to_string()

    def valueAsPythonString(self, value: "LandsklimAnalysis", context) -> str:
        return value.to_string()

    def type(self) -> str:
        return 'QgsProcessingParameterAnalysis'


"""
Situation parameter
"""


class SituationWidgetWrapper(WidgetWrapper):
    """
    Represents the widget wrapper to select an analysis situation from Processing parameters panel
    """

    def __init__(self, *args, **kwargs):
        self.panel: QComboBox = None
        self.analysis: "LandsklimAnalysis" = None
        super(SituationWidgetWrapper, self).__init__(*args, **kwargs)

    def postInitialize(self, wrappers):
        """
        Once parameters' widget are initialized, link parent's widget signal to oneself
        """
        for wrapper in wrappers:
            if wrapper.param.name() == self.param.parent:
                self.parentValueChanged(wrapper)
                wrapper.widgetValueHasChanged.connect(self.parentValueChanged)
                break

    def parentValueChanged(self, wrapper):
        """
        When parent value changed (analysis changed), update situations to match analysis' situations
        """
        self.analysis = wrapper.value()
        self.fill_combobox()

    def fill_combobox(self):
        self.panel.clear()
        if self.analysis is not None:
            for situation in self.analysis.get_station_situations():  # type: int
                self.panel.addItem(self.analysis.get_situation_name(situation), situation)

    def createWidget(self):
        """
        Only create the combobox : we don't know values yet
        """
        self.panel = QComboBox()
        return self.panel

    def setValue(self, value: int):
        self.panel.setCurrentIndex(self.panel.findData(value, QtCore.Qt.UserRole))

    def value(self) -> int:
        return self.panel.currentData(QtCore.Qt.UserRole)


class QgsProcessingParameterSituation(QgsProcessingParameterDefinition):
    """
    Represents the Processing Parameter Definition to represents a situation through Processing
    """

    def __init__(self, name='', description='', parent=None, default=None, optional=False):
        QgsProcessingParameterDefinition.__init__(self, name, description, default, optional)
        self.parent = parent
        base_package_name = __name__.split('.')[0]  # according to context (plugin or tests case), base package name can be 'lisdqs_for_qgis' or 'lisdqs-for-qgis'
        self.setMetadata({
            'widget_wrapper': {
                'class': '{0}.processing.processing_parameter_analysis.SituationWidgetWrapper'.format(base_package_name)}})

    def isDynamic(self) -> bool:
        return True

    def dependsOnOtherParameters(self) -> List[str]:
        if self.parent is not None:
            return [self.parent]
        else:
            return []

    def valueAsString(self, value: int, context) -> Tuple[str, bool]:
        return str(value), True

    def valueAsPythonComment(self, value: int, context) -> str:
        return str(value)

    def valueAsPythonString(self, value: int, context) -> str:
        return str(value)

    def type(self) -> str:
        return 'QgsProcessingParameterSituation'


"""
Interpolation parameter
"""


class InterpolationWidgetWrapper(WidgetWrapper):
    """
    Represents the widget wrapper to select an analysis interpolation from Processing parameters panel
    """

    def __init__(self, *args, **kwargs):
        self.panel: QComboBox = None
        self.analysis: "LandsklimAnalysis" = None
        super(InterpolationWidgetWrapper, self).__init__(*args, **kwargs)

    def postInitialize(self, wrappers):
        """
        Once parameters' widget are initialized, link parent's widget signal to oneself
        """
        for wrapper in wrappers:
            if wrapper.param.name() == self.param.parent:
                self.parentValueChanged(wrapper)
                wrapper.widgetValueHasChanged.connect(self.parentValueChanged)
                break

    def parentValueChanged(self, wrapper):
        """
        When parent value changed (analysis changed), update situations to match analysis' situations
        """
        self.analysis = wrapper.value()
        self.fill_combobox()

    def fill_combobox(self):
        self.panel.clear()
        if self.analysis is not None:
            for interpolation in self.analysis.get_interpolations():  # type: "LandsklimInterpolation"
                self.panel.addItem(interpolation.get_name(), interpolation)

    def createWidget(self):
        """
        Only create the combobox : we don't know values yet
        """
        self.panel = QComboBox()
        return self.panel

    def setValue(self, value: "LandsklimInterpolation"):
        self.panel.setCurrentIndex(self.panel.findData(value, QtCore.Qt.UserRole))

    def value(self) -> "LandsklimInterpolation":
        return self.panel.currentData(QtCore.Qt.UserRole)


class QgsProcessingParameterInterpolation(QgsProcessingParameterDefinition):
    """
    Represents the Processing Parameter Definition to represents an Interpolation through Processing
    """

    def __init__(self, name='', description='', parent=None, default=None, optional=False):
        QgsProcessingParameterDefinition.__init__(self, name, description, default, optional)
        self.parent = parent
        base_package_name = __name__.split('.')[0]  # according to context (plugin or tests case), base package name can be 'lisdqs_for_qgis' or 'lisdqs-for-qgis'
        self.setMetadata({
            'widget_wrapper': {
                'class': '{0}.processing.processing_parameter_analysis.InterpolationWidgetWrapper'.format(base_package_name)}})

    def isDynamic(self) -> bool:
        return True

    def dependsOnOtherParameters(self) -> List[str]:
        if self.parent is not None:
            return [self.parent]
        else:
            return []

    def valueAsString(self, value: "LandsklimInterpolation", context) -> Tuple[str, bool]:
        return str(value), True

    def valueAsPythonComment(self, value: "LandsklimInterpolation", context) -> str:
        return str(value)

    def valueAsPythonString(self, value: "LandsklimInterpolation", context) -> str:
        return str(value)

    def type(self) -> str:
        return 'QgsProcessingParameterInterpolation'


