import abc
from typing import Union, Dict, Tuple, List, Optional
import os
import inspect

from PyQt5.QtCore import QVariant
from osgeo import gdal, gdal_array
from osgeo import osr
from qgis.PyQt.QtCore import QCoreApplication
from qgis._core import QgsProcessingParameterString, QgsProcessingParameterDefinition, QgsVectorLayer, \
    QgsFeatureRequest, QgsVectorFileWriter, QgsProject, QgsField, QgsFeature, QgsMapLayer, Qgis
from qgis.core import QgsProcessing, QgsProcessingAlgorithm, QgsProcessingException, QgsProcessingParameterRasterLayer, \
    QgsProcessingParameterNumber, QgsProcessingParameterRasterDestination, QgsRasterLayer, QgsProcessingParameterBoolean, QgsProcessingContext
from qgis import processing
import numpy as np

from landsklim.lk.utils import LandsklimUtils


class LandsklimProcessingAlgorithm(QgsProcessingAlgorithm):
    """
    Landsklim Processing algorithm.
    Handle rasters input/output
    Handle metadata collection
    Handle GDAL <> Numpy conversions methods
    """

    def __init__(self):
        super().__init__()
        self.__driver = gdal.GetDriverByName('GTiff')

    def tr(self, string: str, context_name: Optional[str] = None) -> str:
        """caller_class = inspect.stack()[1][0].f_locals["self"].__class__.__name__ if context_name is None else context_name
        res = QCoreApplication.translate(caller_class, string)"""
        res = QCoreApplication.translate('Processing', string)
        return res

    @abc.abstractmethod
    def createInstance(self):
        raise NotImplementedError()

    @abc.abstractmethod
    def name(self) -> str:
        """
        Unique name of the algorithm
        """
        raise NotImplementedError()

    @abc.abstractmethod
    def displayName(self) -> str:
        """
        Displayed name of the algorithm
        """
        raise NotImplementedError()

    def group(self) -> str:
        return self.tr('Landsklim')

    def groupId(self) -> str:
        return 'landsklim'

    @abc.abstractmethod
    def shortHelpString(self) -> str:
        raise NotImplementedError()

    @abc.abstractmethod
    def initAlgorithm(self, config=None):
        raise NotImplementedError()

    @staticmethod
    def source_to_array(raster_source: str, band: int = 1) -> np.array:
        return LandsklimUtils.source_to_array(raster_source, band)

    @abc.abstractmethod
    def processAlgorithm(self, parameters, context, feedback):
        raise NotImplementedError()

    def parameterAsAnalysis(self, parameters, name, context):
        return parameters[name]

    def parameterAsSituation(self, parameters, name, context):
        return parameters[name]

    def parameterAsInterpolation(self, parameters, name, context):
        return parameters[name]

    def rasters_average(self, layers: List[QgsRasterLayer]) -> np.ndarray:
        """
        Get the average value of a list of rasters

        :param layers: List of rasters
        :type layers: List[QgsRasterLayer]
        """
        array_sum: Optional[np.ndarray] = None
        total: int = 0
        for layer in layers:  # type: QgsRasterLayer
            if array_sum is None:
                array_sum = np.array(self.source_to_array(layer.source()))
            else:
                array_sum = array_sum + self.source_to_array(layer.source())
            total += 1
        mean: np.ndarray = array_sum / total
        return mean


    def write_point_geopackage(self, output_shapefile: str, source: QgsVectorLayer, data: np.ndarray, data_label: str, no_data: Union[int, float]):
        """
        Write point shapefile on disk.
        If the shapefile already exists, add/replace feature specified by data_label by new data

        :param output_shapefile: Path to the shapefile on disk
        :type output_shapefile: str

        :param source: Source shapefile as a reference if the output shapefile doesn't exist
        :type source: QgsVectorLaye

        :param data: Data to write on disk
        :type data: np.ndarray

        :param data_label: Feature name to add
        :type data_label: str

        :param no_data: No data value
        :type no_data: Union[int, float]
        """
        provider = "GPKG"  # source.dataProvider().storageType()
        print("[write_point_shapefile]")
        if not os.path.exists(output_shapefile):
            features_id = [feature.id() for feature in source.getFeatures()]
            # create a new layer with all features
            new_layer = source.materialize(QgsFeatureRequest().setFilterFids(features_id))
            save_options = QgsVectorFileWriter.SaveVectorOptions()
            save_options.driverName = provider
            save_options.fileEncoding = source.dataProvider().encoding()
            transform_context = QgsProject.instance().transformContext()
            error = QgsVectorFileWriter.writeAsVectorFormatV3(new_layer, output_shapefile, transform_context, save_options)
            if error[0] == QgsVectorFileWriter.NoError:
                print("Save Success !")
            else:
                print("Error !")
                print(error)

        layer: QgsVectorLayer = QgsVectorLayer(output_shapefile, '')
        success = layer.dataProvider().addAttributes([QgsField(data_label, QVariant.Double, "double")])
        layer.updateFields()
        res_field_index = layer.fields().indexFromName(data_label)
        layer.startEditing()
        for i, feature in enumerate(layer.getFeatures()):  # type: int, QgsFeature
            attrs = {res_field_index: float(round(data[i], 10))}
            layer.changeAttributeValues(feature.id(), attrs)
        layer.commitChanges()

    def get_spatial_reference(self, map_layer: QgsMapLayer) -> osr.SpatialReference:
        out_srs: osr.SpatialReference = osr.SpatialReference()
        out_srs.ImportFromWkt(map_layer.crs().toWkt())
        # out_srs.ImportFromEPSG(int(map_layer.crs().authid().split(':')[-1]))
        return out_srs

    def write_numpy_array(self, output_path: str, raster: np.ndarray):
        """
        Write a numpy array on disk

        :param output_path: Path to the file on disk
        :type output_path: str

        :param raster: Raster array to write on disk
        :type raster: np.ndarray
        """
        np.savez_compressed(output_path, raster=raster.astype(np.float64))

    def load_numpy_array(self, path: str) -> np.ndarray:
        """
        Load a numpy array from disk

        :param path: Path on disk
        :type path: str

        :returns: Raster array
        :type: np.ndarray
        """
        return np.load(path)["raster"]

    def write_raster(self, output_raster: str, raster: np.ndarray, srs: osr.SpatialReference, geotransform: tuple, no_data: Optional[Union[int, float]]):
        """
        Write raster on disk through GDAL

        :param output_raster: Path to the raster file on disk
        :type output_raster: str

        :param raster: Raster array to write on disk
        :type raster: np.ndarray

        :param srs: Spatial reference object to georeference the raster
        :type srs: osr.SpatialReference

        :param geotransform: Geotransform object to georeference the raster
        :type geotransform: tuple

        :param no_data: No data value
        :type no_data: Optional[Union[int, float]]
        """

        # Can't write on an existing raster if the layer is opened into QGIS.
        LandsklimUtils.free_path(output_raster)

        outdata: gdal.Dataset = self.__driver.Create(output_raster, raster.shape[1], raster.shape[0], 1, gdal.GDT_Float32)
        outdata.SetProjection(srs.ExportToWkt())
        outdata.SetGeoTransform(geotransform)
        outband: gdal.Band = outdata.GetRasterBand(1)
        if no_data is not None:
            print("[Raster output] Set {0} as no data".format(no_data))
            outband.SetNoDataValue(no_data)
        outband.WriteArray(raster)
        outband.FlushCache()
        outdata = None
        outband = None

    def get_raster_metadata(self, parameters: Dict, context: QgsProcessingContext, source_layer: QgsRasterLayer) -> Tuple[Optional[Union[int, float]], tuple]:
        """
        Load metadatas from input raster (no data and geotransform)

        :param parameters: Processing algorithm inputs
        :type parameters: Dict

        :param context: Processing context
        :type context: QgsProcessingContext

        :param source_layer: Input raster layer used to extract metadata.
        :type source_layer: QgsRasterLayer

        :returns: No data value and geotransform
        :rtype: Tuple[Optional[Union[int, float]], tuple]
        """

        source = gdal.Open(source_layer.source())
        geotransform = source.GetGeoTransform()
        no_data = source.GetRasterBand(1).GetNoDataValue()
        source = None

        custom_no_data: int = self.parameterAsInt(parameters, 'INPUT_CUSTOM_NO_DATA', context)
        if 'INPUT_CUSTOM_NO_DATA' in parameters and parameters['INPUT_CUSTOM_NO_DATA'] is not None:
            no_data = custom_no_data
        return no_data, geotransform

    def get_current_memory_usage(self):
        """
        Memory usage in kB
        """
        import os
        import platform
        if platform.system() == "Darwin":
            return 0
        elif platform.system() == 'Linux':
            with open('/proc/self/status') as f:
                memusage = f.read().split('VmRSS:')[1].split('\n')[0][:-3]
            return int(memusage.strip())
        else:
            return 0

    def qgis_version(self) -> Tuple[int, int]:
        """
        Get the QGIS version

        :returns: A tuple with the major and the minor. For example: (3, 30)
        :rtype: Tuple[int, int]
        """
        qgs_version = Qgis.version().split('.')
        qgs_version_major, qgs_version_minor = int(qgs_version[0]), int(qgs_version[1])
        return qgs_version_major, qgs_version_minor
