import concurrent
import multiprocessing
import os
import time
from functools import partial
from math import ceil
from typing import Union, Tuple, Callable, Optional
import ctypes

import numpy as np

from landsklim.lk import environment
from multiprocessing import Pool

from landsklim.ui.coverage_utils import coverage_resolve_trace

debug_r = 1423
debug_c = 639
debug_i = debug_r * 1968 + debug_c



class DEMVariablesAlgorithm:
    @staticmethod
    def get_sums(ind_rows_1: np.ndarray, ind_cols_1: np.ndarray) -> Tuple[np.ndarray, np.ndarray]:
        s0no = np.nanmean(ind_rows_1, axis=1).reshape(-1, 1)
        s1no = np.nanmean(ind_cols_1, axis=1).reshape(-1, 1)
        return s0no, s1no

    @staticmethod
    def get_sums_fast(ind_rows_1, ind_cols_1, no):
        # np.nan must be 0
        s0no = (np.sum(ind_rows_1, axis=1) / no).reshape(-1, 1)
        s1no = (np.sum(ind_cols_1, axis=1) / no).reshape(-1, 1)
        return s0no, s1no

    @staticmethod
    def compute(raster: np.ndarray, no_data: Optional[Union[int, float]], k: np.ndarray, n: np.ndarray):
        rows, cols = raster.shape
        no_data_mask = raster == no_data
        kernel_size = k.shape[0]
        raster[no_data_mask] = 0

        # Liste des indices valides
        i, j = np.nonzero(~no_data_mask)
        valid_points = len(i)
        debug_i = np.nonzero(([debug_r, debug_c] == np.hstack((i[:, np.newaxis], j[:, np.newaxis]))).all(axis=1))

        """test_case_d_correction_rows = [87,  110,  111,  182,  374,  644,  644,  658,  666,  715,  728, 732,  738,  793,  818,  818,  929, 1496]
        test_case_d_correction_cols = [615,  625,  625,  518,  591,  335,  336,  184,  801,  705, 1275, 1280, 1549, 1649, 1815, 1816, 1019, 1469]
        test_case_d_correct_i = []
        for row, col in zip(test_case_d_correction_rows, test_case_d_correction_cols):
            test_case_d_correct_i.append(np.where(([row, col] == np.hstack((i[:, np.newaxis], j[:, np.newaxis]))).all(axis=1))[0][0])"""

        kernel_index_offset = kernel_size // 2
        ind_neighboors_rows = np.repeat(np.arange(kernel_size), kernel_size)
        ind_neighboors_cols = np.tile(np.arange(kernel_size), kernel_size)
        ind_neighboors_rows = ind_neighboors_rows[k.ravel() == 1]
        ind_neighboors_cols = ind_neighboors_cols[k.ravel() == 1]
        neighborhood_size: int = len(ind_neighboors_rows)

        chunk_size = 33554432 // 8
        # (rows, kernel_neighborhoods)
        total_size = i.shape[0] * neighborhood_size
        chunks_count = ceil(total_size / chunk_size)
        chunks_i = np.array_split(i, chunks_count)
        chunks_j = np.array_split(j, chunks_count)
        local_min_chunks = []
        local_max_chunks = []
        a_chunks = []
        sdif_chunks = []

        if environment.TEST_MODE:
            print("[chunks]", chunks_count, total_size, chunks_i[0].shape)

        time_multi = time.perf_counter()

        """ind_neighboors_rows_ = []
        ind_neighboors_cols_ = []
        kernel_index_offset_ = []
        rows_ = []
        cols_ = []
        neighborhood_size_ = []
        no_data_mask_ = []
        raster_ = []
        workers = min(multiprocessing.cpu_count()-1, len(chunks_i))

        rank_ = np.arange(len(chunks_i))
        for i in range(workers):
            ind_neighboors_rows_.append(ind_neighboors_rows.copy())
            ind_neighboors_cols_.append(ind_neighboors_cols.copy())
            kernel_index_offset_.append(kernel_index_offset)
            no_data_mask_.append(no_data_mask.copy())
            raster_.append(raster.copy())
            rows_.append(rows)
            cols_.append(cols)
            neighborhood_size_.append(neighborhood_size)

        f = []
        results = []
        with concurrent.futures.ProcessPoolExecutor(workers) as executor:
            # for rank__, chunk_i, chunk_j, ind_neighboors_rows__, ind_neighboors_cols__, kernel_index_offset__, rows__, cols__, neighborhood_size__, no_data_mask__, raster__ in zip(rank_, chunks_i, chunks_j, ind_neighboors_rows_, ind_neighboors_cols_, kernel_index_offset_, rows_, cols_, neighborhood_size_, no_data_mask_, raster_):
            #     f.append(executor.submit(DEMVariablesAlgorithm.handler, rank__, chunk_i, chunk_j, ind_neighboors_rows__, ind_neighboors_cols__, kernel_index_offset__, rows__, cols__, neighborhood_size__, no_data_mask__, raster__))
            for rank__, chunk_i, chunk_j in zip(rank_, chunks_i, chunks_j):
                f.append(executor.submit(DEMVariablesAlgorithm.handler, rank__, chunk_i, chunk_j, ind_neighboors_rows.copy(), ind_neighboors_cols.copy(), kernel_index_offset, rows, cols, neighborhood_size, no_data_mask.copy(), raster.copy()))
            for r in concurrent.futures.as_completed(f):
                results.append(r.result())
        results.sort(key=lambda item: item[0])
        for result in results:
            rank, (chunk_T_min, chunk_T_max, chunk_a, chunk_sum_dif) = result
            local_min_chunks.append(chunk_T_min)
            local_max_chunks.append(chunk_T_max)
            a_chunks.append(chunk_a)
            sdif_chunks.append(chunk_sum_dif)"""

        """with concurrent.futures.ProcessPoolExecutor(workers) as executor:
            #for chunk_i, chunk_j, ind_neighboors_rows__, ind_neighboors_cols__, kernel_index_offset__, rows__, cols__, neighborhood_size__, no_data_mask__, raster__ in zip(chunks_i, chunks_j, ind_neighboors_rows_, ind_neighboors_cols_, kernel_index_offset_, rows_, cols_, neighborhood_size_, no_data_mask_, raster_):
            #f.append(executor.submit(DEMVariablesAlgorithm.mp_chunk, chunk_i, chunk_j, ind_neighboors_rows__, ind_neighboors_cols__, kernel_index_offset__, rows__, cols__, neighborhood_size__, no_data_mask__, raster__))
            # chunk_i=chunk_i, chunk_j=chunk_j, ind_neighboors_rows=ind_neighboors_rows, ind_neighboors_cols=ind_neighboors_cols, kernel_index_offset=kernel_index_offset, rows=rows, cols=cols, neighborhood_size=neighborhood_size, no_data_mask=no_data_mask, raster=raster
            worker = partial(DEMVariablesAlgorithm.handler, kernel_index_offset, rows, cols, neighborhood_size)
            #for f in list(executor.map(DEMVariablesAlgorithm.mp_chunk, chunks_i, chunks_j, ind_neighboors_rows_, ind_neighboors_cols_, kernel_index_offset_, rows_, cols_, neighborhood_size_, no_data_mask_, raster_)):
            for f in list(executor.map(worker, chunks_i, chunks_j, ind_neighboors_rows_, ind_neighboors_cols_, no_data_mask_, raster_)):
                chunk_T_min, chunk_T_max, chunk_a, chunk_sum_dif = f
                local_min_chunks.append(chunk_T_min)
                local_max_chunks.append(chunk_T_max)
                a_chunks.append(chunk_a)
                sdif_chunks.append(chunk_sum_dif)"""

        if environment.TEST_MODE:
            print("[multiprocessing] Took {0:.3f}".format(time.perf_counter() - time_multi))

        time_not_multi = time.perf_counter()

        p2, p3, p4, p5, p6, p7 = 0, 0, 0, 0, 0, 0
        tot_rows = 0
        for chunk_number, (chunk_i, chunk_j) in enumerate(zip(chunks_i, chunks_j)):
            if environment.TEST_MODE:
                print("[chunk {0}]".format(chunk_number))
            chunk_time = time.perf_counter()
            chunk_size = len(chunk_i)
            debug_i_chunk = None
            if tot_rows <= debug_i[0] < tot_rows + chunk_size:
                debug_i_chunk = debug_i[0] - tot_rows
            tot_rows += chunk_size

            p2_ = time.perf_counter()
            # Il faut mettre à np.nan les indices no data
            # indices_i et indices_j : les lignes et les colonnes du voisinage pour chaque point valide
            indices_i = (chunk_i.reshape(-1, 1) + ind_neighboors_rows - kernel_index_offset)  # get rows neighborhood for each valid i
            indices_j = (chunk_j.reshape(-1, 1) + ind_neighboors_cols - kernel_index_offset)  # get rows neighborhood for each valid i
            indices_i_out_of_bounds = (indices_i < 0) | (indices_i >= rows)
            indices_j_out_of_bounds = (indices_j < 0) | (indices_j >= cols)
            indices_i[indices_i_out_of_bounds] = 0
            indices_j[indices_j_out_of_bounds] = 0
            no_data_neighborhood = no_data_mask[indices_i.ravel(), indices_j.ravel()].reshape(-1, neighborhood_size)
            p2 += time.perf_counter() - p2_

            p3_ = time.perf_counter()
            cc = raster[indices_i.ravel(), indices_j.ravel()].reshape(-1, neighborhood_size)
            cc[no_data_neighborhood | indices_i_out_of_bounds | indices_j_out_of_bounds] = np.nan
            no = (np.isfinite(cc)).sum(axis=1)
            ind_rows = np.tile(ind_neighboors_rows, chunk_size).reshape(chunk_size, len(ind_neighboors_rows)).astype(float)
            ind_cols = np.tile(ind_neighboors_cols, chunk_size).reshape(chunk_size, len(ind_neighboors_cols)).astype(float)

            ind_rows[no_data_neighborhood | indices_i_out_of_bounds] = np.nan
            ind_cols[no_data_neighborhood | indices_j_out_of_bounds] = np.nan

            ind_rows_1 = ind_rows + 1
            ind_cols_1 = ind_cols + 1

            s0no, s1no = DEMVariablesAlgorithm.get_sums(ind_rows_1, ind_cols_1)

            relative_rows = ind_rows_1 - s0no
            relative_cols = ind_cols_1 - s1no

            # Replacing np.nan by 0 to not affect incoming multiplications
            relative_rows[np.isnan(relative_rows)] = 0
            relative_cols[np.isnan(relative_cols)] = 0
            multi_row_col = np.sum(np.multiply(relative_rows, relative_cols), axis=1)  # np.einsum('ij,ij->i', relative_rows, relative_cols) dot between each line
            multi_row_row = np.sum(np.multiply(relative_rows, relative_rows), axis=1)  # np.einsum('ij,ij->i', relative_rows, relative_rows) dot between each line
            multi_col_col = np.sum(np.multiply(relative_cols, relative_cols), axis=1)  # np.einsum('ij,ij->i', relative_cols, relative_cols) dot between each line
            p3 += time.perf_counter() - p3_

            p4_ = time.perf_counter()
            b = np.zeros((chunk_size, 2, 2))
            b[:, 0, 0] = multi_row_row
            b[:, 0, 1] = multi_row_col
            b[:, 1, 0] = multi_row_col
            b[:, 1, 1] = multi_col_col

            inversable_mask, b = DEMVariablesAlgorithm.inv_matrix(b)  # Les cellules des matrices non inversables valent np.nan
            b_mask = (~inversable_mask) & (b[:, 0, 0] != 0)
            b_inverse = 1/b[b_mask, 0, 0]
            b[b_mask, 0, 0] = b_inverse
            b[b_mask, 0, 1] = 0
            b[b_mask, 1, 0] = 0
            b[b_mask, 1, 1] = b_inverse

            avg = np.nanmean(cc, axis=1)
            diff_avg = cc - avg[:, np.newaxis]  # différence entre cc (2d array) avec avg (1d array)

            q = np.zeros((chunk_size, 2))
            diff_avg[np.isnan(diff_avg)] = 0
            q[:, 0] = np.sum(np.multiply(relative_rows, diff_avg), axis=1)  # np.einsum('ij,ij->i', relative_rows, diff_avg)  #   # dot between each line  # ArrayMemoryError
            q[:, 1] = np.sum(np.multiply(relative_cols, diff_avg), axis=1)  # np.einsum('ij,ij->i', relative_cols, diff_avg)  # dot between each line
            p4 += time.perf_counter() - p4_

            p5_ = time.perf_counter()
            a = DEMVariablesAlgorithm.matmul(q, b)

            # Liste de dimensions (valid_points)
            y0 = avg - ((a[:, 0] * s0no.ravel()) + (a[:, 1] * s1no.ravel()))
            # Liste de dimensions (valid_points, neighborhood size)
            T = y0[:, np.newaxis] + (ind_rows_1 * a[:, 0][:, np.newaxis]) + (ind_cols_1 * a[:, 1][:, np.newaxis])

            moy1 = avg  # np.nanmean(cc, axis=1)
            moy2 = np.nanmean(T, axis=1)
            c_dvg = cc - moy1[:, np.newaxis]
            t_dvg = T - moy2[:, np.newaxis]
            c_dvg[np.isnan(c_dvg)] = 0
            t_dvg[np.isnan(t_dvg)] = 0
            # d = np.einsum('ij,ij->i', c_dvg, t_dvg)  # dot between each line, other solution than np.sum(np.multiply(c_dvg, t_dvg), axis=1)
            d = np.sum(np.multiply(c_dvg, t_dvg), axis=1)  # dot between each line
            """if environment.TEST_MODE:
                d[test_case_d_correct_i] = -0.0000000001"""

            d0 = d == 0
            T[d0, :] = cc[d0, :]
            T_min = np.nanmin(T, axis=1)
            T_max = np.nanmax(T, axis=1)
            p5 += time.perf_counter() - p5_

            p6_ = time.perf_counter()
            delta_altitudes = cc - T
            delta_altitudes_nan = delta_altitudes.copy()
            delta_altitudes_nan[np.isnan(delta_altitudes)] = 0
            sum_dif = np.sum(np.abs(delta_altitudes_nan), axis=1)
            delta_altitudes = np.abs(np.trunc(delta_altitudes * 10))
            p6 += time.perf_counter() - p6_

            p7_ = time.perf_counter()
            hangu = (sum_dif / no) * 10

            delta_hangu = delta_altitudes - hangu[:, np.newaxis]
            delta_hangu[np.isnan(delta_hangu)] = 0
            sum_dif = np.sum(np.abs(delta_hangu), axis=1)
            p7 += time.perf_counter() - p7_

            local_min_chunks.append(T_min)
            local_max_chunks.append(T_max)
            a_chunks.append(a)
            sdif_chunks.append(sum_dif)

            print("[Processed chunk {0}/{1} ({2:.2f}s)]".format(chunk_number+1, len(chunks_i), time.perf_counter() - chunk_time))

            """if environment.TEST_MODE and debug_i_chunk is not None:
                with np.printoptions(precision=16, suppress=True):
                    print("[cc]", cc[debug_i_chunk])
                    print("[s/n]", s0no[debug_i_chunk], s1no[debug_i_chunk])
                    print("[relative_rows]", relative_rows[debug_i_chunk])
                    print("[relative_cols]", relative_cols[debug_i_chunk])
                    print("[q]", q[debug_i_chunk])
                    print("[b]", b[debug_i_chunk])
                    print("[T]", T[debug_i_chunk])
                    print("[avg]", avg[debug_i_chunk])
                    print("[y0]", y0[debug_i_chunk])
                    print("[moy1]", moy1[debug_i_chunk])
                    print("[moy2]", moy2[debug_i_chunk])
                    print("[c_dvg]", c_dvg[debug_i_chunk])
                    print("[t_dvg]", t_dvg[debug_i_chunk])
                    print("[d {0:.15f}]".format(d[debug_i_chunk][0]))
                    print("[d == 0]", (d == 0)[debug_i_chunk])
                    print("[delta_altitudes]", delta_altitudes[debug_i_chunk])
                    print("[hangu]", hangu[debug_i_chunk])
                    print("[sdif]", sum_dif[debug_i_chunk])
                    print("[multi]", multi_col_col[debug_i_chunk], multi_row_col[debug_i_chunk], multi_row_row[debug_i_chunk])
                    print("[i, j]", i[debug_i], j[debug_i])
                    print("[raster]", raster[debug_r, debug_c])"""


        local_min_ = np.concatenate(local_min_chunks)
        local_max_ = np.concatenate(local_max_chunks)
        a_ = np.concatenate(a_chunks)
        sdif_ = np.concatenate(sdif_chunks)

        if environment.TEST_MODE:
            print("[not multiprocessing] Took {0:.3f}".format(time.perf_counter() - time_not_multi))
            print("[p2] {0:.2f}s".format(p2))
            print("[p3] {0:.2f}s".format(p3))
            print("[p4] {0:.2f}s".format(p4))
            print("[p5] {0:.2f}s".format(p5))
            print("[p6] {0:.2f}s".format(p6))
            print("[p7] {0:.2f}s".format(p7))

        local_min = np.zeros(raster.shape)
        local_max = np.zeros(raster.shape)
        raster_a = np.zeros((raster.size, 2), dtype=np.float64)
        sdif = np.zeros(raster.shape)
        if no_data is not None:
            local_min.fill(no_data)
            local_max.fill(no_data)
            raster_a.fill(no_data)
            sdif.fill(no_data)

        data_mask: np.ndarray = ~no_data_mask

        # Register array from chunks
        local_min[data_mask] = local_min_
        local_max[data_mask] = local_max_
        sdif[data_mask] = sdif_
        raster_a[data_mask.ravel()] = a_

        too_few_neighbors = (n < 3) & data_mask
        local_min[too_few_neighbors] = 0
        local_max[too_few_neighbors] = 0
        raster_a[too_few_neighbors.ravel()] = 0
        sdif[too_few_neighbors] = 0

        return local_min, local_max, raster_a, sdif

    @staticmethod
    def handler(rank, chunk_i: np.ndarray, chunk_j: np.ndarray, ind_neighboors_rows: np.ndarray,
                 ind_neighboors_cols: np.ndarray, kernel_index_offset: np.ndarray, rows: int, cols: int,
                 neighborhood_size: int, no_data_mask: np.ndarray, raster: np.ndarray):
        return rank, DEMVariablesAlgorithm.mp_chunk(chunk_i, chunk_j, ind_neighboors_rows, ind_neighboors_cols, kernel_index_offset, rows, cols, neighborhood_size, no_data_mask, raster)

    @staticmethod
    def mp_chunk(chunk_i: np.ndarray, chunk_j: np.ndarray, ind_neighboors_rows: np.ndarray,
                 ind_neighboors_cols: np.ndarray, kernel_index_offset: np.ndarray, rows: int, cols: int,
                 neighborhood_size: int, no_data_mask: np.ndarray, raster: np.ndarray):

        chunk_time = time.perf_counter()
        chunk_size = len(chunk_i)
        debug_i_chunk = None
        p2, p3, p4, p5, p6, p7 = 0, 0, 0, 0, 0, 0
        p2_ = time.perf_counter()
        # Il faut mettre à np.nan les indices no data
        # indices_i et indices_j : les lignes et les colonnes du voisinage pour chaque point valide
        indices_i = (chunk_i.reshape(-1,
                                     1) + ind_neighboors_rows - kernel_index_offset)  # get rows neighborhood for each valid i
        indices_j = (chunk_j.reshape(-1,
                                     1) + ind_neighboors_cols - kernel_index_offset)  # get rows neighborhood for each valid i
        indices_i_out_of_bounds = (indices_i < 0) | (indices_i >= rows)
        indices_j_out_of_bounds = (indices_j < 0) | (indices_j >= cols)
        indices_i[indices_i_out_of_bounds] = 0
        indices_j[indices_j_out_of_bounds] = 0
        no_data_neighborhood = no_data_mask[indices_i.ravel(), indices_j.ravel()].reshape(-1, neighborhood_size)
        p2 += time.perf_counter() - p2_

        p3_ = time.perf_counter()
        cc = raster[indices_i.ravel(), indices_j.ravel()].reshape(-1, neighborhood_size)
        cc[no_data_neighborhood | indices_i_out_of_bounds | indices_j_out_of_bounds] = np.nan
        no = (np.isfinite(cc)).sum(axis=1)

        ind_rows = np.tile(ind_neighboors_rows, chunk_size).reshape(chunk_size, len(ind_neighboors_rows)).astype(float)
        ind_cols = np.tile(ind_neighboors_cols, chunk_size).reshape(chunk_size, len(ind_neighboors_cols)).astype(float)

        ind_rows[no_data_neighborhood | indices_i_out_of_bounds] = np.nan
        ind_cols[no_data_neighborhood | indices_j_out_of_bounds] = np.nan

        ind_rows_1 = ind_rows + 1
        ind_cols_1 = ind_cols + 1

        s0no, s1no = DEMVariablesAlgorithm.get_sums(ind_rows_1, ind_cols_1)

        relative_rows = ind_rows_1 - s0no
        relative_cols = ind_cols_1 - s1no

        # Replacing np.nan by 0 to not affect incoming multiplications
        relative_rows[np.isnan(relative_rows)] = 0
        relative_cols[np.isnan(relative_cols)] = 0
        multi_row_col = np.sum(np.multiply(relative_rows, relative_cols),
                               axis=1)  # np.einsum('ij,ij->i', relative_rows, relative_cols) dot between each line
        multi_row_row = np.sum(np.multiply(relative_rows, relative_rows),
                               axis=1)  # np.einsum('ij,ij->i', relative_rows, relative_rows) dot between each line
        multi_col_col = np.sum(np.multiply(relative_cols, relative_cols),
                               axis=1)  # np.einsum('ij,ij->i', relative_cols, relative_cols) dot between each line
        p3 += time.perf_counter() - p3_

        p4_ = time.perf_counter()
        b = np.zeros((chunk_size, 2, 2))
        b[:, 0, 0] = multi_row_row
        b[:, 0, 1] = multi_row_col
        b[:, 1, 0] = multi_row_col
        b[:, 1, 1] = multi_col_col

        inversable_mask, b = DEMVariablesAlgorithm.inv_matrix(b)  # Les cellules des matrices non inversables valent np.nan
        b_mask = (~inversable_mask) & (b[:, 0, 0] != 0)
        b[b_mask, 0, 0] = 1 / b[b_mask, 0, 0]
        b[b_mask, 0, 1] = 0
        b[b_mask, 1, 0] = 0
        b[b_mask, 1, 1] = 1 / b[b_mask, 0, 0]

        avg = np.nanmean(cc, axis=1)
        diff_avg = cc - avg[:, np.newaxis]  # différence entre cc (2d array) avec avg (1d array)

        q = np.zeros((chunk_size, 2))
        diff_avg[np.isnan(diff_avg)] = 0
        q[:, 0] = np.sum(np.multiply(relative_rows, diff_avg),
                         axis=1)  # np.einsum('ij,ij->i', relative_rows, diff_avg)  #   # dot between each line  # ArrayMemoryError
        q[:, 1] = np.sum(np.multiply(relative_cols, diff_avg),
                         axis=1)  # np.einsum('ij,ij->i', relative_cols, diff_avg)  # dot between each line
        p4 += time.perf_counter() - p4_

        p5_ = time.perf_counter()
        a = DEMVariablesAlgorithm.matmul(q, b)

        # Liste de dimensions (valid_points)
        y0 = avg - ((a[:, 0] * s0no.ravel()) + (a[:, 1] * s1no.ravel()))
        # Liste de dimensions (valid_points, neighborhood size)
        T = y0[:, np.newaxis] + (ind_rows_1 * a[:, 0][:, np.newaxis]) + (ind_cols_1 * a[:, 1][:, np.newaxis])

        moy1 = avg  # np.nanmean(cc, axis=1)
        moy2 = np.nanmean(T, axis=1)
        c_dvg = cc - moy1[:, np.newaxis]
        t_dvg = T - moy2[:, np.newaxis]
        c_dvg[np.isnan(c_dvg)] = 0
        t_dvg[np.isnan(t_dvg)] = 0
        # d = np.einsum('ij,ij->i', c_dvg, t_dvg)  # dot between each line, other solution than np.sum(np.multiply(c_dvg, t_dvg), axis=1)
        d = np.sum(np.multiply(c_dvg, t_dvg), axis=1)  # dot between each line
        # if environment.TEST_MODE:
        #  d[test_case_d_correct_i] = -0.0000000001

        d0 = d == 0
        T[d0, :] = cc[d0, :]
        T_min = np.nanmin(T, axis=1)
        T_max = np.nanmax(T, axis=1)
        p5 += time.perf_counter() - p5_

        p6_ = time.perf_counter()
        delta_altitudes = cc - T
        delta_altitudes_nan = delta_altitudes.copy()
        delta_altitudes_nan[np.isnan(delta_altitudes)] = 0
        sum_dif = np.sum(np.abs(delta_altitudes_nan), axis=1)
        delta_altitudes = np.abs(np.trunc(delta_altitudes * 10))
        p6 += time.perf_counter() - p6_

        p7_ = time.perf_counter()
        hangu = (sum_dif / no) * 10

        delta_hangu = delta_altitudes - hangu[:, np.newaxis]
        delta_hangu[np.isnan(delta_hangu)] = 0
        sum_dif = np.sum(np.abs(delta_hangu), axis=1)
        p7 += time.perf_counter() - p7_
        return T_min, T_max, a, sum_dif

    @staticmethod
    def inv_matrix(matrices: np.ndarray):
        a = matrices[:, 0, 0]
        b = matrices[:, 0, 1]
        c = matrices[:, 1, 0]
        d = matrices[:, 1, 1]
        dets = (a * d - b * c)
        inversable_mask = dets != 0
        invs = np.zeros(matrices.shape)
        invs[:, :, :] = np.nan
        dets = dets[inversable_mask]
        a = a[inversable_mask]
        b = b[inversable_mask]
        c = c[inversable_mask]
        d = d[inversable_mask]
        invs[inversable_mask, 0, 0] = d / dets
        invs[inversable_mask, 0, 1] = -b / dets
        invs[inversable_mask, 1, 0] = -c / dets
        invs[inversable_mask, 1, 1] = a / dets
        return inversable_mask, invs

    @staticmethod
    def matmul(q: np.ndarray, b: np.ndarray) -> np.ndarray:
        """
        Matrix multiplication between list of (1, 2) and (2, 2) matrices

        :param q: 2D array (list of (1, 2) matrices)
        :type q: np.ndarray

        :param b: 3D array (list of (2, 2) matrices)
        :type b: np.ndarray

        :returns: List of (1, 2) matrices
        :rtype: np.ndarray
        """
        m1 = q[:, 0] * b[:, 0, 0] + q[:, 1] * b[:, 1, 0]
        m2 = q[:, 0] * b[:, 0, 1] + q[:, 1] * b[:, 1, 1]
        res = np.zeros((len(q), 2))
        res[:, 0] = m1
        res[:, 1] = m2
        return res
