import gc
import os.path
from functools import partial
from multiprocessing import Pool
import multiprocessing as mp
from typing import Union, Tuple, Callable
import ctypes

from osgeo import gdal
from osgeo import osr
from osgeo.osr import SpatialReference
from qgis.PyQt.QtCore import QCoreApplication
from qgis._core import QgsProcessingParameterRasterLayer
from qgis.core import QgsProcessing, QgsProcessingAlgorithm, QgsProcessingException, QgsRasterLayer
from qgis import processing
import numpy as np
import math
from scipy.signal import convolve2d

from landsklim.processing.landsklim_processing_regressor_algorithm import LandsklimProcessingRegressorAlgorithm
from landsklim.lk.utils import LandsklimUtils
from landsklim.lk import environment


class SineProcessingAlgorithm(LandsklimProcessingRegressorAlgorithm):
    """
    Processing algorithm computing sine from a DEM (from its orientation regressor)
    """

    INPUT_ORIENTATION = 'INPUT_ORIENTATION'

    def __init__(self):
        super().__init__({'OUTPUT': 'Sine raster'})

    def createInstance(self):
        return SineProcessingAlgorithm()

    def name(self) -> str:
        """
        Unique name of the algorithm
        """
        return 'sine'

    def displayName(self) -> str:
        """
        Displayed name of the algorithm
        """
        return self.tr('Sine')

    def shortHelpString(self) -> str:
        return self.tr('Compute windowed sine from DEM')

    def add_dependencies(self):
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                SineProcessingAlgorithm.INPUT_ORIENTATION,
                self.tr('Orientation', "LandsklimProcessingRegressorAlgorithm")
            )
        )

    def compute_sine(self, orientations: np.array) -> np.array:
        """
        Compute sine of a raster

        :param orientations: Orientations raster. No data is already represented as np.nan
        :type orientations: np.ndarray

        :returns: Sine values for each cell of the raster
        :rtype: np.ndarray
        """
        return np.sin(np.deg2rad(orientations))

    def processAlgorithm(self, parameters, context, feedback):
        """
        Called when a processing algorithm is run
        """
        # Load input raster and its metadata
        input_raster: QgsRasterLayer = self.parameterAsRasterLayer(parameters, 'INPUT', context)
        input_orientation: QgsRasterLayer = self.parameterAsRasterLayer(parameters, SineProcessingAlgorithm.INPUT_ORIENTATION, context)

        no_data, geotransform = self.get_raster_metadata(parameters, context, input_raster)
        input_window = self.parameterAsInt(parameters, 'INPUT_WINDOW', context)
        additional_variables_folder = self.parameterAsString(parameters, 'ADDITIONAL_VARIABLES_FOLDER', context).strip()
        out_srs: SpatialReference = self.get_spatial_reference(input_raster)

        # Path of the layer is given. If a temporary layer is selected, layer is created in qgis temp dir
        out_sine = self.parameterAsOutputLayer(parameters, 'OUTPUT', context)

        np_orientations = LandsklimUtils.raster_to_array(input_orientation)

        """regressor_orientation = RegressorFactory.get_regressor(RegressorOrientation.class_name(), input_window, 1)
        orientation_raster_path = regressor_orientation.get_path()
        if not os.path.exists(orientation_raster_path):
            raise RuntimeError("Orientation raster is not found")

        np_orientations = LandsklimUtils.source_to_array(orientation_raster_path)"""
        output_no_data = -9999
        np_orientations[np_orientations == output_no_data] = np.nan  # no_data is the same for each regressors

        np_sine = self.compute_sine(np_orientations)
        np_sine[np.isnan(np_sine)] = output_no_data

        self.write_raster(out_sine, np_sine, out_srs, geotransform, output_no_data)

        return {'OUTPUT': out_sine}
