import time
from typing import Union

from osgeo import gdal
from osgeo import osr
from osgeo.osr import SpatialReference
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import QgsProcessing, QgsProcessingAlgorithm, QgsProcessingException, QgsProcessingParameterRasterLayer, \
    QgsProcessingParameterNumber, QgsProcessingParameterRasterDestination, QgsRasterLayer, QgsProcessingParameterBoolean
from qgis import processing
import numpy as np
import math

from scipy import signal
from scipy.signal import convolve2d, correlate2d
from scipy.ndimage import maximum_filter, minimum_filter, convolve, generic_filter

from landsklim.processing.landsklim_processing_regressor_algorithm import LandsklimProcessingRegressorAlgorithm
from landsklim.lk.utils import LandsklimUtils


class EncasementProcessingAlgorithm(LandsklimProcessingRegressorAlgorithm):
    """
    Processing algorithm computing altitude average from a DEM
    """

    def __init__(self):
        super().__init__({'OUTPUT': 'Encasement raster'})

    def createInstance(self):
        return EncasementProcessingAlgorithm()

    def name(self) -> str:
        """
        Unique name of the algorithm
        """
        return 'encasement'

    def displayName(self) -> str:
        """
        Displayed name of the algorithm
        """
        return self.tr('Encasement')

    def shortHelpString(self) -> str:
        return self.tr('Compute windowed encasement from DEM')

    def add_dependencies(self):
        """
        No dependencies
        """
        pass

    def compute_encasement(self, raster: np.array, kernel_size: int, no_data: Union[int, float], output_no_data: Union[int, float]):
        k = self.create_kernel(kernel_size)
        no_data_mask = raster == no_data

        convolution_kernel = np.where(k == 0, 0, -1)
        convolution_kernel[kernel_size // 2, kernel_size // 2] = k.sum() - 1
        n = self.get_neighboors_count_raster(raster, k, no_data)

        raster[no_data_mask] = 0
        encasement_raw = (signal.convolve(raster, convolution_kernel, mode='same')/k.sum())
        encasement = ((encasement_raw*k.sum())-((k.sum()-n) * raster))/n
        encasement[n < 3] = 0
        encasement[no_data_mask] = output_no_data

        return encasement

    def processAlgorithm(self, parameters, context, feedback):
        """
        Called when a processing algorithm is run
        """
        # Load input raster and its metadata
        input_raster: QgsRasterLayer = self.parameterAsRasterLayer(parameters, 'INPUT', context)
        no_data, geotransform = self.get_raster_metadata(parameters, context, input_raster)

        # Load other params
        input_window = self.parameterAsInt(parameters, 'INPUT_WINDOW', context)

        out_srs: SpatialReference = self.get_spatial_reference(input_raster)

        # Path of the layer is given. If a temporary layer is selected, layer is created in qgis temp dir
        out_path = self.parameterAsOutputLayer(parameters, 'OUTPUT', context)

        np_input = LandsklimUtils.raster_to_array(input_raster)
        np_output = np.copy(np_input)

        # Compute windowed average
        output_no_data = -9999
        np_output = self.compute_encasement(np_output, input_window, no_data, output_no_data)
        # Erase every computed data by no data where necessary
        np_output[np_input == no_data] = output_no_data

        self.write_raster(out_path, np_output, out_srs, geotransform, output_no_data)

        return {'OUTPUT': out_path}
