import os
import sys
import time
from collections import defaultdict
from typing import List, Tuple, Dict, Optional
import numpy as np
from PyQt5.QtCore import QVariant

from PyQt5.QtWidgets import QLabel, QApplication, QWidget, QTabWidget, QCheckBox
from pykrige import OrdinaryKriging

from qgis.PyQt.QtWidgets import QDialog, QVBoxLayout, QGridLayout, QLayout, QHBoxLayout, \
    QRadioButton, QDoubleSpinBox, QSpacerItem, QSizePolicy, QGroupBox, QComboBox
from qgis._core import QgsVectorLayer, QgsProcessing, QgsProject, QgsField, QgsGraduatedSymbolRenderer, \
    QgsApplication, QgsStyle, QgsColorRamp
from qgis import processing
from qgis.core import edit

from landsklim.lk.landsklim_constants import DATASET_COLUMN_X, DATASET_COLUMN_Y
from landsklim.lk.regression_model import MultipleRegressionModel
from landsklim.lk.phase_kriging import PhaseKriging
from landsklim.lk.phase_multiple_regression import PhaseMultipleRegression
from landsklim.lk.phase_composite import PhaseComposite
from landsklim.ui.charts import LandsklimChart

from landsklim.lk.landsklim_analysis import LandsklimAnalysis
from landsklim.lk.phase import IPhase


class WidgetLocalMultipleRegressionCorrelationView(QWidget):
    def __init__(self, local_phase: PhaseMultipleRegression, parent=None):
        super(WidgetLocalMultipleRegressionCorrelationView, self).__init__(parent=parent)
        self.__local_phase: PhaseMultipleRegression = local_phase
        self.regressor_view = None
        self.setLayout(self.create_widget())

    def clear(self):
        self.regressor_view.destroy_viz()

    def coef_by_family(self, tab_widget: QTabWidget):
        current_family = []
        for regressor in self.__local_phase.get_regressors_name():  # type: str
            if len(current_family) > 0 and '_'.join(current_family[0].split('_')[:-1]) != '_'.join(regressor.split('_')[:-1]):
                tab_widget.addTab(WidgetMultipleRegressionCoefficientsView(self.__local_phase, current_family), '_'.join(current_family[0].split('_')[:-1]))
                current_family = []
            current_family.append(regressor)
        if len(current_family) > 0:
            tab_widget.addTab(WidgetMultipleRegressionCoefficientsView(self.__local_phase, current_family), '_'.join(current_family[0].split('_')[:-1]))

    def create_widget(self) -> QLayout:
        main_layout: QVBoxLayout = QVBoxLayout()
        tab_widget: QTabWidget = QTabWidget()
        self.regressor_view = WidgetMultipleRegressionRegressorsView(self.__local_phase)
        tab_widget.addTab(self.regressor_view, self.tr("Regressors"))
        # self.coef_by_family(tab_widget)
        main_layout.addWidget(tab_widget)
        return main_layout


class WidgetGlobalMultipleRegressionCorrelationView(QWidget):
    def __init__(self, global_phase: PhaseMultipleRegression, parent=None):
        super(WidgetGlobalMultipleRegressionCorrelationView, self).__init__(parent=parent)
        self.__global_phase: PhaseMultipleRegression = global_phase
        self.__variables_correlations: Dict[str, Dict[int, float]] = self.__global_phase.get_variables_correlation()
        self.__variables_labels: Dict[str, Dict[int, str]] = self.__generate_labels()
        self.__chart_variables = LandsklimChart(fig_height=4)
        self.__checkboxes: List[QCheckBox] = []
        layout: QLayout = QHBoxLayout()
        checkbox_layout: QLayout = QVBoxLayout()
        layout.addLayout(checkbox_layout)
        layout.addWidget(self.__chart_variables.get_canvas())
        self.setLayout(layout)
        self.init(checkbox_layout)

    def __generate_labels(self) -> Dict[str, Dict[int, str]]:
        labels: Dict[str, Dict[int, str]] = {}
        for variable, corr_dict in self.__variables_correlations.items():
            labels[variable] = {}
            for window, coef in corr_dict.items():
                labels[variable][window] = "{0} ({1}) R2 = {2:.2f}".format(variable, window, coef)
        return labels

    def init(self, checkbox_layout: QLayout):
        for regressor in self.__variables_correlations:
            checkbox_regressor: QCheckBox = QCheckBox(regressor)
            checkbox_regressor.setChecked(True)
            checkbox_regressor.clicked.connect(self.refresh_chart)
            self.__checkboxes.append(checkbox_regressor)
            checkbox_layout.addWidget(checkbox_regressor)
        self.refresh_chart()

    def refresh_chart(self):
        variables_correlations = self.__variables_correlations.copy()
        variables_labels = self.__variables_labels.copy()
        for checkbox, key in zip(self.__checkboxes, self.__variables_correlations.keys()):
            if not checkbox.isChecked():
                variables_correlations.pop(key)
                variables_labels.pop(key)
        self.__chart_variables.plot_lines(variables_correlations, self.tr("Window size"), "r", self.tr("Correlation between each variable and the response variable"), labels=variables_labels)

class WidgetMultipleRegressionCoefficientsView(QWidget):
    def __init__(self, local_phase: PhaseMultipleRegression, regressors_name: List[str], parent=None):
        super(WidgetMultipleRegressionCoefficientsView, self).__init__(parent=parent)
        self.__local_phase: PhaseMultipleRegression = local_phase
        self.__regressors_name: List[str] = regressors_name
        self.__chart_coefficients_polygons: LandsklimChart = LandsklimChart(fig_height=2)
        self.__chart_coefficients_repartition: LandsklimChart = LandsklimChart()
        self.__layout: QLayout = QVBoxLayout()
        self.__layout.addWidget(self.__chart_coefficients_polygons.get_canvas())
        self.__layout.addWidget(self.__chart_coefficients_repartition.get_canvas())
        self.setLayout(self.__layout)
        self.init()

    def init(self):
        # Coefficients are a numpy array, containing the list of coefficients for each polygon for each regressors
        # regressor not included in an analysis is represented as a NaN
        coefficients: np.ndarray = self.__local_phase.get_coefficients_array()
        regressor_names: List[str] = self.__local_phase.get_regressors_name()
        models_count: int = self.__local_phase.get_models_count()
        if len(self.__regressors_name) == 0:
            self.__chart_coefficients_repartition.plot_scatter(np.repeat(np.linspace(0, models_count-1, models_count).reshape(1, -1), len(regressor_names), axis=0), coefficients.T, regressor_names, "Polygons", "Coefficient", "", plot_fit=False)
        else:
            coefficients_indices = [regressor_names.index(regressor_name) for regressor_name in self.__regressors_name]
            coefficients_regressor: np.ndarray = coefficients[:, coefficients_indices]
            self.__chart_coefficients_repartition.plot_scatter(np.repeat(np.linspace(0, models_count-1, models_count).reshape(1, -1), len(self.__regressors_name), axis=0), coefficients_regressor.T, self.__regressors_name, "Polygons", "Coefficient", "", plot_fit=False)

        """chart_data = {}
        for i, regressor in enumerate(self.__local_phase.get_regressors_name()):
            chart_data[regressor] = {key: value for key, value in zip(np.linspace(0, self.__local_phase.get_models_count()-1, self.__local_phase.get_models_count()), coefficients[:, i])}"""



class WidgetMultipleRegressionRegressorsView(QWidget):

    def __init__(self, local_phase: PhaseMultipleRegression, parent=None):
        super(WidgetMultipleRegressionRegressorsView, self).__init__(parent=parent)
        self.__local_phase: PhaseMultipleRegression = local_phase
        self.__chart_view = LandsklimChart(fig_height=4)

        self.__layer_viz: Optional[QgsVectorLayer] = None
        self.__attribute_table: Dict[str, np.ndarray] = {}
        self.__layout: QLayout = QVBoxLayout()
        self.__layout_controls: QLayout = QHBoxLayout()
        self.__radio_distribution: QRadioButton = QRadioButton()
        self.__radio_distribution.setText(self.tr("Distribution"))
        self.__radio_distribution.setChecked(True)
        self.__radio_distribution.clicked.connect(self.on_radio_click)
        self.__radio_frequency: QRadioButton = QRadioButton()
        self.__radio_frequency.setText(self.tr("Frequency"))
        self.__radio_frequency.clicked.connect(self.on_radio_click)
        self.__label_threshold: QLabel = QLabel()
        self.__label_threshold.setText(self.tr("R min"))
        self.__sp_threshold: QDoubleSpinBox = QDoubleSpinBox()
        self.__sp_threshold.setMaximum(1)
        self.__sp_threshold.setMinimum(0)
        self.__sp_threshold.setSingleStep(0.1)
        self.__sp_threshold.setValue(0.3)
        self.__sp_threshold.valueChanged.connect(self.on_spinbox_edit)

        self.__group_box: QGroupBox = QGroupBox()
        self.__group_box_layout: QHBoxLayout = QHBoxLayout()
        self.__radio_viz_r: QRadioButton = QRadioButton()
        self.__radio_viz_r.setText("r")
        self.__radio_viz_r.clicked.connect(self.on_radio_viz_click)
        self.__radio_viz_coef: QRadioButton = QRadioButton()
        self.__radio_viz_coef.setText("Coef")
        self.__radio_viz_coef.clicked.connect(self.on_radio_viz_click)
        self.__panel_variables = QComboBox()
        for regressor_name in self.__local_phase.get_regressors_name():  # str
            self.__panel_variables.addItem(regressor_name, regressor_name)
        self.__panel_variables.currentIndexChanged.connect(self.on_selection_changed)
        self.__group_box_layout.addWidget(self.__radio_viz_r)
        self.__group_box_layout.addWidget(self.__radio_viz_coef)
        self.__group_box_layout.addWidget(self.__panel_variables)
        self.__group_box.setLayout(self.__group_box_layout)
        self.__group_box.setTitle(self.tr("Visualization"))

        self.__layout_controls.addWidget(self.__radio_distribution)
        self.__layout_controls.addWidget(self.__radio_frequency)
        self.__layout_controls.addWidget(self.__label_threshold)
        self.__layout_controls.addWidget(self.__sp_threshold)
        self.__layout_controls.addItem(QSpacerItem(0, 0, QSizePolicy.Expanding, QSizePolicy.Expanding))
        self.__layout_controls.addWidget(self.__group_box)
        self.__layout.addItem(self.__layout_controls)
        self.__layout.addWidget(self.__chart_view.get_canvas())
        self.setLayout(self.__layout)
        self.update()

    def destroy_viz(self):
        if self.__layer_viz is not None:
            QgsProject.instance().removeMapLayer(self.__layer_viz)

    def on_spinbox_edit(self, i=0):
        self.update()

    def on_radio_click(self):
        self.update()

    def on_radio_viz_click(self):
        self.update_viz()

    def on_selection_changed(self):
        self.update_viz()

    def __polygons_as_vector_layer(self) -> QgsVectorLayer:
        params = {'INPUT': self.__local_phase.get_polygons_path(),  # self.__local_phase.get_polygons_layer().qgis_layer().source(),
                  'FIELD': 'POLYGON',
                  'BAND': '1',
                  'EIGHT_CONNECTEDNESS': True,
                  'OUTPUT': QgsProcessing.TEMPORARY_OUTPUT}
        result = processing.run('gdal:polygonize', params)
        return QgsVectorLayer(result['OUTPUT'], "gdal_polygonize")

    def __initialize_viz_attribute_table(self, no_data: int, polygons: int):
        self.__attribute_table = {}
        for regressor_name in self.__local_phase.get_regressors_name():  # type: str
            self.__attribute_table["{0}_r".format(regressor_name)] = np.full((polygons), no_data, dtype=float)
            self.__attribute_table["{0}_coef".format(regressor_name)] = np.full((polygons), no_data, dtype=float)
        for i, polygon_model in enumerate(self.__local_phase.get_model()):  # type: MultipleRegressionModel
            for pc_regressor, pc_coef in polygon_model.get_pearson_correlations().items():  # type: str, Tuple[float, float]
                self.__attribute_table["{0}_r".format(pc_regressor)][i] = pc_coef[0]
            for label, coefficient in polygon_model.get_coefficients().items():
                self.__attribute_table["{0}_coef".format(label)][i] = coefficient

    def __create_viz_layer(self):
        no_data = -2
        polygons: int = len(self.__local_phase.get_model())
        self.__initialize_viz_attribute_table(no_data, polygons)

        polygonized_vector: QgsVectorLayer = self.__polygons_as_vector_layer()
        data_provider = polygonized_vector.dataProvider()
        polygon_column: int = data_provider.fieldNameIndex('POLYGON')
        polygons_order = [f.attributes()[polygon_column] for f in polygonized_vector.getFeatures()]

        tally = defaultdict(list)
        for i, item in enumerate(polygons_order):
            tally[item].append(i)

        # Add attributes for each regressors
        with edit(polygonized_vector):
            new_attributes = [QgsField(field_name, QVariant.Double) for field_name, _ in self.__attribute_table.items()]
            data_provider.addAttributes(new_attributes)
        polygonized_vector.updateFields()

        # Add data of each regressor for each polygons
        with edit(polygonized_vector):
            for field_name, field_values in self.__attribute_table.items():
                field_index: int = data_provider.fieldNameIndex(field_name)
                for i, value in enumerate(field_values):
                    if value != no_data:
                        for polygon_indices in tally[i]:  # A polygon can be a multi-polygon, so gdal:polygonize created multiple vector polygon for a Landsklim 'polygon'
                            polygonized_vector.changeAttributeValue(polygon_indices+1, field_index, float(value))
        self.__layer_viz = polygonized_vector

    def update_viz(self):
        if self.__layer_viz is None:
            self.__create_viz_layer()
            QgsProject.instance().addMapLayer(self.__layer_viz)
        selected_regressor: str = self.__panel_variables.currentData()
        target_field = "{0}_{1}".format(selected_regressor, "r" if self.__radio_viz_r.isChecked() else "coef")
        field_values = self.__attribute_table[target_field]
        field_values = field_values[field_values != -2]
        min_value = -1 if self.__radio_viz_r.isChecked() else -np.abs(field_values).max() if len(field_values) > 0 else 0
        max_value = 1 if self.__radio_viz_r.isChecked() else np.abs(field_values).max() if len(field_values) > 0 else 0
        renderer = self.__make_renderer(target_field, min_value, max_value)
        self.__layer_viz.setRenderer(renderer)
        self.__layer_viz.triggerRepaint()

    def __make_renderer(self, target_field: str, min_value: float, max_value: float):
        renderer = QgsGraduatedSymbolRenderer()
        classif_method = QgsApplication.classificationMethodRegistry().method("EqualInterval")
        renderer.setClassificationMethod(classif_method)
        renderer.setClassAttribute(target_field)
        color_ramp: QgsColorRamp = QgsStyle().defaultStyle().colorRamp('PuOr')
        color_ramp.invert()
        renderer.updateColorRamp(color_ramp)
        classes_count = 20
        renderer.updateClasses(self.__layer_viz, classes_count)
        step = (max_value - min_value) / classes_count
        for i in range(classes_count):
            min_range, max_range = min_value + i * step, min_value + (i + 1) * step
            renderer.updateRangeLowerValue(i, min_range)
            renderer.updateRangeUpperValue(i, max_range)
        return renderer

    def update(self):
        if self.__radio_distribution.isChecked():
            self.__sp_threshold.setEnabled(False)
            self.chart_distribution()
        else:
            self.__sp_threshold.setEnabled(True)
            self.chart_frequency()

    def chart_distribution(self):
        regressors_names: List[str] = self.__local_phase.get_regressors_name()
        regressors_r = {}
        for regressor_name in regressors_names:
            regressors_r[regressor_name] = []
        for model in self.__local_phase.get_model():  # type: MultipleRegressionModel
            for pc_regressor, pc_coef in model.get_pearson_correlations().items():  # type: str, Tuple[float, float]
                regressors_r[pc_regressor].append(pc_coef[0])

        data = np.array([value for key, value in regressors_r.items()]).T

        self.__chart_view.plot_boxplot(data, x_ticks=list(regressors_r.keys()), y_lim=(-1, 1), x_label='', y_label='r', title='')

    def chart_frequency(self):
        r_threshold = self.__sp_threshold.value()
        polygons: int = len(self.__local_phase.get_model())
        pearson_correlations: List[Dict[str, Tuple[float, float]]] = []
        for model in self.__local_phase.get_model():  # type: MultipleRegressionModel
            pearson_correlations.append(model.get_pearson_correlations())
        regressors_names: List[str] = self.__local_phase.get_regressors_name()
        regressors_positive: Dict[str, int] = {}
        regressors_negative: Dict[str, int] = {}
        for regressor_name in regressors_names:  # type: str
            regressor_pearson_correlations: List[float] = [m[regressor_name][0] for m in pearson_correlations]
            regressors_positive[regressor_name] = sum([1 for pc in regressor_pearson_correlations if pc >= r_threshold])
            regressors_negative[regressor_name] = sum([1 for pc in regressor_pearson_correlations if pc < 0 and abs(pc) >= r_threshold])

        positive_values: np.ndarray = np.array([(value/polygons)*100 for value in regressors_positive.values()])
        negative_values: np.ndarray = np.array([-(value/polygons)*100 for value in regressors_negative.values()])
        self.__chart_view.plot_bar(positive_values, negative_values, regressors_names, y_axis_as_percent=True, x_label="", y_label="Fréquence", title="")

class DialogPhaseView(QDialog):

    @staticmethod
    def create_phase_view(phase: IPhase) -> QWidget:
        if phase.class_name() == PhaseMultipleRegression.class_name():
            return WidgetPhaseMultipleRegressionView(phase)
        if phase.class_name() == PhaseKriging.class_name():
            return WidgetPhaseKrigingView(phase)
        if phase.class_name() == PhaseComposite.class_name():
            return WidgetPhaseCompositeView(phase)

    """def closeEvent(self, evnt):
        for i in range(self.__tab_widget.count()):
            self.__tab_widget.widget(i).clear()
        super(DialogPhaseView, self).closeEvent(evnt)"""

    def __init__(self, situation: int, analysis: LandsklimAnalysis, parent=None):
        super(DialogPhaseView, self).__init__(parent)
        self.__analysis: LandsklimAnalysis = analysis
        self.__situation_number: int = situation

        self.__tab_widget = QTabWidget()
        for phase in analysis.get_phases(self.__situation_number):  # type: IPhase
            phase_view: QWidget = self.create_phase_view(phase)
            self.__tab_widget.addTab(phase_view, phase.name())

        self.__layout: QLayout = QVBoxLayout()
        self.__layout.addWidget(self.__tab_widget)

        self.setLayout(self.__layout)

        situation_name: str = self.__analysis.get_situation_name(self.__situation_number) if self.__situation_number > -1 else "Invalid"
        self.setWindowTitle("{0} - {1} - {2}".format(self.__analysis.get_configuration_name(), self.__analysis.get_name(), situation_name))
        screen_height: int = int(QApplication.desktop().screenGeometry().height() * 0.7)
        self.resize(self.width(), screen_height)


class WidgetPhaseMultipleRegressionView(QWidget):

    def __init__(self, phase: PhaseMultipleRegression, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.__phase: PhaseMultipleRegression = phase
        self.__layout: QLayout = QVBoxLayout()
        self.__chart_correlation = LandsklimChart(fig_height=4)
        #self.__widget_variables: Optional[QWidget] = None
        self.__formula_label: Optional[QLabel] = None
        self.__checkbox_unstandardized: Optional[QCheckBox] = None
        self.init_widget()
        self.setLayout(self.__layout)

    def is_global(self) -> bool:
        return len(self.__phase.get_model()) == 1

    """def clear(self):
        if not self.is_global():
            self.__widget_variables.regressor_view.destroy_viz()"""

    def init_widget(self):
        self.fill_header()
        self.__layout.addWidget(self.plot_correlation())
        self.__layout.addWidget(QLabel(self.tr("Estimates and residuals where obtained through cross-validation.\nEach point was excluded from the regression model where its estimation was computed. (except the autocorrelation of residuals)")))
        """if self.is_global():
            self.__widget_variables = self.plot_variables_correlation_global()
        else:
            self.__widget_variables = self.plot_variables_correlation_local()
        self.__layout.addWidget(self.__widget_variables)"""

    def formula_is_unstandardized(self) -> bool:
        return self.__checkbox_unstandardized.isChecked()

    def refresh_formula(self):
        self.__formula_label.setText(self.__phase.get_formula(unstandardized=self.formula_is_unstandardized()))

    def fill_header(self):
        grid: QGridLayout = QGridLayout()
        grid.setColumnStretch(0, 1)
        grid.setColumnStretch(1, 3)
        grid.setColumnStretch(2, 1)

        self.__checkbox_unstandardized = QCheckBox(self.tr("Unstandardized"))
        self.__checkbox_unstandardized.setChecked(False)
        self.__checkbox_unstandardized.clicked.connect(self.refresh_formula)

        formula = self.__phase.get_formula(unstandardized=self.formula_is_unstandardized())
        self.__formula_label = QLabel(formula)
        self.__formula_label.setWordWrap(True)

        grid.addWidget(QLabel(self.tr("Equation")), 0, 0)
        grid.addWidget(self.__formula_label, 0, 1)

        grid.addWidget(self.__checkbox_unstandardized, 0, 2)

        grid.addWidget(QLabel(self.tr("R² (adjusted)")), 1, 0)
        grid.addWidget(QLabel("{0:.3f} ({1:.3f})".format(self.__phase.r2(), self.__phase.get_adjusted_r2())), 1, 1)

        grid.addWidget(QLabel(self.tr("Residual standard deviation")), 2, 0)
        grid.addWidget(QLabel("{0:.3f}".format(self.__phase.get_residuals_standard_deviation())), 2, 1)

        grid.addWidget(QLabel(self.tr("RMSE")), 3, 0)
        grid.addWidget(QLabel("{0:.3f}".format(self.__phase.get_rmse())), 3, 1)

        """grid.addWidget(QLabel("Residual Standard Error"), 3, 0)
        grid.addWidget(QLabel("{0:.3f}".format(self.__phase.get_rse())), 3, 1)"""

        grid.addWidget(QLabel(self.tr("Autocorrelation of residual values")), 4, 0)
        grid.addWidget(QLabel("{0:.3f}".format(self.__phase.get_residuals_autocorrelation())), 4, 1)

        self.__layout.addLayout(grid)

    def plot_correlation(self):
        X, y = self.__phase.prepare_dataset()
        y_hat = self.__phase.predict_cv()
        rse: float = self.__phase.get_rse()
        data = []
        # TODO: Get station fid/id : ID of the station needs to be added on the dataset with the same behaviour of DATASET_COLUMN_X and DATASET_COLUMN_Y columns (filtered)
        for cx, cy, py, py_hat in zip(self.__phase.get_dataset()[DATASET_COLUMN_X].values, self.__phase.get_dataset()[DATASET_COLUMN_Y].values, y, y_hat):
            data.append(self.tr("[{0}, {1}]\nActual value : {2:.2f}\nModel prediction : {3:.2f}").format(cx, cy, py, py_hat))
        return self.__chart_correlation.plot_scatter(y, y_hat, self.tr('Stations'), self.tr('Actual value'), self.tr('Estimation'), self.tr("Model predictions and actual values"), rse_interval=rse, data=data)

    def plot_variables_correlation_local(self) -> QWidget:
        widget: WidgetLocalMultipleRegressionCorrelationView = WidgetLocalMultipleRegressionCorrelationView(self.__phase)
        return widget

    def plot_variables_correlation_global(self) -> QWidget:
        """
        Plot each variable
        """
        widget: WidgetGlobalMultipleRegressionCorrelationView = WidgetGlobalMultipleRegressionCorrelationView(self.__phase)
        return widget


class WidgetPhaseKrigingView(QWidget):

    def __init__(self, phase: PhaseKriging, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.__phase: PhaseKriging = phase
        self.__chart_variogram = LandsklimChart()
        self.__layout: QLayout = QVBoxLayout()
        self.init_widget()
        self.setLayout(self.__layout)

    def clear(self):
        # No QGIS layer can be displayed by this phase, nothing to clear
        pass

    def init_widget(self):
        self.fill_header()
        kriging_model: OrdinaryKriging = self.__phase.get_model()
        lags: np.ndarray = kriging_model.lags
        semivariance: np.ndarray = kriging_model.semivariance
        function: np.ndarray = kriging_model.variogram_function(kriging_model.variogram_model_parameters, kriging_model.lags)
        self.__layout.addWidget(self.__chart_variogram.plot_variogram(lags, semivariance, function))
        self.__layout.addWidget(QLabel(self.tr("Estimates and residuals where obtained through cross-validation.\nEach point was excluded from the regression model where its estimation was computed. (except the autocorrelation of residuals)")))

    def fill_header(self):
        grid: QGridLayout = QGridLayout()
        grid.setColumnStretch(0, 1)
        grid.setColumnStretch(1, 3)

        grid.addWidget(QLabel(self.tr("R² (adjusted)")), 0, 0)
        grid.addWidget(QLabel("{0:.3f} ({1:.3f})".format(self.__phase.r2(), self.__phase.get_adjusted_r2())), 0, 1)

        self.__layout.addLayout(grid)


class WidgetPhaseCompositeView(QWidget):

    def __init__(self, phase: PhaseComposite, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.__phase: PhaseComposite = phase
        self.__layout: QLayout = QVBoxLayout()
        self.__chart_correlation = LandsklimChart()
        self.init_widget()
        self.setLayout(self.__layout)

    def clear(self):
        # No QGIS layer can be displayed by this phase, nothing to clear
        pass

    def init_widget(self):
        self.fill_header()
        self.__layout.addWidget(self.plot_correlation())
        self.__layout.addWidget(QLabel(self.tr("Estimates and residuals where obtained through cross-validation.\nEach point was excluded from the regression model where its estimation was computed. (except the autocorrelation of residuals)")))

    def fill_header(self):
        grid: QGridLayout = QGridLayout()
        grid.setColumnStretch(0, 1)
        grid.setColumnStretch(1, 3)

        grid.addWidget(QLabel(self.tr("R² (adjusted)")), 0, 0)
        grid.addWidget(QLabel("{0:.3f} ({1:.3f})".format(self.__phase.r2(), self.__phase.get_adjusted_r2())), 0, 1)

        grid.addWidget(QLabel(self.tr("Autocorrelation of residual values")), 1, 0)
        grid.addWidget(QLabel("{0:.3f}".format(self.__phase.get_residuals_autocorrelation())), 1, 1)

        self.__layout.addLayout(grid)

    def plot_correlation(self):
        X, y = self.__phase.prepare_dataset()
        y_hat = self.__phase.predict_cv()
        data = []
        rse: float = self.__phase.get_rse()
        # TODO: Get station fid/id : ID of the station needs to be added on the dataset with the same behaviour of DATASET_COLUMN_X and DATASET_COLUMN_Y columns (filtered)
        for cx, cy, py, py_hat in zip(self.__phase.get_dataset()[DATASET_COLUMN_X].values, self.__phase.get_dataset()[DATASET_COLUMN_Y].values, y, y_hat):
            data.append(self.tr("[{0}, {1}]\nActual value : {2:.2f}\nModel prediction : {3:.2f}").format(cx, cy, py, py_hat))

        return self.__chart_correlation.plot_scatter(y, y_hat, self.tr('Stations'), self.tr('Actual value'), self.tr('Estimation'), self.tr("Model predictions and actual values"), rse_interval=rse, data=data)
