import datetime
import os
import sys
from typing import List, Tuple, Dict, Optional, Type

from PyQt5.QtCore import Qt, QSize
from PyQt5.QtWidgets import QListWidgetItem, QWidget, QSizePolicy
from qgis.PyQt import uic
from qgis.PyQt import QtWidgets, QtCore
from qgis._core import QgsField, QgsApplication
from qgis.core import QgsMapLayerProxyModel, QgsRasterLayer, QgsVectorLayer

from landsklim.lk.landsklim_interpolation import LandsklimInterpolation
from landsklim.lk.utils import LandsklimUtils
from landsklim.lk.landsklim_analysis import LandsklimAnalysisMode, LandsklimAnalysis
from landsklim.lk.phase import IPhase
from landsklim.lk.regressor import Regressor
from landsklim.lk.regressor_factory import RegressorDefinition
from landsklim.ui.utils import LandsklimTableModelRegressors, LandsklimTableColumnType, \
    QListWidgetItemDataComparable, LandsklimTableSituationsDate
from landsklim.ui.landsklim_dialog import LandsklimDialog
from landsklim import landsklim
from landsklim.lk.map_layer import MapLayerCollection


# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'view_save_netcdf.ui'))


class ViewSaveNetCdf(LandsklimDialog, FORM_CLASS):
    """
    Represents the save netcdf file dialog

    :ivar LandsklimAnalysis analysis: Analysis of the exported interpolation

    :ivar LandsklimInterpolation interpolation: Exported interpolation
    """
    def __init__(self, analysis: LandsklimAnalysis, interpolation: LandsklimInterpolation, parent=None):
        """Constructor."""
        self.__analysis: LandsklimAnalysis = analysis
        self.__interpolation: LandsklimInterpolation = interpolation
        super(ViewSaveNetCdf, self).__init__(parent)
        self.setModal(True)
        self.units = {
            self.tr("degC (Celsius)"): "degC",
            self.tr("degF (Fahrenheit)"): "degF",
            self.tr("K (Kelvin)"): "K",
            self.tr("R (Rankine)"): "R",
            self.tr("mm"): "mm",
            self.tr("cm"): "cm"
        }
        self.init_combo_unit()

    def init_ui(self):
        """
        Init UI components
        """
        self.label_help_standard_name.setPixmap(QgsApplication.getThemeIcon("mActionContextHelp.png").pixmap(QSize(16, 16)))
        self.label_help_standard_name.setToolTip(self.tr("standard_name attribute of the NetCDF file must be taken from the standard name table"))

        self.file_widget_save.setFilter(self.tr("NetCDF (*.nc)"))

        self.edit_long_name.setText("Air temperature")
        self.edit_standard_name.setText("air_temperature")

        date_list = [datetime.datetime(2000, 7, 1, 0, 0, 0) + datetime.timedelta(days=x) for x in range(len(self.__analysis.get_situations_names()))]

        self.__table_model = LandsklimTableSituationsDate(situations=self.__analysis.get_situations_names(), dates=date_list)
        self.table_situations.setModel(self.__table_model)

        self.label_icon_info_crs.setPixmap(QgsApplication.getThemeIcon("mIconInfo.svg").pixmap(QSize(16, 16)))
        self.label_info_crs.setText(self.tr("CRS of exported variables : EPSG:4326 WGS 84"))

    def init_combo_unit(self):
        for text, data in self.units.items():
            self.combo_unit.addItem(text, data)


    def refresh_ui(self):
        """
        Refresh UI components after a specific action was triggered
        """
        pass

    def get_long_variable_name(self) -> str:
        return self.edit_long_name.text()

    def get_standard_variable_name(self) -> str:
        return self.edit_standard_name.text()

    def get_unit(self) -> str:
        current_text = self.combo_unit.currentText()
        if current_text not in self.units:
            data = current_text
        else:
            data = self.units[current_text]
        return data

    def get_dates(self) -> Dict[int, datetime.datetime]:
        return {k:v for (k,v) in zip(self.__analysis.get_station_situations(), self.__table_model.get_dates())}

    def get_file_path(self) -> str:
        return self.file_widget_save.filePath()

    def input_are_valid(self) -> Tuple[bool, List[str]]:
        """
        Check if user inputs are valid
        """
        errors = []
        is_ok = True

        if self.get_long_variable_name() == "":
            is_ok = False
            errors.append("ERROR_LONG_VARIABLE_NAME")
        if self.get_standard_variable_name() == "":
            is_ok = False
            errors.append("ERROR_STANDARD_VARIABLE_NAME")
        if self.get_file_path() == "":
            is_ok = False
            errors.append("ERROR_FILE_PATH")
        if self.get_unit() == "":
            is_ok = False
            errors.append("ERROR_UNIT")
        if len(self.get_dates()) != len(self.__analysis.get_station_situations()):
            is_ok = False
            errors.append("ERROR_DATES")
        return is_ok, errors

