import gc
import os.path
from functools import partial
from multiprocessing import Pool
import multiprocessing as mp
from typing import Union, Tuple, Callable
import ctypes

from osgeo import gdal
from osgeo import osr
from osgeo.osr import SpatialReference
from qgis.PyQt.QtCore import QCoreApplication
from qgis._core import QgsProcessingParameterRasterLayer
from qgis.core import QgsProcessing, QgsProcessingAlgorithm, QgsProcessingException, QgsRasterLayer
from qgis import processing
import numpy as np
import math
from scipy.signal import convolve2d

from landsklim.processing.landsklim_processing_regressor_algorithm import LandsklimProcessingRegressorAlgorithm
from landsklim.lk.utils import LandsklimUtils
from landsklim.lk import environment


class CosineProcessingAlgorithm(LandsklimProcessingRegressorAlgorithm):
    """
    Processing algorithm computing cosine from a DEM (from its orientation regressor)
    """

    INPUT_ORIENTATION = 'INPUT_ORIENTATION'

    def __init__(self):
        super().__init__({'OUTPUT': 'Cosine raster'})

    def createInstance(self):
        return CosineProcessingAlgorithm()

    def name(self) -> str:
        """
        Unique name of the algorithm
        """
        return 'cosine'

    def displayName(self) -> str:
        """
        Displayed name of the algorithm
        """
        return self.tr('Cosine')

    def shortHelpString(self) -> str:
        return self.tr('Compute windowed cosine from DEM')

    def add_dependencies(self):
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                CosineProcessingAlgorithm.INPUT_ORIENTATION,
                self.tr('Orientation', "LandsklimProcessingRegressorAlgorithm")
            )
        )

    def compute_cosine(self, orientations: np.array) -> np.array:
        """
        Compute cosine of a raster

        :param orientations: Orientations raster. No data is already represented as np.nan
        :type orientations: np.ndarray

        :returns: Cosine values for each cell of the raster
        :rtype: np.ndarray
        """
        return np.cos(np.deg2rad(orientations))

    def processAlgorithm(self, parameters, context, feedback):
        """
        Called when a processing algorithm is run
        """
        # Load input raster and its metadata
        input_raster: QgsRasterLayer = self.parameterAsRasterLayer(parameters, 'INPUT', context)
        input_orientation: QgsRasterLayer = self.parameterAsRasterLayer(parameters, CosineProcessingAlgorithm.INPUT_ORIENTATION, context)

        no_data, geotransform = self.get_raster_metadata(parameters, context, input_raster)
        input_window = self.parameterAsInt(parameters, 'INPUT_WINDOW', context)
        additional_variables_folder = self.parameterAsString(parameters, 'ADDITIONAL_VARIABLES_FOLDER', context).strip()
        out_srs: SpatialReference = self.get_spatial_reference(input_raster)

        # Path of the layer is given. If a temporary layer is selected, layer is created in qgis temp dir
        out_cosine = self.parameterAsOutputLayer(parameters, 'OUTPUT', context)

        np_orientations = LandsklimUtils.raster_to_array(input_orientation)

        """regressor_orientation = RegressorFactory.get_regressor(RegressorOrientation.class_name(), input_window, 1)
        orientation_raster_path = regressor_orientation.get_path()
        if not os.path.exists(orientation_raster_path):
            raise RuntimeError("Orientation raster is not found")
        np_orientations = LandsklimUtils.source_to_array(orientation_raster_path)"""

        output_no_data = -9999
        np_orientations[np_orientations == output_no_data] = np.nan  # no_data is the same for each regressor

        np_cosine = self.compute_cosine(np_orientations)
        np_cosine[np.isnan(np_cosine)] = output_no_data

        self.write_raster(out_cosine, np_cosine, out_srs, geotransform, output_no_data)

        return {'OUTPUT': out_cosine}
