import abc
import time
from abc import abstractmethod
from typing import List, Tuple, Optional, Dict
import os

from PyQt5.QtCore import QCoreApplication
from qgis.core import QgsRasterLayer, QgsProject

from landsklim.lk.cache import qgis_project_cache
from landsklim.lk.map_layer import RasterLayer


class Regressor(metaclass=abc.ABCMeta):
    r"""
    Represents a regressor

    :param windows: List of available windows for this regressor
    :type windows: List[int]

    :param polynomial_degree: Compute polynomial degree of this regressor
    :type polynomial_degree: int

    :param intermediate_layers: List of intermediate layers prefix created to compute regressor
    :type intermediate_layers: List[str]
    """

    def __init__(self, windows: int, polynomial_degree: int, intermediate_layers: List[str] = None):
        self._windows: int = windows
        self._polynomial_degree: int = polynomial_degree
        self._intermediate_layers: List[str] = [] if intermediate_layers is None else intermediate_layers
        self._raster: Optional[RasterLayer] = None

    @staticmethod
    def tr(string) -> str:
        return QCoreApplication.translate('Regressor', string)

    @staticmethod
    @abstractmethod
    def class_name() -> str:
        raise NotImplementedError()

    @staticmethod
    @abstractmethod
    def min_window() -> int:
        """
        :returns: Minimal possible window for this regressor
        :rtype: int
        """
        raise NotImplementedError()

    def get_windows(self) -> int:
        """
        :returns: Regressor window
        :rtype: int
        """
        return self._windows

    def get_polynomial_degree(self) -> int:
        """
        :returns: Polynomial degree
        :rtype: int
        """
        return self._polynomial_degree

    def set_raster_layer(self, raster: RasterLayer):
        self._raster = raster

    def get_raster_layer(self):
        return self._raster

    def get_intermediate_layers(self) -> List[str]:
        """
        :returns: List of intermediate layers prefixes
        :rtype: List[str]
        """
        return self._intermediate_layers

    def layer_name(self) -> str:
        """
        Name of the generated layer
        """
        return "{0}_{1}".format(self.prefix(), self._windows)

    @abc.abstractmethod
    def prefix(self) -> str:
        """
        Regressor prefix
        """
        raise NotImplementedError

    @abc.abstractmethod
    def name(self) -> str:
        """
        Regressor name
        """
        raise NotImplementedError

    @abc.abstractmethod
    def compute(self, source_raster: QgsRasterLayer):
        """
        :param source_raster: Data used to compute predictive variable
        :type source_raster: QgsRasterLayer
        """
        raise NotImplementedError

    def specific_source_raster(self) -> Optional[RasterLayer]:
        """
        Give a specific raster layer to apply the processing algorithm on.
        If None, processing algorithm is applied on project' DEM.
        """
        return None

    def project_path(self) -> str:
        return os.path.join(qgis_project_cache().homePath(), 'Landsklim')

    def get_path(self) -> str:
        project_home_path: str = qgis_project_cache().homePath()  # QgsProject().instance().homePath()
        return os.path.join(project_home_path, 'Landsklim', 'regressors', 'source',
                            '{0}_{1}.tif'.format(self.prefix(), self._windows))

    def _create_path(self) -> Tuple[str, bool]:
        """
        Initialize and eventually create directories for the raster.

        :returns: Tuple containing the raster path and a boolean (True if the raster already exists, otherwise False)
        :rtype: Tuple[str, bool]
        """
        raster_path = self.get_path()
        if not os.path.exists(os.path.dirname(raster_path)):
            os.makedirs(os.path.dirname(raster_path))
        return raster_path, os.path.exists(raster_path)

    def equals(self, other):
        """
        Compare if two regressors responds to the same definition
        """
        if not isinstance(other, Regressor):
            return NotImplemented

        return (self.prefix() == other.prefix() and self.class_name() == other.class_name() and self.get_windows() == other.get_windows()
                and self._polynomial_degree == other._polynomial_degree)

    def __str__(self):
        return "<{0}> {1} ({2}, {3})".format(Regressor.__name__, self.class_name(), self.get_windows(), self.get_polynomial_degree())

    def to_json(self) -> Dict:
        state_dict: Dict = self.__dict__.copy()
        return state_dict