import os
import sys
from typing import List, Tuple
import numpy as np
import pandas as pd
from PyQt5.QtCore import Qt, QRect
from PyQt5.QtGui import QPalette, QColor

from PyQt5.QtWidgets import QLabel, QApplication, QWidget, QScrollArea, QFormLayout, QSizePolicy

from qgis.PyQt.QtWidgets import QListWidgetItem, QDialog, QPushButton, QVBoxLayout, QGridLayout, QLayout


from landsklim.lk.landsklim_analysis import LandsklimAnalysis


class WidgetVariablesCorrelation(QWidget):
    """
    Widget showing regressors correlation table for an analysis
    """

    def __init__(self, analysis: LandsklimAnalysis, parent=None):
        super(WidgetVariablesCorrelation, self).__init__(parent)
        self.__analysis: LandsklimAnalysis = analysis

        self.setMinimumWidth(1000)

        layout = QVBoxLayout()
        self.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)

        layout.addWidget(QLabel(self.tr("Correlation table")))
        scroll_area = QScrollArea(self)
        scroll_area.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        scroll_area.setWidgetResizable(True)
        scroll_area_widget = QWidget()
        vertical_layout = QVBoxLayout(scroll_area_widget)
        vertical_layout.addLayout(self.fill_table())
        scroll_area.setWidget(scroll_area_widget)
        layout.addWidget(scroll_area)
        self.setLayout(layout)

    def get_correlation_gradient_color(self, ratio: float) -> str:
        """
        Get background-color stylesheet instruction from correlation value
        """
        start = (119, 221, 119)
        end = (255, 105, 97)
        red = int(ratio * start[0] + (1 - ratio) * end[0]) if not np.isnan(ratio) else 0
        green = int(ratio * start[1] + (1 - ratio) * end[1]) if not np.isnan(ratio) else 0
        blue = int(ratio * start[2] + (1 - ratio) * end[2]) if not np.isnan(ratio) else 0
        return "background-color: rgb({0}, {1}, {2});".format(red, green, blue)

    def fill_table(self):
        """
        Fill correlation table
        """
        correlations: pd.DataFrame = self.__analysis.get_correlation_table()
        grid: QGridLayout = QGridLayout()
        grid_size = len(correlations)
        for i in range(grid_size+1):
            grid.setColumnStretch(i, 1)
            grid.setRowStretch(i, 1)
            if i < grid_size:
                label_col: QLabel = QLabel(correlations.columns[i])
                label_col.setAlignment(Qt.AlignCenter)
                label_row: QLabel = QLabel(correlations.columns[i])
                label_row.setAlignment(Qt.AlignCenter)
                grid.addWidget(label_col, i+1, 0)
                grid.addWidget(label_row, 0, i+1)

        for i in range(grid_size):
            for j in range(i, grid_size):
                background_cell_widget = QWidget()
                background_cell_widget.setContentsMargins(4, 4, 4, 4)
                background_cell_widget.setStyleSheet(self.get_correlation_gradient_color(np.abs(correlations.iat[i, j])))
                label_cor: QLabel = QLabel("{0:.3f}".format(correlations.iat[i, j]))
                label_cor.setAlignment(Qt.AlignCenter)
                label_cor.setToolTip("{0} - {1}".format(correlations.columns[i], correlations.columns[j]))
                grid.addWidget(background_cell_widget, i+1, j+1)
                grid.addWidget(label_cor, i+1, j+1)
        return grid

