from PyQt5.QtCore import QDateTime, QDate, QSize
from qgis._core import QgsUnitTypes

from landsklim.ui.landsklim_dialog import LandsklimDialog
from landsklim.lk.map_layer import MapLayerCollection, RasterLayer
from landsklim.lk.regressor_factory import RegressorFactory

"""
Crash when loading plugin : NoSuchClassError: Unknown C++ class: QgsMapLayerProxyModel
Because QgsMapLayerComboBox filters are set from UI Designer. In the .ui file, remove sections like
<property name="filters">
        <set>QgsMapLayerProxyModel::MeshLayer|QgsMapLayerProxyModel::PluginLayer|QgsMapLayerProxyModel::RasterLayer</set>
</property>
inside QgsMapLayerComboBox widgets
These filters can be set from Python like this :
from qgis.core import QgsMapLayerProxyModel
self.mMapLayer.setFilters(QgsMapLayerProxyModel.LineLayer)
(https://github.com/qgis/QGIS/issues/38472)
"""

import os
from typing import List, Tuple, Optional

from qgis.gui import QgsMapLayerComboBox
from qgis.PyQt import uic
from qgis.PyQt import QtWidgets
from qgis.core import QgsMapLayerProxyModel, QgsApplication, QgsRasterLayer, QgsVectorLayer, QgsMapLayer
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QStyle, QVBoxLayout, QHBoxLayout, QPushButton, QSizePolicy, QLabel, QLayout, QMessageBox

from landsklim.lk.landsklim_project import LandsklimProject

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'view_new_project.ui'))


class ViewNewProject(LandsklimDialog, FORM_CLASS):
    """
    Represents the new project dialog
    """
    def __init__(self, parent=None, project: LandsklimProject = None, is_update: bool = False):
        """Constructor."""
        self._project: LandsklimProject = project
        self._is_update: bool = is_update
        self.__delete_project: bool = False
        super(ViewNewProject, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect

    def delete_project_enabled(self) -> bool:
        return self.__delete_project

    def init_ui(self):
        """
        Init UI components
        """
        self.mlcb_station1.setFilters(QgsMapLayerProxyModel.PointLayer)
        self.mlcb_dem.setFilters(QgsMapLayerProxyModel.RasterLayer)
        self.mlcb_dem.layerChanged.connect(self.update_crs_message)
        self.mlcb_variable1.setFilters(QgsMapLayerProxyModel.RasterLayer)
        self.pb_add_stations.setIcon(QgsApplication.getThemeIcon("mActionAdd.svg"))
        self.pb_add_stations.clicked.connect(lambda: self.button_new_slot(self.layout_stations, QgsMapLayerProxyModel.PointLayer))
        self.pb_add_variable.setIcon(QgsApplication.getThemeIcon("mActionAdd.svg"))
        self.pb_add_variable.clicked.connect(lambda: self.button_new_slot(self.layout_variables, QgsMapLayerProxyModel.RasterLayer))

        self.label_icon_info_dem.setPixmap(QgsApplication.getThemeIcon("mIconInfo.svg").pixmap(QSize(16, 16)))
        self.label_info_dem.setText("")
        self.label_icon_warning_dem.setPixmap(QgsApplication.getThemeIcon("mIconWarning.svg").pixmap(QSize(16, 16)))

        self.update_no_data_station_status()
        self.update_crs_message()
        self.cb_stations_nodata.clicked.connect(self.update_no_data_station_status)

        """
        If a project is already defined and created (not empty), update fields value with project sources
        """
        if self._project is not None and self._is_update:
            self.__set_project_name(self._project.to_string())
            self.__set_project_stations(self._project.get_stations_sources().get_qgis_layers())
            self.__set_project_dem(self._project.get_dem().qgis_layer())
            self.__set_project_variables(self._project.get_variables().get_qgis_layers())
            self.__set_project_date(self._project.get_date())
            self.__set_project_stations_no_data(self._project.get_stations_no_data())
            self.disable_update_status()
            self.pb_delete_project.clicked.connect(self.button_delete_project)
        else:
            self.pb_delete_project.hide()

    def update_crs_message(self):
        selected_dem: QgsRasterLayer = self.get_project_dem()
        if selected_dem is not None:
            euclidean_projection: str = self.tr("Euclidean coordinates")
            geographic_projection: str = self.tr("Geographic coordinates")
            projection: str = geographic_projection if selected_dem.crs().mapUnits() == QgsUnitTypes.DistanceDegrees else euclidean_projection
            self.label_info_dem.setText(self.tr("CRS of interpolations : {0} ({1})").format(selected_dem.crs().authid(), projection))
            self.set_geographic_warning(enabled=selected_dem.crs().isGeographic())

    def set_geographic_warning(self, enabled: bool):
        self.label_icon_warning_dem.setHidden(not enabled)
        self.label_warning_dem.setHidden(not enabled)

    def update_no_data_station_status(self):
        enabled: bool = self.cb_stations_nodata.isChecked()
        self.sb_stations_nodata.setEnabled(enabled)

    def disable_update_status(self):
        self.le_name.setEnabled(False)
        """for i in range (self.layout_stations.count()):
            self.layout_stations.itemAt(i).itemAt(0).widget().setEnabled(False)  # Disable the combobox
            self.layout_stations.itemAt(i).itemAt(1).widget().setEnabled(False)  # Disable the button"""
        self.pb_add_stations.setEnabled(True)
        self.mlcb_dem.setEnabled(False)

    def button_delete_project(self) -> int:
        res = QMessageBox.warning(self, "Landsklim", self.tr("Delete the linked Landsklim project ?"), QMessageBox.Yes | QMessageBox.No)
        if res == QMessageBox.Yes:
            self.__delete_project = True
            return self.reject()

    def button_new_slot(self, parent_layout: QLayout, map_layer_proxy_model: QgsMapLayerProxyModel):
        """
        Add a new spot to register stations
        """
        index = parent_layout.count()
        new_layout: QHBoxLayout = QHBoxLayout()

        mlcb_slot: QgsMapLayerComboBox = QgsMapLayerComboBox()
        mlcb_slot.setFilters(map_layer_proxy_model)
        mlcb_slot.setShowCrs(True)

        pb_remove: QPushButton = QPushButton()
        pb_remove.setIcon(QgsApplication.getThemeIcon("mActionRemove.svg"))
        pb_remove.setSizePolicy(QSizePolicy.Fixed, QSizePolicy.Fixed)
        pb_remove.clicked.connect(lambda: self.button_remove_slot(new_layout, parent_layout))

        new_layout.addWidget(mlcb_slot)
        new_layout.addWidget(pb_remove)

        parent_layout.addLayout(new_layout)

    def button_remove_slot(self, layout: QLayout, parent_layout: QLayout):
        """
        Remove a stations shapefile from layout
        """
        parent_layout.removeItem(layout)
        self.clean_layout(layout)
        parent_layout.update()

    def refresh_ui(self):
        """
        Refresh UI components after a specific action was triggered
        """
        pass

    def __set_project_name(self, name: str):
        self.le_name.insert(name)

    def __set_project_date(self, date: Tuple[int, int, int]):
        self.date_sun.setDate(QDate(date[0], date[1], date[2]))

    def __set_project_stations_no_data(self, stations_no_data: Optional[float]):
        self.cb_stations_nodata.setChecked(stations_no_data is not None)
        if stations_no_data is not None:
            self.sb_stations_nodata.setValue(stations_no_data)
        self.update_no_data_station_status()

    def __set_project_stations(self, stations: List[QgsVectorLayer]):
        for i, station in enumerate(stations):  # type: QgsVectorLayer
            if i >= self.layout_stations.count():
                self.button_new_slot(self.layout_stations, QgsMapLayerProxyModel.PointLayer)
            map_layer = self.layout_stations.itemAt(i).itemAt(0).widget()
            map_layer.setLayer(station)

    def __set_project_dem(self, dem: QgsRasterLayer):
        self.mlcb_dem.setLayer(dem)

    def __set_project_variables(self, variables: List[QgsRasterLayer]):
        # Set first MapLayerComboBox empty if no variable defined
        if len(variables) == 0:
            variables = [None]

        for i, variable in enumerate(variables):  # type: QgsRasterLayer
            if i >= self.layout_variables.count():
                self.button_new_slot(self.layout_variables, QgsMapLayerProxyModel.RasterLayer)
            map_layer = self.layout_variables.itemAt(i).itemAt(0).widget()
            map_layer.setLayer(variable)

    def get_no_data_stations(self) -> Optional[float]:
        return self.sb_stations_nodata.value() if self.cb_stations_nodata.isChecked() else None

    def get_project_name(self) -> str:
        """
        :returns: The project name
        :rtype: str
        """
        return self.le_name.text()

    def get_layers_collection(self, parent_layout: QLayout) -> List[QgsMapLayer]:
        """
        Get a collection of QGIS layer under a multiple layers selection layout (stations or variables)

        :param parent_layout: Parent container layout of map layers
        :type parent_layout: QLayout
        """
        layers_collection = []
        for i in range(parent_layout.count()):
            item_layout: QLayout = parent_layout.itemAt(i)
            cb_map_layer: QgsMapLayerComboBox = item_layout.itemAt(0).widget()
            if isinstance(cb_map_layer, QgsMapLayerComboBox) and cb_map_layer.currentLayer() is not None:
                layers_collection.append(cb_map_layer.currentLayer())
        return layers_collection

    def get_project_stations(self) -> List[QgsVectorLayer]:
        """
        :returns: The stations shapefiles
        :rtype: List[QgsVectorLayer]
        """
        return self.get_layers_collection(self.layout_stations)

    def get_project_dem(self) -> QgsRasterLayer:
        """
        :returns: The project DEM
        :rtype: QgsRasterLayer
        """
        selected_layer: QgsMapLayer = self.mlcb_dem.currentLayer()
        return selected_layer

    def get_project_variables(self) -> List[QgsRasterLayer]:
        """
        :returns: The additional predictive variable rasters
        :rtype: List[QgsRasterLayer]
        """
        return self.get_layers_collection(self.layout_variables)

    def get_project_date(self) -> QDate:
        return self.date_sun.date()

    def check_no_duplicates(self, layers_list: List[QgsMapLayer]) -> bool:
        """
        Check that all layers in the list are not duplicated
        """
        return len(set(layers_list)) == len(layers_list)

    def check_crs_is_same(self, layers_list: List[QgsMapLayer]) -> bool:
        """
        :param layers_list: List of QgsMapLayer
        :type layers_list: QgsMapLayer

        :returns: True if all layers are on the same CRS, otherwise False
        :rtype: bool
        """
        is_same = True
        base_crs = None
        for layer in layers_list:  # type: QgsMapLayer
            base_crs = layer.crs() if base_crs is None else base_crs
            is_same = False if layer.crs() != base_crs else is_same

        return is_same and base_crs is not None

    def check_pixel_size_is_same(self, dem: QgsRasterLayer, layers_list: List[QgsRasterLayer]) -> bool:
        is_same: bool = True
        x, y = dem.rasterUnitsPerPixelX(), dem.rasterUnitsPerPixelY()
        for layer in layers_list:  # type: QgsRasterLayer
            lx, ly = layer.rasterUnitsPerPixelX(), layer.rasterUnitsPerPixelY()
            if lx != x or ly != y:
                is_same = False
        return is_same

    def get_project(self) -> LandsklimProject:
        """
        Getter for _project attribute
        """
        return self._project

    def update_project(self):
        """
        Update LandsklimProject sources and names without modifying configurations, analysis and interpolations.
        ViewNewProject is allowed to call LandsklimProject.update_project_definition as it own the responsability of hosted LandsklimProject object
        """
        project_name = self.get_project_name()
        project_stations = MapLayerCollection.create_from_vector_layers(self.get_project_stations()) # if not self._is_update else self._project.get_stations_sources()
        project_dem = RasterLayer(self.get_project_dem()) if not self._is_update else self._project.get_dem()
        project_variables = MapLayerCollection.create_from_raster_layers(self.get_project_variables())
        project_date = self.get_project_date()
        project_stations_no_data = self.get_no_data_stations()

        self._project.update_project_definition(project_name, project_stations, project_dem, project_variables, project_date, project_stations_no_data)

    def input_are_valid(self) -> Tuple[bool, List[str]]:
        """
        Check if all sources are valid.
        - No null entries for required fields
        - All layers are on the same CRS
        """
        is_ok = True
        errors = []

        # Project name must be defined
        if self.get_project_name() == "":
            is_ok = False
            errors.append("ERROR_PROJECT_NAME")

        # DEM must be defined
        if self.get_project_dem() is None:
            is_ok = False
            errors.append("ERROR_NO_DEM")

        # At least one shapefile stations must be defined
        if len(self.get_project_stations()) == 0:
            is_ok = False
            errors.append("ERROR_NO_STATIONS")

        # All layers must share the same CRS
        layers = []
        for layer in self.get_project_stations():  # type: QgsRasterLayer
            layers.append(layer)

        for layer in self.get_project_variables() :  # type: QgsRasterLayer
            layers.append(layer)
            if layer.name().split("_", 1) in RegressorFactory.default_prefixes():
                is_ok = False
                errors.append("ERROR_ADDITIONAL_VARIABLE_CAN_T_HAVE_THIS_NAME")

        if not self.check_crs_is_same(layers):
            is_ok = False
            errors.append("ERROR_CRS")

        if not self.check_no_duplicates(self.get_project_stations()):
            is_ok = False
            errors.append("ERROR_STATIONS_DUPLICATES")

        if not self.check_no_duplicates(self.get_project_variables()):
            is_ok = False
            errors.append("ERROR_VARIABLES_DUPLICATES")

        if not self.check_pixel_size_is_same(self.get_project_dem(), self.get_project_variables()):
            is_ok = False
            errors.append("ERROR_ADDITIONAL_VARIABLE_PIXEL_SIZE")

        return is_ok, errors
