import os

from abc import ABC, abstractmethod
from typing import List, Tuple, Dict

from qgis.PyQt import uic
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtWidgets import QLayout, QMessageBox
from qgis.core import QgsMapLayerProxyModel

from landsklim.lk.regressor_raster_variable import RegressorRasterVariable
from landsklim.lk.regressor import Regressor
from landsklim.lk.regressor_factory import RegressorDefinition
from landsklim import landsklim


class LandsklimDialog(QtWidgets.QDialog):
    """
    Represents the base dialog class for each Landsklim dialogs
    containing methods to manage projects, configuration and analysis
    """
    def __init__(self, parent=None):
        super(LandsklimDialog, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self._instance: landsklim.Landsklim = landsklim.Landsklim.instance()
        self._message_box = None
        self.setupUi(self)
        self.init_ui()

    def init_ui(self):
        """
        Init UI components
        """
        pass

    def refresh_ui(self):
        """
        Refresh UI components after a specific action was triggered
        """
        pass

    def input_are_valid(self) -> Tuple[bool, List[str]]:
        """
        Check if user inputs are valid
        """
        raise NotImplementedError

    def get_plugin_instance(self):
        return self._instance

    def get_configurations(self):
        """
        Get available configurations for this project
        """
        self.get_plugin_instance().get_landsklim_project()
        if self.get_plugin_instance().get_landsklim_project() is None:
            raise landsklim.NoLandsklimProjectException()

        return self.get_plugin_instance().get_landsklim_project().get_configurations()

    def get_analysis(self):
        """
        Get available analysis for the selected configuration
        """
        if self.get_plugin_instance().get_landsklim_project() is None:
            raise landsklim.NoLandsklimProjectException()

        return self.get_plugin_instance().get_landsklim_project().get_analysis()

    def clean_layout(self, layout: QLayout):
        item = layout.takeAt(0)
        while item is not None:
            layout.removeWidget(item.widget())
            item = layout.takeAt(0)

    def accept(self):
        is_valid, errors = self.input_are_valid()
        if is_valid:
            super().accept()
        elif self._instance.gui_exists():  # Ensure there is an interface before showing message
            message = self.tr("Errors have been reported.")
            self._message_box = QMessageBox.critical(self, "Landsklim", "{0}\n{1}".format(message, errors), QMessageBox.Ok)
            self._message_box = None

    def regressors_to_regressor_definitions(self, regressors: List[Regressor]) -> List[RegressorDefinition]:
        """
        Convert a list of Regressor to a list of RegressorDefinition

        :param regressors: List of regressors to convert
        :type regressors: List[Regressor]

        :returns: regressors converted into a list of RegressorDefinition
        :rtype: List[RegressorDefinition]
        """
        regressors_dict: Dict[str, List[(List[int], int)]] = {}

        if self._instance.get_landsklim_project() is not None:
            for regressor_name in self._instance.get_landsklim_project().available_regressors():  # type: str
                regressors_dict[regressor_name] = [[], 1]

        for regressor in regressors:
            regressor_name = regressor.prefix() if regressor.class_name() == RegressorRasterVariable.class_name() else regressor.class_name()
            regressors_dict[regressor_name][0].append(regressor.get_windows())
            regressors_dict[regressor_name][1] = regressor.get_polynomial_degree()
        res: List[RegressorDefinition] = []
        for class_name, (windows, polynomial_degree) in regressors_dict.items():
            res.append(RegressorDefinition(class_name, windows, polynomial_degree))
        return res
