from typing import Union, Tuple, List

from qgis._core import QgsProcessingParameterEnum, QgsProcessingParameterDefinition, QgsProcessingParameterExtent, \
    QgsProcessingParameterVectorLayer, QgsRectangle, QgsVectorLayer, QgsProcessingParameterVectorDestination, \
    QgsProcessingParameterFeatureSink, QgsProcessingParameterField, QgsField, QgsGeometry
from qgis.core import QgsProcessing, QgsProcessingAlgorithm, QgsProcessingParameterRasterLayer, \
    QgsProcessingParameterNumber, QgsProcessingParameterRasterDestination, QgsRasterLayer, QgsProcessingParameterBoolean

import numpy as np

from landsklim.processing.landsklim_processing_tool_algorithm import LandsklimProcessingToolAlgorithm

try:
    from geopandas import GeoDataFrame
except ImportError:
    pass


class MoranIProcessingAlgorithm(LandsklimProcessingToolAlgorithm):
    """
    Processing algorithm computing altitude average from a DEM
    """
    INPUT_POINTS_SHAPEFILE = 'INPUT_POINTS_SHAPEFILE'
    INPUT_FIELD = 'INPUT_FIELD'
    OUTPUT = 'OUTPUT'

    def createInstance(self):
        return MoranIProcessingAlgorithm()

    def name(self) -> str:
        """
        Unique name of the algorithm
        """
        return 'morani'

    def displayName(self) -> str:
        """
        Displayed name of the algorithm
        """
        return self.tr("Moran's I (on points)")

    def shortHelpString(self) -> str:
        return self.tr("Compute Moran's I on points based on the nearest neighbor method")

    def parameters(self):
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.INPUT_POINTS_SHAPEFILE,
                self.tr('Shapefile'),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                self.INPUT_FIELD,
                self.tr('Field'),
                type=QgsProcessingParameterField.Numeric,
                parentLayerParameterName=self.INPUT_POINTS_SHAPEFILE
            )
        )

    def initAlgorithm(self, config=None):
        """
        Define inputs and outputs for the main input
        """
        self.parameters()

    """
    # Functionnal implementation using GeoPandas
    
    def __nearest(self, data: pd.DataFrame, row: Series):
        data_copy = data.copy().drop(row.name)
        return data_copy.distance(row.geometry).idxmin()
        
    def moran_i_geodataframe(self, data: "GeoDataFrame") -> float:
        data = data.copy()
        w = np.zeros((len(data), len(data)))
        data['nearest'] = data.apply(lambda row: self.__nearest(data, row), axis=1)
        for index, row in data.iterrows():
            w[index, row["nearest"]] = 1
        return self.__moran_i(data['value'].values, w)
    """

    def moran_i(self, data: QgsVectorLayer, value_field_index: int) -> float:
        """
        Compute auto-correlation on a points dataset

        :param data: Source points layer
        :type data: QgsVectorLayer

        :param value_field_index: Index of field containing values to compute Moran's I on
        :type value_field_index: int

        :returns: Moran's I
        :rtype: float
        """
        data: QgsVectorLayer = self.copy_layer_to_memory(data)
        if len(data) < 2:
            raise RuntimeError("Shapefile must contain at least 2 points to compute Moran's I")
        nearest = {}
        w = np.zeros((len(data), len(data)))
        values = []
        for i, feat in enumerate(data.getFeatures()):  # type: QgsFeature
            geometry: QgsGeometry = feat.geometry()
            value = feat.attributes()[value_field_index]
            values.append(value)
            best_j = np.nan
            best_distance = np.nan
            for j, feat_search in enumerate(data.getFeatures()):  # type: QgsFeature
                if i != j:
                    geometry_search: QgsGeometry = feat_search.geometry()
                    distance: float = geometry_search.distance(geometry)
                    if best_distance is np.nan or distance < best_distance:
                        best_distance = distance
                        best_j = j
            nearest[i] = best_j
        for key, value in nearest.items():
            w[key, value] = 1
        del data
        return self.__moran_i(np.array(values), w)

    def copy_layer_to_memory(self, source: QgsVectorLayer) -> QgsVectorLayer:
        """
        Utility method designed to clone a QgsVectorLayer of points into a new layer with a memory provider, so data can be safely manipulated.

        :param source: Layer to clone
        :type source: QgsVectorLayer

        :returns: Cloned layer
        :rtype: QgsVectorLayer
        """
        feats = [feat for feat in source.getFeatures()]

        mem_layer: QgsVectorLayer = QgsVectorLayer("Point?crs=EPSG:4326", "duplicated_layer", "memory")
        mem_layer.setCrs(source.crs())

        mem_layer_data = mem_layer.dataProvider()
        attr = source.dataProvider().fields().toList()
        mem_layer_data.addAttributes(attr)
        mem_layer.updateFields()
        mem_layer_data.addFeatures(feats)
        return mem_layer

    def __moran_i(self, data: np.ndarray, w: np.ndarray) -> float:
        """
        Compute Moran's Index formula

        :param data: List of sample
        :type data: np.ndarray

        :param w: Matrix of weights between each sample
        :type w: np.ndarray

        :returns: Moran's Index
        :rtype: float
        """
        z_mean = data.mean()
        n = len(data)
        m = w.sum()
        if m == 0:
            raise RuntimeError("Moran's I : sum of weights must be greater than 0")
        weights = 0
        divider = 0
        for i in range(len(w)):
            zi = data[i]
            for j in range(len(w)):
                zj = data[j]
                weights += w[i, j] * (zi - z_mean) * (zj - z_mean)
            divider += (zi - z_mean) * (zi - z_mean)
        return (n/m) * (weights/divider)

    def processAlgorithm(self, parameters, context, feedback):
        """
        Called when a processing algorithm is run
        """
        input_points: QgsVectorLayer = self.parameterAsVectorLayer(parameters, self.INPUT_POINTS_SHAPEFILE, context)
        # Field name is got as a unique string (only one field is expected)
        input_field: str = self.parameterAsString(parameters, self.INPUT_FIELD, context)
        input_fields = [field.name() for field in input_points.fields()]
        field_index = input_fields.index(input_field)

        moran_i = self.moran_i(input_points, field_index)

        return {self.OUTPUT: moran_i}
