from typing import List, Dict, Union, Tuple
import os
import abc

import numpy as np
from osgeo import gdal, gdal_array
from qgis._core import QgsRectangle

from qgis.core import QgsMapLayer, QgsRasterLayer, QgsVectorLayer, QgsProject

from landsklim.lk.landsklim_constants import LandsklimLayerType, LAYER_TYPE_PATH


class MapLayer(metaclass=abc.ABCMeta):
    """
    Represents a map layer as interface with QGIS objects.

    :ivar str _landsklim_code:
        Represents unique layer ID created by Landsklim.
        It is formatted as ``[LAYER_TYPE]_[LAYER_CONTENT]``
        For the explicative variable derivated from altitude with a window size of 3, ``landsklim_code`` will be :
        ``expl_variables_altit_3``

    """

    def __init__(self, qgis_layer: QgsMapLayer):
        self._path: str = ""
        self._qgis_layer: QgsMapLayer = qgis_layer
        self._id: str = qgis_layer.id()
        self._landsklim_code: str = qgis_layer.customProperty('landsklim_code')
        if self._landsklim_code is not None and self.get_layer_type() == LandsklimLayerType.Kriging:
            if "[C_" not in self._landsklim_code:
                raise RuntimeError("Exception: Kriging map layer is invalid")

    def __getstate__(self):
        return {"id": self._id, "landsklim_code": self._landsklim_code}
        # return self._id, self._landsklim_code

    def __setstate__(self, state):
        # 0.8.0 migration
        if "lisdqs_code" in state:
            state["landsklim_code"] = state["lisdqs_code"]
        self._id = state["id"]
        self._landsklim_code = state["landsklim_code"]
        # self._id, self._landsklim_code = state
        self.resolve()

    def id(self):
        return self._qgis_layer.id()

    @abc.abstractmethod
    def qgis_layer(self):
        raise NotImplementedError

    @abc.abstractmethod
    def extension(self) -> str:
        raise NotImplementedError

    @abc.abstractmethod
    def open_qgis_layer(self, path: str, layer_name: str) -> QgsMapLayer:
        raise NotImplementedError

    def get_layer_type(self) -> LandsklimLayerType:
        """
        :returns: Layer type according to its landsklim code
        :rtype: LandsklimLayerType
        """

        layers_type = [LandsklimLayerType.ExplicativeVariable, LandsklimLayerType.Polygons, LandsklimLayerType.AdditionalLayer, LandsklimLayerType.Interpolation, LandsklimLayerType.Kriging]
        for layer_type in layers_type:
            if self._landsklim_code.startswith(str(layer_type)):
                return layer_type
        return LandsklimLayerType.NotDefined

    def get_layer_file_name(self) -> str:
        """
        :returns: Layer file name derived from its self._landsklim_code attribute.
            For example : {self._landsklim_code: 'expl_variable_altit_1'} returns altit_1.tif
        :rtype: str
        """
        return "{0}.{1}".format(self._landsklim_code.replace('{0}_'.format(self.get_layer_type()), '', 1).split("[")[0], self.extension())

    def get_layer_subfolders(self) -> List[str]:
        subfolders: List[str] = []
        if "[C_" in self._landsklim_code:
            configuration_folder = self._landsklim_code.split("[C_")[1].split("][A_")[0]
            analysis_folder = self._landsklim_code.split("][A_")[1].split("][I_")[0].replace("]", "")
            subfolders.append(configuration_folder)
            subfolders.append(analysis_folder)
            if "[I_" in self._landsklim_code:
                interpolation_folder = self._landsklim_code.split("][I_")[1].split("][P_")[0].replace("]", "")
                phase_folder = self._landsklim_code.split("][P_")[1].replace("]", "")
                subfolders.append(interpolation_folder)
                subfolders.append(phase_folder)
        return subfolders

    def resolve(self):
        """
        Get the QgsMapLayer referenced with this MapLayer object.
        By default, get the layer with the same ID.
        If the map layer was deleted by used, try to link the map layer with the same landsklim_code.
        If no map layer was found, reload the layer. The layer source is determined from the landsklim_code
        """
        project_instance = QgsProject.instance()
        map_layers: Dict[str, QgsMapLayer] = project_instance.mapLayers()
        resolved: bool = False
        if self._id in map_layers:
            self._qgis_layer = map_layers[self._id]
            resolved = True
        else:
            for qgis_layer_id, qgis_layer in map_layers.items():
                if qgis_layer.customProperty('landsklim_code', 'no_property') == self._landsklim_code:
                    self._qgis_layer = qgis_layer
                    resolved = True
        # No QGIS layer is linked. Need to load the layer.
        # Get the layer source from the landsklim_code attribute.
        if not resolved:
            qgis_project: QgsProject = QgsProject.instance()
            layer_type: LandsklimLayerType = self.get_layer_type()
            inner_path = LAYER_TYPE_PATH[layer_type]
            subfolders = self.get_layer_subfolders()
            if len(subfolders) > 0:
                inner_path = inner_path.format(*subfolders)
            path = os.path.join(qgis_project.homePath(), 'Landsklim', inner_path, self.get_layer_file_name())

            layer_name = os.path.basename(path).replace('.{0}'.format(self.extension()), '')
            layer: QgsMapLayer = self.open_qgis_layer(path, layer_name)
            # New QgsMapLayer, update referenced id so the MapLayer keeps a reference on the QgsMapLayer !
            self._id = layer.id()
            layers_subfolders = "" if "[C_" not in self._landsklim_code else "{0}{1}".format("[C_", self._landsklim_code.split("[C_")[1])
            loaded_layer_lk_code = "{0}_{1}{2}".format(layer_type, layer_name, layers_subfolders)
            layer.setCustomProperty('landsklim_code', loaded_layer_lk_code)
            print("[Add from map_layer]", layer.source(), self._landsklim_code)
            qgis_project.addMapLayer(layer)
            self._qgis_layer = layer
        return self._qgis_layer

    def to_json(self) -> Dict:
        state_dict: Dict = self.__getstate__()
        return state_dict


class VectorLayer(MapLayer):

    def __init__(self, qgis_layer: QgsVectorLayer):
        super().__init__(qgis_layer)

    def qgis_layer(self) -> QgsVectorLayer:
        return self.resolve()
        # return self._qgis_layer

    def extension(self) -> str:
        return "gpkg"

    def open_qgis_layer(self, path: str, layer_name: str) -> QgsMapLayer:
        return QgsVectorLayer(path, layer_name)


class RasterLayer(MapLayer):

    def __init__(self, qgis_layer: QgsRasterLayer):
        super().__init__(qgis_layer)

    def qgis_layer(self) -> QgsRasterLayer:
        return self.resolve()
        # return self._qgis_layer

    def extension(self) -> str:
        return "tif"

    def open_qgis_layer(self, path: str, layer_name: str) -> QgsMapLayer:
        return QgsRasterLayer(path, layer_name)

    def no_data(self) -> Union[int, float]:
        qgis_layer: QgsRasterLayer = self.qgis_layer()
        no_data_value = np.nan
        if qgis_layer is not None and os.path.exists(qgis_layer.source()):
            dataset: gdal.Dataset = gdal.Open(qgis_layer.source())
            no_data_value = dataset.GetRasterBand(1).GetNoDataValue()
            dataset = None  # Close the gdal.Dataset (in a non Pythonic way)
        return no_data_value

    def __clip(self, extent: QgsRectangle) -> gdal.Dataset:
        gdal.UseExceptions()
        memory_layer_name = '/vsimem/clip.tif'
        window = (extent.xMinimum(), extent.yMaximum(), extent.xMaximum(), extent.yMinimum())
        raster: QgsRasterLayer = self.qgis_layer()
        gdal.Translate(memory_layer_name, raster.source(), projWin=window)
        ds: gdal.Dataset = gdal.Open(memory_layer_name)
        return ds

    def clip_metadata(self, extent: QgsRectangle) -> Tuple[Union[int, float], List]:
        """
        Get metadata (no data and geotransform) of a clipping raster by an extent

        :param extent: Extent used to clip the raster
        :type extent: QgsRectangle

        :returns: Metadata (tuple with no data value and geotransform object)
        :rtype: Tuple[Union[int, float], List]
        """
        ds: gdal.Dataset = self.__clip(extent)
        geotransform: List = ds.GetGeoTransform()
        no_data: Union[int, float] = ds.GetRasterBand(1).GetNoDataValue()
        ds = None  # free gdal.Dataset
        return no_data, geotransform

    def clip(self, extent: QgsRectangle) -> np.ndarray:
        """
        Clip raster by an extent

        :param extent: Extent used to clip the raster
        :type extent: QgsRectangle

        :returns: clipped raster as numpy array
        :rtype: np.ndarray
        """
        raster: QgsRasterLayer = self.qgis_layer()
        ds: gdal.Dataset = self.__clip(extent)
        pixelType = raster.dataProvider().dataType(1)
        array: np.ndarray = np.array(ds.GetRasterBand(1).ReadAsArray(), dtype=gdal_array.GDALTypeCodeToNumericTypeCode(pixelType))
        ds = None  # free gdal.Dataset
        return array


class MapLayerCollection(list):
    """
    Represent a collection of MapLayers
    """

    def __init__(self):
        super(MapLayerCollection, self).__init__()

    @staticmethod
    def create_from_raster_layers(rasters_list: List[QgsRasterLayer]):
        collection: MapLayerCollection = MapLayerCollection()
        for layer in rasters_list:
            collection.append(RasterLayer(layer))
        return collection

    @staticmethod
    def create_from_vector_layers(vectors_list: List[QgsVectorLayer]):
        collection: MapLayerCollection = MapLayerCollection()
        for layer in vectors_list:
            collection.append(VectorLayer(layer))
        return collection

    def get_qgis_layers(self):
        qgis_layers = []
        for layer in self:
            qgis_layers.append(layer.qgis_layer())
        return qgis_layers

