# Needed to support type hint LandsklimProject for LandsklimConfiguration.get_project() method
# Convert type hint as str to avoid Class Not Found exceptions
from __future__ import annotations

from typing import List, Dict, Any
from typing import TYPE_CHECKING
import os
import shutil

# To avoid cyclic import of LandsklimProject as this import is mainly used for type hinting
from qgis._core import QgsProject

from landsklim.lk.utils import LandsklimUtils

if TYPE_CHECKING:
    from landsklim.lk.landsklim_project import LandsklimProject


from landsklim.lk.landsklim_analysis import LandsklimAnalysis
from landsklim.lk.regressor import Regressor


class LandsklimConfiguration:
    """
    Represents a configuration.
    Configuration defines predictive variables for the interpolation model
    """

    def __init__(self, name: str, project: LandsklimProject):
        self._name: str = name
        self._lk_project: LandsklimProject = project
        self._analysis: List[LandsklimAnalysis] = []

    def __setstate__(self, state: Dict[str, Any]):
        for k, v in state.items():
            if k == "_lisdqs_project":
                k = "_lk_project"
            setattr(self, LandsklimUtils.rename_attr("LISDQSConfiguration", "LandsklimConfiguration", k), v)

    def get_project(self) -> LandsklimProject:
        return self._lk_project

    def add_analysis(self, analysis: LandsklimAnalysis):
        self._analysis.append(analysis)

    def get_analysis(self):
        return self._analysis

    def get_name(self) -> str:
        return self._name

    def __str__(self):
        return self.get_name()

    def to_string(self):
        return self.get_name()

    def delete_analysis(self, analysis: LandsklimAnalysis):
        analysis_path = analysis.get_path()
        if analysis in self._analysis:
            analysis.delete()
            self._analysis.remove(analysis)
        return analysis_path

    def to_json(self) -> Dict:
        state_dict: Dict = self.__dict__.copy()
        state_dict.pop('_lk_project', None)
        return state_dict
