"""
From QGIS 3.28, QgsProject.instance() is unusually long (up to 0.5s). To avoid performance issues,
QgsProject's current instance is cached and updated when necessary.
"""

from typing import Optional
from qgis._core import QgsProject

import numpy as np
import pandas as pd

from landsklim.lk.logger import Log

# ===============================
# QGIS project cache
# ===============================


_qgis_project_cache: Optional[QgsProject] = None


def qgis_project_cache() -> QgsProject:
    """
    Get QgsProject current instance

    :returns: QgsProject instance
    :rtype: QgsProject
    """
    global _qgis_project_cache
    if _qgis_project_cache is None:
        _qgis_project_cache = QgsProject().instance()
    return _qgis_project_cache


def update_qgis_project_cache(qgis_project: QgsProject):
    """
    Update QgsProject instance.

    :param qgis_project: The current QgsProject instance
    :type qgis_project: QgsProject
    """
    global _qgis_project_cache
    _qgis_project_cache = qgis_project


# ===============================
# Interpolation cache
# ===============================

# This is not a best practice but for speed convenience this is a working temporary solution

_enable_caching: bool = False
_polygon_points_cache: Optional[np.ndarray] = None
_variables_dataset_cache: Optional[pd.DataFrame] = None


def enable_interpolation_cache():
    global _enable_caching, _polygon_points_cache, _variables_dataset_cache
    Log.info("[enable caching]")
    _enable_caching = True
    _polygon_points_cache = None
    _variables_dataset_cache = None


def disable_interpolation_cache():
    global _enable_caching, _polygon_points_cache, _variables_dataset_cache
    Log.info("[disable caching]")
    _enable_caching = False
    _polygon_points_cache = None
    _variables_dataset_cache = None


def update_polygon_points_cache(value: np.ndarray):
    global _polygon_points_cache
    _polygon_points_cache = value


def update_variables_dataset_cache(value: pd.DataFrame):
    global _variables_dataset_cache
    _variables_dataset_cache = value


def polygon_points_cache():
    return _polygon_points_cache


def variables_dataset_cache():
    return _variables_dataset_cache


def interpolation_cache_enabled() -> bool:
    return _enable_caching
