"""
Contains chart dialogs on analyses
"""

import os
import sys
from typing import List, Tuple, Dict, Union, Optional
import numpy as np
import pandas as pd
from PyQt5.QtCore import Qt
from PyQt5.QtGui import QPalette, QColor

from PyQt5.QtWidgets import QLabel, QApplication, QWidget, QCheckBox, QTableView, QComboBox, QTabWidget, QHBoxLayout, \
    QFileDialog

from qgis.PyQt.QtWidgets import QListWidgetItem, QDialog, QPushButton, QVBoxLayout, QGridLayout, QLayout
from qgis.PyQt import uic
from qgis.PyQt import QtCore
from qgis._core import QgsField
from qgis.core import QgsMapLayerProxyModel, QgsRasterLayer, QgsVectorLayer

from landsklim.lk.phase_kriging import PhaseKriging
from landsklim.lk.phase_multiple_regression import PhaseMultipleRegression
from landsklim.lk.landsklim_constants import DATASET_RESPONSE_VARIABLE, DATASET_COLUMN_X, DATASET_COLUMN_Y
from landsklim.lk.polygons_definition import PolygonsDefinition
from landsklim.lk.regressor import Regressor
from landsklim.lk.utils import LandsklimUtils
from landsklim.ui.utils import DialogSelection, TableModelPandas
from landsklim.ui.landsklim_dialog import LandsklimDialog
from landsklim.ui.charts import LandsklimChart
from landsklim import landsklim
from landsklim.lk.map_layer import MapLayerCollection
from matplotlib.backends.backend_qt5agg import NavigationToolbar2QT as NavigationToolbar

from landsklim.lk.landsklim_analysis import LandsklimAnalysis

SERIES_BY_STATIONS_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), 'widget_series_by_stations.ui'))
SERIES_BY_PHASES_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), 'widget_series_by_phase.ui'))
SERIES_EXPLICATIVE_VARIABLES_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), 'widget_series_explicative_variables.ui'))
DISTRIBUTION_EXPLICATIVE_VARIABLES_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), 'widget_distribution_explicative_variables.ui'))
STATS_EXPLICATIVE_VARIABLES_CLASS, _ = uic.loadUiType(os.path.join(os.path.dirname(__file__), 'widget_stats_explicative_variables.ui'))


class WidgetAnalysisSerieByStation(QWidget, SERIES_BY_STATIONS_CLASS):
    """
    Represents dialog showing series by stations
    """

    def __init__(self, analysis: LandsklimAnalysis, parent=None):
        super(WidgetAnalysisSerieByStation, self).__init__(parent)
        self.setupUi(self)


        self.__analysis: LandsklimAnalysis = analysis

        self.__checkbox_observed: QCheckBox = QCheckBox(self.tr("Observed"))
        self.__checkbox_observed.setChecked(True)
        self.__checkbox_observed.clicked.connect(self.on_checkbox_click)
        self.__checkbox_phases: List[QCheckBox] = []

        if len(self.__analysis.get_station_situations()) > 0:
            for phase in self.__analysis.get_phases(self.__analysis.get_station_situations()[0]):  # type: "IPhase"
                self.__checkbox_phases.append(QCheckBox(phase.name()))

        self.__chart = LandsklimChart()
        self.__toolbar = NavigationToolbar(self.__chart.get_canvas(), self)
        self.layout_chart.addWidget(self.__toolbar)
        self.layout_chart.addWidget(self.__chart.get_canvas())

        self.create_content()

        """screen_height: int = int(QApplication.desktop().screenGeometry().height() * 0.7)
        screen_width: int = screen_height
        self.resize(screen_width, screen_height)"""

    def clear_layout(self, layout: QLayout, delete_widget: bool):
        for i in reversed(range(layout.count())):
            widget: QWidget = layout.itemAt(i).widget()
            widget.setParent(None)
            if delete_widget:
                widget.deleteLater()

    def refresh_phases_checkbox(self):
        self.clear_layout(self.layout_phases, delete_widget=False)

        show_residuals: bool = self.rb_data_type_residuals.isChecked()
        if not show_residuals:
            self.layout_phases.addWidget(self.__checkbox_observed)
        for checkbox in self.__checkbox_phases:
            self.layout_phases.addWidget(checkbox)
            checkbox.clicked.connect(self.on_checkbox_click)

    def refresh_chart(self):
        station: int = self.cb_stations.currentIndex()
        show_residuals: bool = self.rb_data_type_residuals.isChecked()
        show_observed: bool = self.__checkbox_observed.isChecked() and not show_residuals
        show_phases: List[bool] = []
        for checkbox in self.__checkbox_phases:  # type: QCheckBox
            show_phases.append(checkbox.isChecked())

        x_series = {}
        if show_observed:
            x_series["Observed"] = {}
            for i, situation in enumerate(self.__analysis.get_station_situations()):
                x_series["Observed"][i] = self.__analysis._datasets[situation][DATASET_RESPONSE_VARIABLE].values[station]

        for situation_i, situation in enumerate(self.__analysis.get_station_situations()):
            for i, phase in enumerate(self.__analysis.get_phases(situation)):  # type: int, "IPhase"
                if self.__checkbox_phases[i].isChecked():
                    if phase.name() not in x_series:
                        x_series[phase.name()] = {}
                    data_x, data_y = self.__analysis.get_station_data(situation, station)
                    # TODO: Is prediction made on the model built with the entire dataset or without the current station ?
                    prediction: float = phase.predict(data_x, extrapolation_margin=None, no_data=self.__analysis.get_dem().no_data()) # Check it gives the right result? Should
                    if show_residuals:
                        x_series[phase.name()][situation_i] = prediction - data_y
                    else:
                        # Show model prediction
                        x_series[phase.name()][situation_i] = prediction

        x_ticks: Dict[int, str] = {}
        for i, situation in enumerate(self.__analysis.get_station_situations()):
            x_ticks[i] = self.__analysis.get_situation_name(situation)
        x_label: str = self.tr("Situations")

        self.__chart.plot_lines(x_series, x_label, "", "", x_ticks=x_ticks, display_marker=True)
        """self.clear_layout(self.layout_chart, delete_widget=True)
        self.layout_chart.addWidget(self.__chart_provider.plot_lines(x_series, x_label, "", "", x_ticks=x_ticks))
        self.setWindowTitle(self.tr("{0} - Série statistique").format(self.__analysis.get_name()))"""

    def on_station_selected(self, index):
        self.refresh_chart()

    def on_checkbox_click(self):
        self.refresh_chart()

    def on_radio_click(self):
        self.refresh_phases_checkbox()
        self.refresh_chart()

    def create_content(self):
        station_names: List[str] = self.__analysis.get_stations_names()
        for station_name in station_names:
            self.cb_stations.addItem(station_name)

        self.refresh_phases_checkbox()

        self.cb_stations.currentIndexChanged.connect(self.on_station_selected)
        self.rb_data_type_estimations.clicked.connect(self.on_radio_click)
        self.rb_data_type_residuals.clicked.connect(self.on_radio_click)
        self.refresh_chart()


class WidgetAnalysisSerieByPhase(QDialog, SERIES_BY_PHASES_CLASS):
    """
    Represents dialog showing series by phases
    """

    def __init__(self, analysis: LandsklimAnalysis, parent=None):
        super(WidgetAnalysisSerieByPhase, self).__init__(parent)

        self.setupUi(self)

        self.__analysis: LandsklimAnalysis = analysis

        self.create_content()

        self.__chart = LandsklimChart()
        self.__toolbar = NavigationToolbar(self.__chart.get_canvas(), self)
        self.layout_chart.addWidget(self.__toolbar)
        self.layout_chart.addWidget(self.__chart.get_canvas())

        self.rb_data_type_estimations.clicked.connect(self.on_radio_click)
        self.rb_data_type_residuals.clicked.connect(self.on_radio_click)
        self.cb_phases.currentIndexChanged.connect(self.on_phase_selected)

        self.__dialog_stations: DialogSelection = DialogSelection("Sélection des stations...", self.__analysis.get_stations_names(), self.__analysis.get_stations_names())

        self.pb_stations.clicked.connect(self.on_pb_stations_click)

        self.refresh_chart()

        """screen_height: int = int(QApplication.desktop().screenGeometry().height() * 0.7)
        screen_width: int = screen_height
        self.resize(screen_width, screen_height)
        self.setWindowTitle(self.tr("{0} - Série statistique").format(self.__analysis.get_name()))"""

    def create_content(self):
        self.refresh_phases_combobox()

    def on_pb_stations_click(self):
        self.__dialog_stations.exec()
        self.refresh_chart()

    def on_phase_selected(self, index):
        self.refresh_chart()

    def on_radio_click(self):
        self.refresh_phases_combobox()

    def refresh_phases_combobox(self):
        show_residuals: bool = self.rb_data_type_residuals.isChecked()
        self.cb_phases.clear()
        if not show_residuals:
            self.cb_phases.addItem(self.tr("Observed"), -1)
        if len(self.__analysis.get_station_situations()) > 0:
            for i, phase in enumerate(self.__analysis.get_phases(self.__analysis.get_station_situations()[0])):  # type: "IPhase"
                self.cb_phases.addItem(phase.name(), i)
        if self.cb_phases.count() > 0:
            self.cb_phases.setCurrentIndex(0)

    def refresh_chart(self):
        stations = self.__dialog_stations.get_selected()
        stations_name: List[str] = self.__analysis.get_stations_names()
        selected_phase: int = self.cb_phases.currentData()
        show_observed: bool = selected_phase == -1

        show_residuals: bool = self.rb_data_type_residuals.isChecked()

        x_series = {}
        if selected_phase is not None:

            valid_stations = {}
            for situation in self.__analysis.get_station_situations():
                valid_stations[situation] = self.__analysis.get_valid_stations(situation)

            for station in stations:  # type: str
                x_series[station] = {}
                station_index = stations_name.index(station)
                for i, situation in enumerate(self.__analysis.get_station_situations()):
                    if station_index in valid_stations[situation]:
                        if show_observed:
                            x_series[station][i] = self.__analysis._datasets[situation][DATASET_RESPONSE_VARIABLE].values[np.nonzero(valid_stations[situation] == station_index)[0][0]]
                        else:
                            phase: "IPhase" = self.__analysis.get_phases(situation)[selected_phase]
                            station_dataset_index = np.nonzero(valid_stations[situation] == station_index)[0][0]  #valid_stations[situation].index(station_index)
                            data_x, data_y = self.__analysis.get_station_data(situation, station_dataset_index)
                            # TODO: Is prediction made on CV or not ? Because currently Phase 3 ~== Observed
                            prediction: float = phase.predict(data_x, extrapolation_margin=None, no_data=self.__analysis.get_dem().no_data()).ravel()[0]  # Check it gives the right result? Should
                            if show_residuals:
                                x_series[station][i] = prediction - data_y.values[0]
                            else:
                                # Show phase prediction
                                x_series[station][i] = prediction
                    else:
                        x_series[station][i] = np.nan
        x_ticks: Dict[int, str] = {}
        for i, situation in enumerate(self.__analysis.get_station_situations()):
            x_ticks[i] = self.__analysis.get_situation_name(situation)
        x_label: str = "Situations"

        self.__chart.plot_lines(x_series, x_label, "", "", x_ticks=x_ticks, display_marker=True)


class WidgetAnalysisExplicativeVariableDistribution(QDialog, DISTRIBUTION_EXPLICATIVE_VARIABLES_CLASS):
    """
    Represents dialog showing explicative variables metrics distribution
    """

    def __init__(self, analysis: LandsklimAnalysis, parent=None):
        super(WidgetAnalysisExplicativeVariableDistribution, self).__init__(parent)
        self.setupUi(self)

        self.__analysis: LandsklimAnalysis = analysis

        self.__chart = LandsklimChart()
        self.__toolbar = NavigationToolbar(self.__chart.get_canvas(), self)
        self.layout_chart.addWidget(self.__toolbar)
        self.layout_chart.addWidget(self.__chart.get_canvas())

        variables: List[str] = [regressor.layer_name() for regressor in self.__analysis.get_regressors()]
        for variable in variables:
            self.cb_explicative_variable.addItem(variable)
        if len(variables) > 0:
            self.cb_explicative_variable.setCurrentIndex(0)
        self.cb_explicative_variable.currentIndexChanged.connect(self.on_variable_selected)

        self.rb_metric_r.clicked.connect(self.on_radio_click)
        self.rb_metric_coef.clicked.connect(self.on_radio_click)

        self.refresh_chart()

        """screen_height: int = int(QApplication.desktop().screenGeometry().height() * 0.7)
        screen_width: int = screen_height
        self.resize(screen_width, screen_height)
        self.setWindowTitle(self.tr("{0} - Distribution variable explicative").format(self.__analysis.get_name()))"""

    def on_variable_selected(self, index):
        self.refresh_chart()

    def on_radio_click(self):
        self.refresh_chart()

    def refresh_chart(self):
        show_r: bool = self.rb_metric_r.isChecked()
        selected_variable: int = self.cb_explicative_variable.currentIndex()
        metric_collection: List[float] = []
        variable: str = ""
        if selected_variable > -1:
            regressor: Regressor = self.__analysis._regressors[selected_variable]
            variable = regressor.layer_name()
            for i, situation in enumerate(self.__analysis.get_station_situations()):  # type: int, int
                metric: float = np.nan
                if show_r:
                    metric = self.__analysis.get_pearson_correlations(situation)[regressor][0]
                else:
                    regression_coefficients = self.__analysis.get_regression_coefficients(situation)
                    metric = regression_coefficients[regressor][0] if regressor in regression_coefficients else np.nan
                if not np.isnan(metric):
                    metric_collection.append(metric)
        x_label = "r" if show_r else "coef"
        title = "{1} - {0}".format(variable, self.tr("Distribution"))
        self.__chart.plot_hist(np.array(metric_collection), x_label=x_label, y_label="", title=title)


class WidgetAnalysisExplicativeVariableSeries(QDialog, SERIES_EXPLICATIVE_VARIABLES_CLASS):
    """
    Represents dialog showing explicative variables series
    """

    def __init__(self, analysis: LandsklimAnalysis, parent=None):
        super(WidgetAnalysisExplicativeVariableSeries, self).__init__(parent)
        self.setupUi(self)

        self.__analysis: LandsklimAnalysis = analysis

        self.__chart = LandsklimChart()
        self.__toolbar = NavigationToolbar(self.__chart.get_canvas(), self)
        self.layout_chart.addWidget(self.__toolbar)
        self.layout_chart.addWidget(self.__chart.get_canvas())

        variables: List[str] = [regressor.layer_name() for regressor in self.__analysis.get_regressors()]
        self.__dialog_variables: DialogSelection = DialogSelection("Sélection des variables", variables, variables)
        self.rb_metric_r.clicked.connect(self.on_radio_click)
        self.rb_metric_coef.clicked.connect(self.on_radio_click)
        self.pb_variables.clicked.connect(self.on_pb_variables_click)
        self.refresh_chart()

        """screen_height: int = int(QApplication.desktop().screenGeometry().height() * 0.7)
        screen_width: int = screen_height
        self.resize(screen_width, screen_height)
        self.setWindowTitle(self.tr("{0} - Série variable explicative").format(self.__analysis.get_name()))"""

    def on_pb_variables_click(self):
        self.__dialog_variables.exec()
        self.refresh_chart()

    def on_radio_click(self):
        self.refresh_chart()

    def refresh_chart(self):
        r_coef: bool = self.rb_metric_r.isChecked()

        x_series: Dict[str, Dict[float, float]] = {}

        for regressor_i in self.__dialog_variables.get_selected(get_indices=True):
            regressor: Regressor = self.__analysis._regressors[regressor_i]
            x_series[regressor.layer_name()] = {}
            for i, situation in enumerate(self.__analysis.get_station_situations()):
                regressor_metric: float
                if r_coef:
                    regressor_metric = self.__analysis.get_pearson_correlations(situation)[regressor][0]
                else:
                    regression_coefficients = self.__analysis.get_regression_coefficients(situation)
                    if regressor in regression_coefficients:
                        regressor_metric = regression_coefficients[regressor][0]
                    else:
                        regressor_metric = np.nan
                if regressor_metric is not None:
                    x_series[regressor.layer_name()][i] = regressor_metric

        x_ticks: Dict[int, str] = {}
        for i, situation in enumerate(self.__analysis.get_station_situations()):
            x_ticks[i] = self.__analysis.get_situation_name(situation)
        x_label: str = "Situations"

        self.__chart.plot_lines(x_series, x_label, "", "", x_ticks=x_ticks)


class WidgetAnalysisExplicativeVariable(QDialog, STATS_EXPLICATIVE_VARIABLES_CLASS):
    """
    Represents dialog showing explicative variables correlation frequency
    """

    def __init__(self, analysis: LandsklimAnalysis, parent=None):
        super(WidgetAnalysisExplicativeVariable, self).__init__(parent)
        self.setupUi(self)

        self.__analysis: LandsklimAnalysis = analysis

        self.__chart = LandsklimChart()
        self.__toolbar = NavigationToolbar(self.__chart.get_canvas(), self)
        self.layout_chart.addWidget(self.__toolbar)
        self.layout_chart.addWidget(self.__chart.get_canvas())

        self.cb_regression.clicked.connect(self.on_checkbox_click)
        self.spinbox_r.valueChanged.connect(self.on_spinbox_edit)
        self.refresh_chart()

        """screen_height: int = int(QApplication.desktop().screenGeometry().height() * 0.7)
        screen_width: int = screen_height
        self.resize(screen_width, screen_height)
        self.setWindowTitle(self.tr("{0} - Statistiques des variables explicatives").format(self.__analysis.get_name()))"""

    def on_spinbox_edit(self, i=0):
        self.refresh_chart()

    def on_checkbox_click(self):
        self.refresh_chart()

    def get_selected_regressors(self, labels: List[str]) -> List["Regressor"]:
        res: List["Regressor"] = []
        for label in labels:
            res.append(self.__analysis.regressor_from_name(label))
        return res

    def refresh_chart(self):
        only_regression: bool = self.cb_regression.isChecked()
        min_threshold: float = self.spinbox_r.value()
        situations_count = len(self.__analysis.get_station_situations())
        regressors_positive: Dict[Regressor, int] = {}
        regressors_negative: Dict[Regressor, int] = {}

        for regressor in self.__analysis.get_regressors():
            regressors_positive[regressor] = 0
            regressors_negative[regressor] = 0

        for situation in self.__analysis.get_station_situations():
            regression_phase: Optional[PhaseMultipleRegression] = None
            for phase in self.__analysis.get_phases(situation):  # type: IPhase
                if phase.class_name() == PhaseMultipleRegression.class_name():
                    regression_phase = phase

            if regression_phase is not None:
                selected_variables: List[Regressor] = self.get_selected_regressors(regression_phase.get_labels())
                for regressor in self.__analysis.get_regressors():
                    pearson_correlation = self.__analysis.get_pearson_correlations(situation)[regressor][0]
                    if abs(pearson_correlation) > min_threshold and ((not only_regression) or regressor in selected_variables):
                        if pearson_correlation < 0:
                            regressors_negative[regressor] += 1
                        else:
                            regressors_positive[regressor] += 1

        positive_values: np.ndarray = np.array([(value/situations_count)*100 for value in regressors_positive.values()])
        negative_values: np.ndarray = np.array([-(value/situations_count)*100 for value in regressors_negative.values()])
        regressors_labels: List[str] = [regressor.layer_name() for regressor in regressors_positive.keys()]
        self.__chart.plot_bar(positive_values, negative_values, regressors_labels, y_axis_as_percent=True, x_label="", y_label=self.tr("Frequency"), title=self.tr("Explicatives variables"))


class WidgetMetricsResume(QWidget):
    def __init__(self, analysis: LandsklimAnalysis, parent=None):
        super(WidgetMetricsResume, self).__init__(parent)
        self.__analysis: LandsklimAnalysis = analysis

        self.__chart = LandsklimChart()
        self.__toolbar = NavigationToolbar(self.__chart.get_canvas(), self)
        layout: QVBoxLayout = QVBoxLayout()

        self.__checkbox_error: QCheckBox = QCheckBox(self.tr("Show error"))
        self.__checkbox_error.setChecked(False)
        self.__checkbox_error.clicked.connect(self.on_checkbox_click)

        layout.addWidget(self.__checkbox_error)
        layout.addWidget(self.__toolbar)
        layout.addWidget(self.__chart.get_canvas())
        self.setLayout(layout)

        self.refresh_chart()

    def refresh_chart(self):

        x_ticks: Dict[int, str] = {}
        x_series: Dict[str, Dict[int, float]] = {}

        if self.__checkbox_error.isChecked():
            x_series = {"std résidus (Phase 1)": {}, "std résidus (Phase 3)": {}}
            for i, situation in enumerate(self.__analysis.get_station_situations()):
                x_ticks[i] = self.__analysis.get_situation_name(situation)
                x_series["std résidus (Phase 1)"][i] = self.__analysis.get_phases(situation)[0].get_residuals_standard_deviation()
                x_series["std résidus (Phase 3)"][i] = self.__analysis.get_phases(situation)[2].get_residuals_standard_deviation()
            self.__chart.plot_lines(x_series, "Situations", "std résidus", "", x_ticks=x_ticks, display_marker=True)
        else:
            all_metrics: List[float] = []
            x_series = {"R² (Phase 1)": {}, "AR² (Phase 1)": {}, "R² (Phase 3)": {}, "AR² (Phase 3)": {}}
            for i, situation in enumerate(self.__analysis.get_station_situations()):
                x_ticks[i] = self.__analysis.get_situation_name(situation)
                metrics = [self.__analysis.get_phases(situation)[0].r2(),
                          self.__analysis.get_phases(situation)[0].get_adjusted_r2(),
                          self.__analysis.get_phases(situation)[2].r2(),
                          self.__analysis.get_phases(situation)[2].get_adjusted_r2()]
                x_series["R² (Phase 1)"][i] = metrics[0]
                x_series["AR² (Phase 1)"][i] = metrics[1]
                x_series["R² (Phase 3)"][i] = metrics[2]
                x_series["AR² (Phase 3)"][i] = metrics[3]
                all_metrics += metrics
            min_value: float = min(0, min(all_metrics)) * 1.1
            self.__chart.plot_lines(x_series, self.tr("Situations"), "R²", "", x_ticks=x_ticks, display_marker=True, y_lim=(min_value, 1))


    def on_checkbox_click(self):
        self.refresh_chart()


class WidgetDataTable(QWidget):
    def __init__(self, analysis: LandsklimAnalysis, parent=None):
        super(WidgetDataTable, self).__init__(parent)
        self.__analysis: LandsklimAnalysis = analysis
        self.setMinimumWidth(1000)

        self.__table: QTableView = QTableView()
        self.__table.horizontalHeader().setStretchLastSection(True)
        self.__table.setAlternatingRowColors(True)
        self.__table.setSelectionBehavior(QTableView.SelectRows)

        self.__create_situations_combobox()
        self.__create_polygons_combobox()
        self.__combobox_situations.currentIndexChanged.connect(self.on_combo_box_value_changed)
        self.__combobox_polygons.currentIndexChanged.connect(self.on_combo_box_value_changed)
        self.refresh_table()

        self.__button_export_csv = QPushButton(self.tr("CSV export"))
        self.__button_export_csv.clicked.connect(self.__export_csv)

        layout: QVBoxLayout = QVBoxLayout()
        layout_header: QHBoxLayout = QHBoxLayout()
        layout_combo: QVBoxLayout = QVBoxLayout()
        layout_combo.addWidget(self.__combobox_situations)
        layout_combo.addWidget(self.__combobox_polygons)
        layout_header.addLayout(layout_combo, 3)
        layout_header.addWidget(self.__button_export_csv, 1)
        layout.addLayout(layout_header)
        layout.addWidget(self.__table)
        self.setLayout(layout)

    def __export_csv(self):
        dataset: pd.DataFrame = self.get_current_dataset()
        LandsklimUtils.export_csv(self, dataset)

    def __create_situations_combobox(self):
        self.__combobox_situations: QComboBox = QComboBox()
        for situation in self.__analysis.get_station_situations():  # type: "LandsklimAnalysis"
            self.__combobox_situations.addItem(self.__analysis.get_situation_name(situation), situation)

    def __create_polygons_combobox(self):
        self.__combobox_polygons: QComboBox = QComboBox()
        self.__combobox_polygons.view().setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        self.__combobox_polygons.addItem(self.tr("Entire dataset"), None)
        if self.__analysis.is_local():
            for i in range(self.__analysis.get_polygons().polygons_count()):
                self.__combobox_polygons.addItem(self.tr("Polygon {0}").format(i), i)

    def on_combo_box_value_changed(self):
        self.refresh_table()

    def get_current_dataset(self) -> pd.DataFrame:
        situation: int = self.__combobox_situations.currentData(QtCore.Qt.UserRole)
        polygon: Optional[int] = self.__combobox_polygons.currentData(QtCore.Qt.UserRole)
        dataset: pd.DataFrame = self.__analysis._datasets[situation]
        if polygon is not None:
            valid_keys = dataset.index.intersection(self.__analysis.get_polygons().polygons_definition()[polygon])
            dataset = dataset.loc[valid_keys]
        return dataset

    def refresh_table(self):
        dataset = self.get_current_dataset()
        table_model: TableModelPandas = TableModelPandas(dataset)
        self.__table.setModel(table_model)


class WidgetPolygons(QWidget):
    def __init__(self, polygons: PolygonsDefinition, local_neighborhood_size: int, parent=None):
        super(WidgetPolygons, self).__init__(parent)
        self.__polygons: PolygonsDefinition = polygons
        self.setMinimumWidth(1000)

        self.__tab_widget = QTabWidget()
        self.__tab_widget.addTab(self.__widget_polygons(), self.tr("Polygons"))
        self.__tab_widget.addTab(self.__widget_centroids(), self.tr("Centroids"))
        self.__tab_widget.addTab(self.__widget_connectedness(), self.tr("Connectedness"))

        layout: QVBoxLayout = QVBoxLayout()
        layout.addWidget(QLabel(self.tr("Local neighborhood size : {0}").format(local_neighborhood_size)))
        layout.addWidget(self.__tab_widget)
        self.setLayout(layout)

    def __widget_dataframe(self, dataframe: pd.DataFrame) -> QWidget:
        widget = QWidget()
        layout = QVBoxLayout()

        table: QTableView = QTableView()
        table.horizontalHeader().setStretchLastSection(True)
        table.setAlternatingRowColors(True)
        table.setSelectionBehavior(QTableView.SelectRows)
        table_model: TableModelPandas = TableModelPandas(dataframe)
        table.setModel(table_model)
        layout.addWidget(table)

        widget.setLayout(layout)
        return widget

    def __widget_polygons(self) -> QWidget:
        polygons: List[List[int]] = self.__polygons.polygons_definition()
        dataset: pd.DataFrame = pd.DataFrame(index=range(len(polygons)),columns=range(max(map(len, polygons))))
        for i, p in enumerate(polygons):
            dataset.iloc[i, :len(p)] = p
        return self.__widget_dataframe(dataset)

    def __widget_centroids(self) -> QWidget:
        dataset: pd.DataFrame = pd.DataFrame(self.__polygons.polygons_centroids(), columns=[DATASET_COLUMN_X, DATASET_COLUMN_Y])
        return self.__widget_dataframe(dataset)

    def __widget_connectedness(self) -> QWidget:
        polygons: List[List[int]] = self.__polygons.polygons_connectedness()
        dataset: pd.DataFrame = pd.DataFrame(index=range(len(polygons)),columns=range(max(map(len, polygons))))
        for i, p in enumerate(polygons):
            dataset.iloc[i, :len(p)] = p
        return self.__widget_dataframe(dataset)
