import datetime
import numpy as np
import pandas as pd
from PyQt5.QtCore import Qt, QDate

from PyQt5.QtWidgets import QLabel, QApplication, QWidget, QLineEdit, QDoubleSpinBox
from matplotlib.backends.backend_qt5agg import NavigationToolbar2QT as NavigationToolbar

from qgis.PyQt.QtWidgets import QListWidgetItem, QDialog, QPushButton, QVBoxLayout, QGridLayout, QLayout, QHBoxLayout, \
    QFrame, QSpinBox, QSpacerItem, QSizePolicy, QDateEdit
from qgis._core import QgsPointXY, QgsCoordinateTransform, QgsCoordinateReferenceSystem

from landsklim.lk.cache import qgis_project_cache
from landsklim.lk.utils import LandsklimUtils
from landsklim.ui.charts import LandsklimChart, ChartAttribute


class QHLine(QFrame):
    def __init__(self):
        super(QHLine, self).__init__()
        self.setFrameShape(QFrame.HLine)
        self.setFrameShadow(QFrame.Sunken)


class QVLine(QFrame):
    def __init__(self):
        super(QVLine, self).__init__()
        self.setFrameShape(QFrame.VLine)
        self.setFrameShadow(QFrame.Sunken)


class WidgetSunHeight(QWidget):
    """
    Represents dialog showing regressors correlation table for an analysis
    """

    def __init__(self, lat: float, lon: float, default_date: QDate, parent=None):
        super(WidgetSunHeight, self).__init__(parent)

        self.__lat = lat
        self.__lon = lon

        self.__layout: QLayout = QVBoxLayout()
        label_title = QLabel(self.tr("Sun Height - Lat: {0:.5f} Lon: {1:.5f}").format(lat, lon))
        label_title.setAlignment(Qt.AlignCenter)
        self.__layout.addWidget(label_title)

        self.__layout_content = QHBoxLayout()

        self.__layout_menu = QVBoxLayout()

        self.__button_day = QPushButton(self.tr("Over the day"))

        layout_day = QHBoxLayout()
        layout_day.addWidget(QLabel(self.tr("Day")))
        self.__edit_date: QDateEdit = QDateEdit()
        self.__edit_date.setDate(default_date)
        layout_day.addWidget(self.__edit_date)

        self.__button_year = QPushButton(self.tr("Over the year"))
        layout_hour = QHBoxLayout()
        layout_hour.addWidget(QLabel(self.tr("Hour (UTC)")))
        self.__edit_hour = QSpinBox()
        self.__edit_hour.setSuffix("h")
        self.__edit_hour.setMaximum(23)
        self.__edit_hour.setMinimum(0)
        layout_hour.addWidget(self.__edit_hour)

        layout_compare = QVBoxLayout()
        layout_compare.addWidget(QLabel("Compare with another location"))
        layout_compare.addWidget(QLabel("(coordinates on WGS 84)"))
        self.__edit_coord_x = QDoubleSpinBox()
        self.__edit_coord_x.setDecimals(6)
        self.__edit_coord_y = QDoubleSpinBox()
        self.__edit_coord_y.setDecimals(6)
        layout_coords = QHBoxLayout()
        layout_coords.addWidget(QLabel("Lon"))
        layout_coords.addWidget(self.__edit_coord_x)
        layout_coords.addWidget(QLabel("Lat"))
        layout_coords.addWidget(self.__edit_coord_y)
        layout_compare.addLayout(layout_coords)

        self.__layout_menu.addWidget(self.__button_day)
        self.__layout_menu.addWidget(self.__edit_date)
        self.__layout_menu.addWidget(QHLine())
        self.__layout_menu.addWidget(self.__button_year)
        self.__layout_menu.addLayout(layout_hour)
        self.__layout_menu.addWidget(QHLine())
        self.__layout_menu.addLayout(layout_compare)
        self.__layout_menu.addItem(QSpacerItem(20, 40, QSizePolicy.Policy.Minimum, QSizePolicy.Policy.Expanding))

        self.__layout_chart = QVBoxLayout()
        self.__layout_content.addLayout(self.__layout_menu)
        self.__layout_content.addWidget(QVLine())
        self.__layout_content.addLayout(self.__layout_chart)

        self.__layout.addLayout(self.__layout_content)

        self.__button_year.clicked.connect(lambda _: self.display_chart(True))
        self.__button_day.clicked.connect(lambda _: self.display_chart(False))

        self.__chart = LandsklimChart(chart_attributes=[ChartAttribute.SUNSET_BACKGROUND, ChartAttribute.PLOT_ONLY_WHITE_LINES, ChartAttribute.DRAW_LINE_ON_0])
        self.__toolbar = NavigationToolbar(self.__chart.get_canvas(), self)
        self.__layout_chart.addWidget(self.__toolbar)
        self.__layout_chart.addWidget(self.__chart.get_canvas())

        label_desc = QLabel(self.tr("Time shown here is not local time but solar time.\nUsed formula doesn't take account of earth obliquity and the eccentricity of its orbit.").format(lat, lon))
        label_desc.setAlignment(Qt.AlignCenter)
        label_desc.setFixedHeight(30)
        self.__layout_chart.addWidget(label_desc)

        self.setLayout(self.__layout)

        self.setWindowTitle(self.tr("Sun height"))

        screen_height: int = int(QApplication.desktop().screenGeometry().height() * 0.7)
        screen_width: int = screen_height
        self.resize(screen_width, screen_height)

    def get_point_for_comparison(self) -> QgsPointXY:
        """
        Get selected point for comparison

        :returns: Selected point on EPSG:4326
        :rtype: QgsPointXY
        """
        x_coord = self.__edit_coord_x.value()
        y_coord = self.__edit_coord_y.value()
        return QgsPointXY(x_coord, y_coord)

    def display_chart(self, is_year: bool):
        reference = self.get_point_for_comparison()

        qdate: QDate = self.__edit_date.date()

        positions = []
        positions_reference = []
        azimuths = []
        x_axis = None
        table = np.vstack((np.repeat(self.__lat, 24), np.repeat(self.__lon, 24), np.arange(24))).T
        table_reference = np.vstack((np.repeat(reference.y(), 24), np.repeat(reference.x(), 24), np.arange(24))).T
        if is_year:
            h = self.__edit_hour.value()
            date = datetime.datetime(qdate.year(), 1, 1, h, 0, 0)
            x_axis = np.arange(365)  # Days
            for _ in x_axis:
                positions.append(LandsklimUtils.sun_height(table, date.day, date.month, date.year)[h])  # (h-1)%24
                positions_reference.append(LandsklimUtils.sun_height(table_reference, date.day, date.month, date.year)[h])  # (h-1)%24
                date += datetime.timedelta(days=1)
        else:
            x_axis = np.arange(24)

            positions = LandsklimUtils.sun_height(table, qdate.day(), qdate.month(), qdate.year())
            positions_reference = LandsklimUtils.sun_height(table_reference, qdate.day(), qdate.month(), qdate.year())
        positions = np.rad2deg(np.array(positions))
        positions_reference = np.rad2deg(np.array(positions_reference))

        dict_altitudes = {h: p for p,h in zip(positions, x_axis)}
        dict_altitudes_reference = {h: p for p,h in zip(positions_reference, x_axis)}
        data = {self.tr('Altitude'): dict_altitudes, self.tr('Altitude (reference point)'): dict_altitudes_reference}

        x_label = self.tr("Year") if is_year else self.tr("Hour")
        self.__chart.plot_lines(data, x_label=x_label, y_label=self.tr("Altitude (degrees)"), title=self.tr("Sun position"), display_marker=not is_year, y_lim=(-90, 90), x_axis_as_month=is_year)
