import os
from typing import Union, List, Tuple, Optional

from qgis.PyQt import uic
from qgis.PyQt import QtWidgets
from qgis.core import QgsMapLayerProxyModel, QgsVectorLayer, QgsRectangle

from landsklim.lk.landsklim_interpolation import LandsklimInterpolationType, ExtrapolationMode
from landsklim.lk.landsklim_analysis import LandsklimAnalysis
from landsklim.lk.landsklim_configuration import LandsklimConfiguration
from landsklim.ui.landsklim_dialog import LandsklimDialog

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'view_new_interpolation.ui'))


class ViewNewInterpolation(LandsklimDialog, FORM_CLASS):
    """
    Represents the new interpolation dialog

    :ivar bool _interpolation_mode:
        Represents the interpolation mode between from a layer extent (``False``) or from a Point shapefile (``True``)

    """
    def __init__(self, parent=None):
        """Constructor."""
        self._interpolation_mode: bool = False
        super(ViewNewInterpolation, self).__init__(parent)

    def init_ui(self):
        """
        Init UI components
        """
        self.mlcb_type_shapefile.setFilters(QgsMapLayerProxyModel.PointLayer)

        for configuration in self.get_configurations():
            self.cb_configuration.addItem(str(configuration), configuration)

        self.cb_configuration.currentIndexChanged.connect(self.on_configuration_changed)
        if len(self.get_configurations()) > 0:
            self.on_configuration_changed(0)

        self.rb_zone_grid.toggled.connect(self.refresh_ui)
        self.rb_zone_points.toggled.connect(self.refresh_ui)

        self.rb_zone_grid.setChecked(True)

        self.extent_type.setOriginalExtent(self.get_plugin_instance().get_landsklim_project().get_dem().qgis_layer().extent(), self.get_plugin_instance().get_landsklim_project().get_dem().qgis_layer().crs())
        self.extent_type.setCurrentExtent(self.get_plugin_instance().get_landsklim_project().get_dem().qgis_layer().extent(), self.get_plugin_instance().get_landsklim_project().get_dem().qgis_layer().crs())
        self.extent_type.setOutputCrs(self.get_plugin_instance().get_landsklim_project().get_dem().qgis_layer().crs())

        self.cb_residual_regression.setEnabled(False)

        self.spinbox_min.setMinimum(float("-inf"))
        self.spinbox_min.setMaximum(float("inf"))
        self.spinbox_max.setMinimum(float("-inf"))
        self.spinbox_max.setMaximum(float("inf"))
        self.checkbox_limits.clicked.connect(self.click_interpolation_limits)
        self.update_limits_spinbox()

    def click_interpolation_limits(self):
        """
        Click on 'Define limits' checkbox
        """
        self.update_limits_spinbox()

    def on_configuration_changed(self, idx):
        self.cb_analysis.clear()
        for analysis in self.get_configurations()[idx].get_analysis():
            self.cb_analysis.addItem(analysis.to_string(), analysis)

    def enable_mode_layer_extent(self):
        self.extent_type.setEnabled(True)
        self.mlcb_type_shapefile.setEnabled(False)

    def enable_model_point_shapefile(self):
        self.extent_type.setEnabled(False)
        self.mlcb_type_shapefile.setEnabled(True)

    def refresh_ui(self):
        """
        Refresh UI components after a specific action was triggered
        """
        self._interpolation_mode = self.rb_zone_points.isChecked()
        if self._interpolation_mode:
            self.enable_model_point_shapefile()
        else:
            self.enable_mode_layer_extent()

    def input_are_valid(self) -> Tuple[bool, List[str]]:
        is_ok = True
        errors = []
        if self.selected_configuration() is None:
            is_ok = False
            errors.append('ERROR_NO_CONFIGURATION')
        if self.selected_analysis() is None:
            is_ok = False
            errors.append('ERROR_NO_ANALYSIS')
        if self.selected_interpolation_extent() is None:
            is_ok = False
            errors.append('ERROR_NO_EXTENT')
        if self.name_already_taken():
            is_ok = False
            errors.append('ERROR_NAME_ALREADY_TAKEN')
        if self.interpolation_name().strip() == "":
            is_ok = False
            errors.append("ERROR_INVALID_NAME")
        if len(self.selected_phases()) == 0:
            is_ok = False
            errors.append("ERROR_INTERPOLATION_TYPE")

        return is_ok, errors

    def interpolation_name(self) -> str:
        return self.le_interpolation_name.text()

    def name_already_taken(self) -> bool:
        res: bool = False
        if self.selected_analysis() is not None:
            res = self.interpolation_name() in [i.get_name() for i in self.selected_analysis().get_interpolations()]
        return res

    def selected_configuration(self) -> LandsklimConfiguration:
        return self.cb_configuration.currentData()

    def selected_analysis(self) -> LandsklimAnalysis:
        return self.cb_analysis.currentData()

    def selected_phases(self) -> List[LandsklimInterpolationType]:
        res = []
        if self.cb_partial_1.isChecked():
            res.append(LandsklimInterpolationType.PartialPhase1)
        if self.cb_partial_2.isChecked():
            res.append(LandsklimInterpolationType.PartialPhase2)
        if self.cb_global.isChecked():
            res.append(LandsklimInterpolationType.Global)
        if self.cb_residual_regression.isChecked():
            res.append(LandsklimInterpolationType.AutoRegression)
        """res = LandsklimInterpolationType.PartialPhase1 if self.cb_partial_1.isChecked() else res
        res = LandsklimInterpolationType.PartialPhase2 if self.cb_partial_2.isChecked() else res
        res = LandsklimInterpolationType.Global if self.cb_global.isChecked() else res
        res = LandsklimInterpolationType.AutoRegression if self.cb_residual_regression.isChecked() else res"""
        return res

    def get_extrapolation_mode(self) -> ExtrapolationMode:
        extrapolation_mode: ExtrapolationMode = ExtrapolationMode.Smooth
        if self.rb_e_smooth.isChecked():
            extrapolation_mode = ExtrapolationMode.Smooth
        if self.rb_e_nodata.isChecked():
            extrapolation_mode = ExtrapolationMode.NoData
        if self.rb_e_value.isChecked():
            extrapolation_mode = ExtrapolationMode.Value
        return extrapolation_mode

    def get_extrapolation_value(self) -> float:
        return self.sb_e_value.value()

    def selected_extrapolation_margin(self) -> float:
        return self.sb_extrapolation_margin.value() / 100

    def is_interpolation_on_grid(self) -> bool:
        return self.rb_zone_grid.isChecked()

    def selected_interpolation_extent(self) -> Union[QgsVectorLayer, QgsRectangle]:
        extent: Union[QgsVectorLayer, QgsRectangle, None] = None
        if self.is_interpolation_on_grid():
            extent = self.extent_type.outputExtent()
            if extent.area() == 0:
                extent = None
        else:
            extent = self.mlcb_type_shapefile.currentLayer()
        return extent

    def update_limits_spinbox(self):
        enabled: bool = self.checkbox_limits.isChecked()
        self.spinbox_min.setEnabled(enabled)
        self.spinbox_max.setEnabled(enabled)

    def get_minimum_value(self) -> Optional[float]:
        return self.spinbox_min.value() if self.checkbox_limits.isChecked() else None

    def get_maximum_value(self) -> Optional[float]:
        return self.spinbox_max.value() if self.checkbox_limits.isChecked() else None
