import os
import sys
import time
from enum import Enum
from typing import List, Tuple, Dict, Optional

import pandas as pd
import pyproj
from PyQt5.QtCore import Qt, QDate, QModelIndex
from PyQt5.QtGui import QStandardItemModel, QStandardItem, QIcon
from PyQt5.QtWidgets import QListWidgetItem, QWidget, QSizePolicy, QLayout, QColumnView
from qgis.PyQt import uic
from qgis.PyQt import QtWidgets, QtCore
from qgis._core import QgsCoordinateTransform, QgsCoordinateReferenceSystem, QgsPointXY

from landsklim.lk.cache import qgis_project_cache
from landsklim.lk.landsklim_project import LandsklimProject
from landsklim.lk.landsklim_analysis import LandsklimAnalysisMode, LandsklimAnalysis
from landsklim.lk.phase import IPhase
from landsklim.lk.phase_composite import PhaseComposite
from landsklim.lk.phase_kriging import PhaseKriging
from landsklim.lk.phase_multiple_regression import PhaseMultipleRegression
from landsklim.lk.utils import LandsklimUtils
from landsklim.ui.widgets_analysis_charts import WidgetAnalysisSerieByStation, \
    WidgetAnalysisExplicativeVariable, WidgetAnalysisExplicativeVariableSeries, \
    WidgetAnalysisExplicativeVariableDistribution, WidgetAnalysisSerieByPhase, WidgetMetricsResume, \
    WidgetDataTable, WidgetPolygons
from landsklim.ui.widgets_phase_view import WidgetPhaseMultipleRegressionView, \
    WidgetPhaseKrigingView, WidgetPhaseCompositeView, WidgetLocalMultipleRegressionCorrelationView, \
    WidgetGlobalMultipleRegressionCorrelationView
from landsklim.ui.widget_variables_correlation import WidgetVariablesCorrelation
from landsklim.ui.widget_sun_height import WidgetSunHeight
from landsklim.ui.landsklim_dialog import LandsklimDialog


# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'view_charts.ui'))


class LandsklimChartType(Enum):
    Correlation_Variables = 0
    AnalysisSeriesByStation = 1
    AnalysisSeriesByPhase = 2
    AnalysisExplicativeVariableSeries = 3
    AnalysisExplicativeVariableDistribution = 4
    AnalysisExplicativeVariable = 5
    Correlation_Table = 6
    Phase = 7
    SunHeight = 8
    MetricsSummary = 9
    Data_Table = 10
    Polygons = 11
    ExportRegression = 12
    ExportCoefficients = 13
    ExportMetrics = 14


class ViewCharts(LandsklimDialog, FORM_CLASS):

    """
    Represents the new analysis dialog
    """
    def __init__(self, landsklim_project: LandsklimProject, parent=None):
        """Constructor."""
        self.__project: LandsklimProject = landsklim_project
        super(ViewCharts, self).__init__(parent)
        self.setModal(False)

    def init_ui(self):
        """
        Init UI components
        """

        icon_chart = ':/plugins/landsklim/mActionLandsklimCharts.png'
        icon_analysis = ':/plugins/landsklim/mActionLandsklimAnalysis.png'
        icon_empty = ':/plugins/landsklim/icon.png'

        model = QStandardItemModel()
        for analysis in self.__project.get_analysis():  # type: LandsklimAnalysis
            item = QStandardItem(QIcon(icon_analysis), analysis.get_name())
            model.appendRow(item)
            for isituation, situation in enumerate(analysis.get_station_situations()):
                situation_item = QStandardItem(QIcon(icon_empty), analysis.get_situation_name(situation))
                item.appendRow(situation_item)

                regression_phase: Optional[PhaseMultipleRegression] = None
                for iphase, phase in enumerate(analysis.get_phases(situation)):
                    item_phase = QStandardItem(QIcon(icon_chart), phase.name())
                    item_phase.setData((LandsklimChartType.Phase, analysis, isituation, iphase), QtCore.Qt.UserRole)
                    situation_item.appendRow(item_phase)
                    regression_phase = phase if phase.class_name() == PhaseMultipleRegression.class_name() else regression_phase

                if regression_phase is not None:
                    item_correlation = QStandardItem(QIcon(icon_chart), self.tr("Coefficients"))
                    item_correlation.setData((LandsklimChartType.Correlation_Variables, analysis, regression_phase), QtCore.Qt.UserRole)
                    situation_item.appendRow(item_correlation)

            item_correlation = QStandardItem(QIcon(icon_chart), self.tr("Correlation table"))
            item_correlation.setData((LandsklimChartType.Correlation_Table, analysis), QtCore.Qt.UserRole)
            item.appendRow(item_correlation)
            item_metrics_resume = QStandardItem(QIcon(icon_chart), self.tr("Summary of metrics"))
            item_metrics_resume.setData((LandsklimChartType.MetricsSummary, analysis), QtCore.Qt.UserRole)
            item.appendRow(item_metrics_resume)
            item_data_table = QStandardItem(QIcon(icon_chart), self.tr("Data table"))
            item_data_table.setData((LandsklimChartType.Data_Table, analysis), QtCore.Qt.UserRole)
            item.appendRow(item_data_table)
            if analysis.is_local():
                item_polygons = QStandardItem(QIcon(icon_chart), self.tr("Polygons"))
                item_polygons.setData((LandsklimChartType.Polygons, analysis), QtCore.Qt.UserRole)
                item.appendRow(item_polygons)

            item_stat_1 = QStandardItem(QIcon(icon_chart), self.tr("Stations series"))
            item_stat_1.setData((LandsklimChartType.AnalysisSeriesByStation, analysis), QtCore.Qt.UserRole)
            item.appendRow(item_stat_1)
            item_stat_2 = QStandardItem(QIcon(icon_chart), self.tr("Phase series"))
            item_stat_2.setData((LandsklimChartType.AnalysisSeriesByPhase, analysis), QtCore.Qt.UserRole)
            item.appendRow(item_stat_2)

            if not analysis.is_local():
                item_stat_3 = QStandardItem(QIcon(icon_chart), self.tr("Explicative variables - Distributions"))
                item_stat_3.setData((LandsklimChartType.AnalysisExplicativeVariableDistribution, analysis), QtCore.Qt.UserRole)
                item.appendRow(item_stat_3)
                item_stat_4 = QStandardItem(QIcon(icon_chart), self.tr("Explicative variables - Series"))
                item_stat_4.setData((LandsklimChartType.AnalysisExplicativeVariableSeries, analysis), QtCore.Qt.UserRole)
                item.appendRow(item_stat_4)
                item_stat_5 = QStandardItem(QIcon(icon_chart), self.tr("Explicative variables - Frequencies"))
                item_stat_5.setData((LandsklimChartType.AnalysisExplicativeVariable, analysis), QtCore.Qt.UserRole)
                item.appendRow(item_stat_5)

            item_export_regression = QStandardItem(QIcon(icon_empty), self.tr("Export Regressions"))
            item_export_regression.setData((LandsklimChartType.ExportRegression, analysis), QtCore.Qt.UserRole)
            item.appendRow(item_export_regression)
            item_export_coefficients = QStandardItem(QIcon(icon_empty), self.tr("Export Correlation coefficients"))
            item_export_coefficients.setData((LandsklimChartType.ExportCoefficients, analysis), QtCore.Qt.UserRole)
            item.appendRow(item_export_coefficients)
            item_export_metrics = QStandardItem(QIcon(icon_empty), self.tr("Export Metrics"))
            item_export_metrics.setData((LandsklimChartType.ExportMetrics, analysis), QtCore.Qt.UserRole)
            item.appendRow(item_export_metrics)



        item_sun = QStandardItem(QIcon(icon_chart), self.tr("Sun height"))
        item_sun.setData((LandsklimChartType.SunHeight,), QtCore.Qt.UserRole)
        model.appendRow(item_sun)
        self.columnview.setColumnWidths([200, 200, 200, 0])
        self.columnview.setResizeGripsVisible(False)
        self.columnview.clicked.connect(self.on_select)
        self.columnview.setModel(model)

    def closeEvent(self, evnt):
        self.clear_layout(self.layout_view)

    def clear_layout(self, layout: QLayout):
        """
        Remove the chart currently displayed
        https://stackoverflow.com/questions/9374063/remove-all-items-from-a-layout
        """
        if layout is not None:
            while layout.count():
                item = layout.takeAt(0)
                widget = item.widget()
                if widget is not None:
                    # A widget define a clear() method destroying gdal_polygonize temporary layer used to display
                    # coefficients (WidgetPhaseMultipleRegressionView)
                    try:
                        widget.clear()
                    except AttributeError as e:
                        pass
                    widget.deleteLater()
                else:
                    self.clear_layout(item.layout())

    def on_select(self, model_index: QModelIndex):
        model_data = self.columnview.model().data(model_index, QtCore.Qt.UserRole)

        if model_data is not None:
            model_data_type = model_data[0]

            analysis_charts = [LandsklimChartType.AnalysisSeriesByStation, LandsklimChartType.AnalysisExplicativeVariable, LandsklimChartType.AnalysisExplicativeVariableSeries, LandsklimChartType.AnalysisExplicativeVariableDistribution, LandsklimChartType.AnalysisSeriesByPhase, LandsklimChartType.Correlation_Table, LandsklimChartType.MetricsSummary, LandsklimChartType.Data_Table, LandsklimChartType.Polygons]
            exports = [LandsklimChartType.ExportMetrics, LandsklimChartType.ExportRegression, LandsklimChartType.ExportCoefficients]

            if model_data_type == LandsklimChartType.SunHeight:
                widget = self.widget_sun_chart()
            if model_data_type in analysis_charts:
                analysis = model_data[1]
                widget = self.widget_analysis_chart(model_data_type, analysis)
            if model_data_type == LandsklimChartType.Phase:
                analysis: LandsklimAnalysis = model_data[1]
                situation: int = analysis.get_station_situations()[model_data[2]]
                phase: IPhase = analysis.get_phases(situation)[model_data[3]]
                widget = self.widget_phase_chart(phase)
            if model_data_type == LandsklimChartType.Correlation_Variables:
                analysis: LandsklimAnalysis = model_data[1]
                phase: PhaseMultipleRegression = model_data[2]
                widget = self.widget_correlation(analysis, phase)
            if model_data_type in exports:
                analysis: LandsklimAnalysis = model_data[1]
                widget = None
                self.export_csv(model_data_type, analysis)

            if widget is not None:
                self.clear_layout(self.layout_view)
                self.layout_view.addWidget(widget)

    def widget_correlation(self, analysis: LandsklimAnalysis, phase_regression: PhaseMultipleRegression) -> QWidget:
        if analysis.is_local():
            return WidgetLocalMultipleRegressionCorrelationView(phase_regression)
        else:
            return WidgetGlobalMultipleRegressionCorrelationView(phase_regression)

    def widget_sun_chart(self) -> QWidget:
        dem_extent = self.__project.get_dem().qgis_layer().extent()
        x_mean = ((dem_extent.xMaximum() - dem_extent.xMinimum()) / 2) + dem_extent.xMinimum()
        y_mean = ((dem_extent.yMaximum() - dem_extent.yMinimum()) / 2) + dem_extent.yMinimum()

        """p1 = pyproj.CRS(self.__project.get_dem().qgis_layer().crs().authid())
        p2 = pyproj.CRS('EPSG:4326')
        lat, lon = pyproj.transform(p1, p2, x_mean, y_mean)"""
        tr = QgsCoordinateTransform(self.__project.get_dem().qgis_layer().crs(), QgsCoordinateReferenceSystem("EPSG:4326"), qgis_project_cache())
        point = tr.transform(QgsPointXY(x_mean, y_mean))
        lon, lat = point.x(), point.y()

        default_date: QDate = QDate(*self.__project.get_date())

        return WidgetSunHeight(lat, lon, default_date)

    def widget_analysis_chart(self, chart_type: LandsklimChartType, analysis: LandsklimAnalysis) -> QWidget:

        widget: QWidget = None
        if chart_type == LandsklimChartType.AnalysisSeriesByStation:
            widget: WidgetAnalysisSerieByStation = WidgetAnalysisSerieByStation(analysis)
        elif chart_type == LandsklimChartType.AnalysisExplicativeVariable:
            widget: WidgetAnalysisExplicativeVariable = WidgetAnalysisExplicativeVariable(analysis)
        elif chart_type == LandsklimChartType.AnalysisExplicativeVariableSeries:
            widget: WidgetAnalysisExplicativeVariableSeries = WidgetAnalysisExplicativeVariableSeries(analysis)
        elif chart_type == LandsklimChartType.AnalysisExplicativeVariableDistribution:
            widget: WidgetAnalysisExplicativeVariableDistribution = WidgetAnalysisExplicativeVariableDistribution(analysis)
        elif chart_type == LandsklimChartType.AnalysisSeriesByPhase:
            widget: WidgetAnalysisSerieByPhase = WidgetAnalysisSerieByPhase(analysis)
        elif chart_type == LandsklimChartType.Correlation_Table:
            widget: WidgetVariablesCorrelation = WidgetVariablesCorrelation(analysis)
        elif chart_type == LandsklimChartType.MetricsSummary:
            widget: WidgetMetricsResume = WidgetMetricsResume(analysis)
        elif chart_type == LandsklimChartType.Data_Table:
            widget: WidgetDataTable = WidgetDataTable(analysis)
        elif chart_type == LandsklimChartType.Polygons:
            widget: WidgetPolygons = WidgetPolygons(analysis.get_polygons(), analysis.get_neighborhood_size())

        return widget

    def widget_phase_chart(self, phase: IPhase) -> QWidget:
        if phase.class_name() == PhaseMultipleRegression.class_name():
            widget = WidgetPhaseMultipleRegressionView(phase)
        if phase.class_name() == PhaseKriging.class_name():
            widget = WidgetPhaseKrigingView(phase)
        if phase.class_name() == PhaseComposite.class_name():
            widget = WidgetPhaseCompositeView(phase)
        return widget

    def export_csv(self, chart_type: LandsklimChartType, analysis: LandsklimAnalysis):
        if chart_type == LandsklimChartType.ExportCoefficients:
            dataset: pd.DataFrame = analysis.get_dataset_coefficients()
        if chart_type == LandsklimChartType.ExportRegression:
            dataset: pd.DataFrame = analysis.get_dataset_regression()
        if chart_type == LandsklimChartType.ExportMetrics:
            dataset: pd.DataFrame = analysis.get_dataset_metrics()
        LandsklimUtils.export_csv(self, dataset)
