import traceback
from typing import List

from PyQt5.QtWidgets import QMessageBox
from qgis.PyQt.QtCore import QThread, pyqtSignal

from landsklim.ui.coverage_utils import coverage_resolve_trace
from landsklim.lk.landsklim_project import LandsklimProject
from landsklim.lk.landsklim_analysis import LandsklimAnalysis, LandsklimInterpolation
from landsklim.lk.regressor import Regressor


class QThreadComputeInterpolations(QThread):
    """
    Thread for computing interpolations for each situation on background

    :param analysis: Analysis containing regressors definitions
    :type analysis: LandsklimAnalysis

    """
    # Signal emitted at startup
    computationStarted = pyqtSignal(int)
    # Signal emitted when a variable finished computing
    interpolationComputed = pyqtSignal(str, int, int)
    # Signal emitted when all variables have been computed
    computationFinished = pyqtSignal()
    # Signal emitted when an exception is raised during interpolation
    interpolationFailed = pyqtSignal(LandsklimAnalysis, LandsklimInterpolation)

    def __init__(self, analysis: LandsklimAnalysis, interpolation: LandsklimInterpolation):
        super().__init__()
        self.__analysis: LandsklimAnalysis = analysis
        self.__interpolation: LandsklimInterpolation = interpolation
        self.__analysis.handle_on_interpolation_computation_started(self.on_computation_started)
        self.__analysis.handle_on_interpolation_computation_step(self.on_interpolation_computed)
        self.__analysis.handle_on_interpolation_computation_finished(self.on_computation_finished)
        self.__analysis.handle_on_interpolation_computation_failed(self.on_interpolation_failed)

    # Merges covered statements from threads inside main code coverage
    @coverage_resolve_trace
    def run(self):
        self.__analysis.compute_interpolations(self.__interpolation)

    @coverage_resolve_trace
    def on_computation_started(self, total: int):

        self.computationStarted.emit(total)

    @coverage_resolve_trace
    def on_interpolation_computed(self, situation_name: str, i: int, total: int):

        self.interpolationComputed.emit(situation_name, i, total)

    @coverage_resolve_trace
    def on_computation_finished(self):
        self.computationFinished.emit()

    @coverage_resolve_trace
    def on_interpolation_failed(self, analysis: LandsklimAnalysis, interpolation: LandsklimInterpolation):
        self.interpolationFailed.emit(analysis, interpolation)


class QThreadComputeVariables(QThread):
    """
    Thread for computing variables for each regressor on background

    :param project: Landsklim project
    :type project: LandsklimProject

    :param only_variables: If True, polygons and models will not be computed once variables are created.
        The value of this attribute is passed through the finish signal
    :type only_variables: bool

    """
    # Signal emitted at startup
    computationStarted = pyqtSignal(int)
    # Signal emitted when a variable finished computing
    explicativeVariableComputed = pyqtSignal(Regressor, int, int, int)
    # Signal emitted when all variables have been computed
    computationFinished = pyqtSignal(bool)

    def __init__(self, project: LandsklimProject, only_variables: bool):
        super().__init__()
        self.__project: LandsklimProject = project
        self.__project.handle_on_explicative_variables_compute_started(self.on_computation_started)
        self.__project.handle_on_regressor_compute_finished(self.on_explicative_variable_computed)
        self.__project.handle_on_explicative_variables_compute_finished(self.on_computation_finished)

        self.__only_variables: bool = only_variables

    # Merges covered statements from threads inside main code coverage
    @coverage_resolve_trace
    def run(self):
        self.__project.create_explicative_variables()

    @coverage_resolve_trace
    def on_computation_started(self, total: int):
        """
        When computation will be launched. Give the total of variables to compute

        :param total: Variables count to compute from regressors
        :type total: int
        """
        self.computationStarted.emit(total)

    @coverage_resolve_trace
    def on_explicative_variable_computed(self, regressor: Regressor, window_size: int, i: int, total: int):
        """
        When a variable finished computing

        :param regressor: Regressor used to compute the variable
        :type regressor: Regressor

        :param window_size: Window size used to compute variable raster from regressor
        :type window_size: int

        :param i: ith variable computed
        :type i: int

        :param total: Variables count to compute
        :type total: int
        """
        self.explicativeVariableComputed.emit(regressor, window_size, i, total)

    @coverage_resolve_trace
    def on_computation_finished(self):
        """
        When all variables have been computed
        """
        self.computationFinished.emit(self.__only_variables)



class QThreadComputePolygons(QThread):
    """
    Thread for computing polygons on background

    :param analysis: Analysis containing neighborhood definition
    :type analysis: LandsklimAnalysis

    """
    # Signal emitted at startup
    computationStarted = pyqtSignal(str)
    # Signal emitted when polygons have been computed
    computationFinished = pyqtSignal()
    # Signal emitted when an error occurred
    computationFailed = pyqtSignal(LandsklimAnalysis)

    def __init__(self, analysis: LandsklimAnalysis):
        super().__init__()
        self.__analysis: LandsklimAnalysis = analysis
        self.__analysis.handle_on_polygons_start(self.on_computation_started)
        self.__analysis.handle_on_polygons_end(self.on_computation_finished)
        self.__analysis.handle_on_polygons_failed(self.on_computation_failed)

    # Merges covered statements from threads inside main code coverage
    @coverage_resolve_trace
    def run(self):
        self.__analysis.create_polygons()

    @coverage_resolve_trace
    def on_computation_started(self, message: str):
        """
        When computation will be launched.

        :param message: Message to display to the user
        :type total: str
        """
        self.computationStarted.emit(message)

    @coverage_resolve_trace
    def on_computation_finished(self):
        """
        When all variables have been computed
        """
        self.computationFinished.emit()

    @coverage_resolve_trace
    def on_computation_failed(self, analysis: LandsklimAnalysis):
        self.computationFailed.emit(analysis)


class QThreadAnalysisModels(QThread):
    """
    Thread for building models for an analysis

    :param analysis: Analysis
    :type analysis: LandsklimAnalysis

    """
    # Signal emitted at startup
    computationStarted = pyqtSignal(int)
    # Signal emitted when to display progression
    computationStep = pyqtSignal(str, int)
    # Signal emitted when all models have been built
    computationFinished = pyqtSignal()
    # Signal emitted when an error occurred
    computationFail = pyqtSignal(LandsklimAnalysis, str)

    def __init__(self, analysis: LandsklimAnalysis):
        super().__init__()
        self.__finished: bool = False
        self.__analysis: LandsklimAnalysis = analysis
        self.__analysis.handle_on_models_computation_step(self.on_computation_step)
        self.__analysis.handle_on_models_computation_started(self.on_computation_started)
        self.__analysis.handle_on_models_computation_finished(self.on_computation_finished)
        self.__analysis.handle_on_models_computation_fail(self.on_computation_fail)

    # Merges covered statements from threads inside main code coverage
    @coverage_resolve_trace
    def run(self):
        self.__analysis.construct_datasets()
        if not self.__finished:
            self.__analysis.construct_models()

    @coverage_resolve_trace
    def on_computation_started(self, total: int):
        """
        When computation will be launch.

        :param total: Situation count
        :type total: int
        """
        self.computationStarted.emit(total)

    @coverage_resolve_trace
    def on_computation_step(self, message: str, i: int):
        """
        When a message needs to be displayed

        :param message: Message to be displayed to the user
        :type message: str

        :param i: progression
        :type i: int
        """
        self.computationStep.emit(message, i)

    @coverage_resolve_trace
    def on_computation_finished(self):
        """
        When all situations have been computed
        """
        self.__finished = True
        self.computationFinished.emit()

    @coverage_resolve_trace
    def on_computation_fail(self, analysis: LandsklimAnalysis, invalid_fields: str):
        """
        When an error occurred
        """
        self.computationFail.emit(analysis, invalid_fields)
