from typing import Union

from osgeo import gdal
from osgeo import osr
from osgeo.osr import SpatialReference
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import QgsProcessing, QgsProcessingAlgorithm, QgsProcessingException, QgsProcessingParameterRasterLayer, \
    QgsProcessingParameterNumber, QgsProcessingParameterRasterDestination, QgsRasterLayer, QgsProcessingParameterBoolean
from qgis import processing
import numpy as np

from landsklim.processing.landsklim_processing_regressor_algorithm import LandsklimProcessingRegressorAlgorithm
from landsklim.lk.utils import LandsklimUtils


class RowsProcessingAlgorithm(LandsklimProcessingRegressorAlgorithm):
    """
    Processing algorithm computing altitude average from a DEM
    """

    def __init__(self):
        super().__init__({'OUTPUT': 'Rows raster'})

    def createInstance(self):
        return RowsProcessingAlgorithm()

    def name(self) -> str:
        """
        Unique name of the algorithm
        """
        return 'rows'

    def displayName(self) -> str:
        """
        Displayed name of the algorithm
        """
        return self.tr('Rows')

    def shortHelpString(self) -> str:
        return self.tr('Compute rows raster from DEM')

    def add_dependencies(self):
        """
        No dependencies
        """
        pass

    def processAlgorithm(self, parameters, context, feedback):
        """
        Called when a processing algorithm is run
        """
        # Load input raster and its metadata
        input_raster: QgsRasterLayer = self.parameterAsRasterLayer(parameters, 'INPUT', context)
        no_data, geotransform = self.get_raster_metadata(parameters, context, input_raster)

        # Load other params
        input_window = self.parameterAsInt(parameters, 'INPUT_WINDOW', context)

        out_srs: SpatialReference = self.get_spatial_reference(input_raster)

        # Path of the layer is given. If a temporary layer is selected, layer is created in qgis temp dir
        out_path = self.parameterAsOutputLayer(parameters, 'OUTPUT', context)

        np_input = LandsklimUtils.raster_to_array(input_raster)

        rows: np.ndarray = np.linspace(0, np_input.shape[0] - 1, np_input.shape[0])
        columns_count: int = np_input.shape[1]
        np_output: np.ndarray = np.append(np.zeros((columns_count, 0)), [rows] * columns_count, 1).T

        # Erase every computed data by no data where necessary
        output_no_data = -9999
        np_output[np_input == no_data] = output_no_data

        self.write_raster(out_path, np_output, out_srs, geotransform, output_no_data)

        return {'OUTPUT': out_path}
