import gc
import sys
import time
from math import ceil
from typing import Union, Tuple, List, Dict, Set, FrozenSet, Optional

import pandas as pd
from PyQt5.QtCore import QVariant
from osgeo.osr import SpatialReference

from qgis._core import QgsProcessingParameterDefinition, \
    QgsProcessingParameterVectorLayer, QgsRectangle, QgsVectorLayer, \
    QgsProcessingParameterFeatureSink, QgsProcessingParameterField, QgsGeometry, QgsFeature, \
    QgsCoordinateTransform, QgsProject, QgsPointXY, QgsRaster
from qgis.core import QgsProcessing, QgsProcessingParameterRasterLayer, \
    QgsProcessingParameterNumber, QgsProcessingParameterRasterDestination, QgsRasterLayer, QgsProcessingParameterBoolean

import numpy as np
from scipy import ndimage
from scipy.ndimage import convolve, binary_dilation

from landsklim.lk.utils import LandsklimUtils
from landsklim.lk.logger import Log
from landsklim.processing.landsklim_processing_tool_algorithm import LandsklimProcessingToolAlgorithm

from landsklim.lk import environment


class PolygonsProcessingAlgorithm(LandsklimProcessingToolAlgorithm):
    """
    Processing algorithm computing polygons
    """
    INPUT_MASK = 'INPUT_MASK'
    INPUT_POINTS_SHAPEFILE = 'INPUT_POINTS_SHAPEFILE'
    INPUT_POINTS_FIELDS = 'INPUT_POINTS_FIELDS'
    INPUT_N = 'INPUT_N'
    INPUT_NO_DATA = 'INPUT_NO_DATA'
    INPUT_RETURN_METADATA = 'INPUT_RETURN_METADATA'
    OUTPUT_RASTER = 'OUTPUT_RASTER'
    OUTPUT_POLYGONS_DEFINITION = 'OUTPUT_POLYGONS_DEFINITION'
    OUTPUT_POLYGONS_CONNECTED_SPACE = 'OUTPUT_POLYGONS_CONNECTED_SPACE'
    OUTPUT_POLYGONS_CENTER_OF_GRAVITY = 'OUTPUT_POLYGONS_CENTER_OF_GRAVITY'

    def createInstance(self) -> "PolygonsProcessingAlgorithm":
        """
        Create a PolygonsProcessingAlgorithm instance
        """
        return PolygonsProcessingAlgorithm()

    def name(self) -> str:
        """
        Unique name of the algorithm
        """
        return 'polygons'

    def displayName(self) -> str:
        """
        Displayed name of the algorithm
        """
        return self.tr("Polygons")

    def shortHelpString(self) -> str:
        return self.tr("Compute polygons")

    def parameters(self):
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                self.INPUT_POINTS_SHAPEFILE,
                self.tr('Shapefile'),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        self.addParameter(
            QgsProcessingParameterField(
                self.INPUT_POINTS_FIELDS,
                self.tr('Situations'),
                parentLayerParameterName=self.INPUT_POINTS_SHAPEFILE,
                allowMultiple=True,
                optional=True,
                defaultToAllFields=True
            )
        )

        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.INPUT_MASK,
                self.tr('Mask raster')
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.INPUT_N,
                self.tr('n')
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.INPUT_NO_DATA,
                self.tr('Value to be considered as NO_DATA in the attribute table'),
                optional=True,
                defaultValue=None
            )
        )

        parameter_is_from_processing_toolbox = QgsProcessingParameterBoolean(
                                            self.INPUT_RETURN_METADATA,
                                                   self.INPUT_RETURN_METADATA,
                                                   defaultValue=False
                                                )

        parameter_is_from_processing_toolbox.setFlags(parameter_is_from_processing_toolbox.flags()
                                                       | QgsProcessingParameterDefinition.FlagHidden)
        self.addParameter(
            parameter_is_from_processing_toolbox
        )


        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.OUTPUT_RASTER,
                self.tr('Output raster')
            )
        )

    def initAlgorithm(self, config=None):
        """
        Define inputs and outputs for the main input
        """
        self.parameters()

    def __closest_stations(self, grid_points: np.ndarray, features: np.ndarray, number_of_rows: int, n: int) -> np.ndarray:
        """
        Get the n closest stations of each pixel of grid_points

        :param grid_points: List of pixels
        :type grid_points: np.ndarray

        :param features: Features position
        :type features:

        :param number_of_rows: Number of rows of the input raster
        :type number_of_rows: int

        :param n: Neighborhood size
        :type n: int

        :returns: List of n closests stations
        :rtype: np.ndarray
        """

        """distances_array: Optional[np.ndarray] = np.zeros((len(x_values), points.featureCount()), dtype=np.float32)
        time_total = time.perf_counter()

        for i, feat in enumerate(points.getFeatures()):  # type: QgsFeature
            time_start = time.perf_counter()
            # check point is valid (on the DEM)
            geometry: QgsGeometry = feat.geometry()
            geographic_position: QgsPointXY = transform.transform(geometry.asPoint())

            # Currently, not useful to remove stations on the NO_DATA 
            # data = mask.dataProvider().identify(geographic_position, QgsRaster.IdentifyFormatValue)
            # Remove stations located on the NO_DATA
            # if not data.isValid() or 1 not in data.results() or data.results()[1] is None:
            #     continue

            x_raster_index = int((geographic_position.x() - extent.xMinimum()) / raster_resolution_x) * raster_resolution_x + extent.xMinimum()
            y_raster_index = int((geographic_position.y() - extent.yMinimum()) / raster_resolution_y) * raster_resolution_y + extent.yMinimum()

            station_position_x, station_position_y = x_raster_index, y_raster_index  # geographic_position.x(), geographic_position.y()
            # Distance formula of QgsPointXY is applied on a vectorized way
            diff_x = station_position_x - x_values
            diff_y = station_position_y - y_values
            station_distances: np.ndarray = np.sqrt((diff_x) * (diff_x) + (diff_y) * (diff_y))
            distances_array[:, i] = station_distances
            Log.info("[polygons][feature {0}] {1}s".format(i, time.perf_counter() - time_start))"""

        max_chunk_block_size = 268435456//16
        total_size = features.shape[0] * number_of_rows  # number of stations x number of rows
        chunks_count = ceil(total_size / max_chunk_block_size)
        chunks = np.array_split(grid_points, chunks_count)
        closest_stations_chunks = []
        time_distances_array = 0
        time_closest_stations = 0
        # To avoid creating too large arrays in memory, the array of valid rows is split in several smaller array in order to compute distance matrix and use np.argpartition
        for grid_points_chunk in chunks:
            time_chunk = time.perf_counter()
            Log.info("[polygons][chunk] {0} pixels".format(len(grid_points_chunk)))
            # distances_array is an array containing for each valid row the distance of each station to this row.
            # distances_array is an array of dimensions (batch_rows_count, n)
            # TODO: distances_array is an array of np.float32 to double capacity in relation to a np.float64 array
            distances_array = LandsklimUtils.distance_matrix(features, grid_points_chunk).T
            time_distances_array += time.perf_counter() - time_chunk
            # distances_array returns features as rows (because calculations are way faster)
            # so the matrix is transposed to set features as columns and raster points as rows
            time_start = time.perf_counter()
            closest_stations_chunks.append(np.sort(np.argpartition(distances_array, n, axis=1)[:, :n], axis=-1).astype(np.uint16))
            time_closest_stations += time.perf_counter() - time_start

        closest_stations = np.concatenate(closest_stations_chunks)
        Log.info("[polygons][distance_array] full time : {0}s".format(time_distances_array))
        Log.info("[polygons][closest_stations] {0}s".format(time_closest_stations))
        Log.info("[polygons][closest_stations] Memory usage : {0} bytes".format(closest_stations.nbytes))

        return closest_stations

    def __add_neighbouring_stations_to_polygons(self, polygons: np.ndarray, polygons_connection_space: List[List[int]]) -> List[List[int]]:
        """
        For each polygon, add stations from neighbouring polygons to the list of stations for that polygon.

        :param polygons: Stations associated with each polygon
        :type polygons: np.ndarray

        :param polygons_connection_space: Neighbours of each polygon
        :type polygons_connection_space: List[List[int]]

        :returns: Stations associated with each polygon
        :rtype: List[List[int]]
        """
        polygons_stations_base: List[List[int]] = polygons.tolist()
        polygons_stations: List[List[int]] = polygons.tolist()
        for i in range(len(polygons)):  # type: int
            polygon_connectedness_list: List[int] = polygons_connection_space[i]
            for connected_polygon in polygon_connectedness_list:
                for station in polygons_stations_base[connected_polygon]:
                    if station not in polygons_stations[i]:
                        polygons_stations[i].append(station)
        return polygons_stations


    def polygons(self, points: QgsVectorLayer, mask: QgsRasterLayer, n: int, fields: List[str], no_data_mask: np.ndarray, vector_no_data: Optional[Union[int, float]]) -> Tuple[np.ndarray, List[List[int]], List[List[int]], np.ndarray]:
        """
        Compute neighborhood polygons

        :param points: Stations
        :type points: QgsVectorLayer

        :param mask: Raster mask to compute polygons on
        :type mask: QgsRasterLayer

        :param n: Neighborhood size
        :type n: int

        :param fields: Build polygons considering valid values of these fields
        :type fields: List[str]

        :param no_data_mask: Boolean array (with the same shape of mask) identifying pixels with no_data.
                             Here, used mostly to speed up calculation
        :type no_data_mask: np.ndarray

        :param vector_no_data: NO_DATA value of the attribute table, if defined
        :type vector_no_data: Optional[Union[int, float]]

        :returns: Tuple containing :

            - np.ndarray with the same shape of the raster defining for each pixel its polygon
            - List[List[int]] of shape (*polygons_count*, n) : Definition of each polygon,
                i.e. the stations included on the station's neighborhood
            - List[List[int]] of connectedness : List of each neighbors polygons
            - np.ndarray of center of gravity : List of each polygons' center of gravity
        :rtype: Tuple[np.ndarray, List[List[int]], List[List[int]], np.ndarray]
        """

        base_no_data_mask = no_data_mask.copy()
        no_data_mask = no_data_mask.ravel()
        if n > points.featureCount():
            raise RuntimeError("n is greater than the total number of stations available")
        mask_array = LandsklimUtils.raster_to_array(mask)
        # Raster mapping each row/cell with its polygon
        raster: np.ndarray = np.zeros(mask_array.shape).ravel()
        # Raster mapping each valid row/cell with its polygon.
        # Used to populate 'raster' but calculation are made on 'raster_data_cells' because it's faster to work
        # without invalid rows (rows in the no_data zone)
        raster_data_cells: np.ndarray = np.zeros(mask_array.shape).ravel()[~no_data_mask]
        if not points.hasSpatialIndex():
            points.dataProvider().createSpatialIndex()

        raster_resolution_x: float = mask.rasterUnitsPerPixelX()
        raster_resolution_y: float = mask.rasterUnitsPerPixelY()
        extent: QgsRectangle = mask.extent()

        # Y coordinate of each pixel is located at the bottom of the pixel
        x_values: np.ndarray = np.linspace(extent.xMinimum(), extent.xMaximum()-raster_resolution_x, mask_array.shape[1])  # TODO: Geographic coordinates ?
        y_values: np.ndarray = np.linspace(extent.yMaximum()-raster_resolution_y, extent.yMinimum(), mask_array.shape[0])  # TODO: Geographic coordinates ?
        variables: np.ndarray = np.array(np.meshgrid(y_values, x_values)).T.reshape(-1, 2)
        variables[:, [0, 1]] = variables[:, [1, 0]]
        # Center coordinates in the middle of each pixel
        # variables[:, 0] = variables[:, 0] + (raster_resolution_x / 2.)
        # variables[:, 1] = variables[:, 1] + (raster_resolution_y / 2.)
        # Keep only cells with valid data
        x_values = variables[:, 0][~no_data_mask]
        y_values = variables[:, 1][~no_data_mask]
        transform = QgsCoordinateTransform(points.crs(), mask.crs(), QgsProject().instance())

        time_total = time.perf_counter()
        # Create features array (couples of (x, y) ) from QgsVectorLayer
        # features = np.zeros((points.featureCount(), 2))
        non_null_features: List[int] = self.get_non_null_features(points, fields, vector_no_data)
        features = []
        valid_features = []
        for i, feat in enumerate(points.getFeatures()):  # type: QgsFeature
            # check point is valid (on the DEM)
            geometry: QgsGeometry = feat.geometry()
            geographic_position: QgsPointXY = transform.transform(geometry.asPoint())
            # Remove stations located on the NO_DATA
            data = mask.dataProvider().identify(geographic_position, QgsRaster.IdentifyFormatValue)
            if not data.isValid() or 1 not in data.results() or data.results()[1] is None:
                continue
            if i not in non_null_features:
                continue

            """features[i, 0] = int((geographic_position.x() - extent.xMinimum()) / raster_resolution_x) * raster_resolution_x + extent.xMinimum()
            features[i, 1] = int((geographic_position.y() - extent.yMinimum()) / raster_resolution_y) * raster_resolution_y + extent.yMinimum()"""
            feature_pos_x = int((geographic_position.x() - extent.xMinimum()) / raster_resolution_x) * raster_resolution_x + extent.xMinimum()
            feature_pos_y = int((geographic_position.y() - extent.yMinimum()) / raster_resolution_y) * raster_resolution_y + extent.yMinimum()
            features.append([feature_pos_x, feature_pos_y])
            valid_features.append(i)  # Keep trace of indices of valid features
        features = np.array(features)
        valid_features = np.array(valid_features)

        # array of grid points : array of dimensions (valid_rows_count, 2)
        grid_points = np.hstack((x_values.reshape(-1, 1), y_values.reshape(-1, 1)))

        closest_stations = self.__closest_stations(grid_points, features, x_values.shape[0], n)

        time_start = time.perf_counter()
        # Polygons definitions
        # pandas' drop_duplicates is way much faster than np.unique
        polygons: np.ndarray = pd.DataFrame(closest_stations).drop_duplicates().to_numpy()  # [~no_data_mask]
        # polygons: np.ndarray = np.unique(closest_stations[~no_data_mask], axis=0)

        # Solution 1 : Very slow with big rasters
        # Expend closest_stations on a new axis (closest_stations[:, None])
        # So we compare matching values for each closest_stations to all polygons (polygons == closest_stations[...]),
        # get rows where all values matches (.all(-1)),
        # search indices of matching values (np.where()),
        # Get indices of matching polygons for each row (np.where()[1])
        # closest_stations[no_data_mask, :] = -1
        # raster = np.where((polygons == closest_stations[:, None]).all(axis=-1))[1]
        # invalid_polygon: np.ndarray = np.full((1, closest_stations.shape[1]), -1)
        # mask_invalid_polygons: np.ndarray = ~(polygons == invalid_polygon).all(axis=1)
        # polygons = polygons[mask_invalid_polygons]
        # https://stackoverflow.com/questions/38674027/find-the-row-indexes-of-several-values-in-a-numpy-array
        # perf_counter : 169s

        # Associate polygon neighborhood with its index (faster than list.index but not optimal)
        """polygons_dict: Dict[Tuple, int] = dict(zip([tuple(k) for k in polygons], range(0, len(polygons))))
        for i, stations_combination in enumerate(closest_stations):
            raster_data_cells[i] = polygons_dict[tuple(stations_combination)]
        raster[~no_data_mask] = raster_data_cells"""
        # perf_counter : 11s

        # Solution 3 : Through pandas join (best for now)
        polygons_pd = pd.DataFrame(polygons, columns=list(np.arange(n)))
        closest_stations_pd = pd.DataFrame(closest_stations, columns=list(np.arange(n)))
        left_on = list(polygons_pd.columns)
        right_on = list(closest_stations_pd.columns)
        polygons_pd['polygon_number'] = np.arange(len(polygons_pd))
        new_df = pd.merge(closest_stations_pd, polygons_pd, how='left', left_on=left_on, right_on=right_on)
        raster[~no_data_mask] = new_df['polygon_number']

        Log.info("[polygons][neighborhood] {0:.2f}s".format(time.perf_counter() - time_start))
        polygons_raster: np.ndarray = raster.reshape(mask_array.shape)
        polygons_raster[base_no_data_mask] = np.nan
        # Create a dataframe with only one column (polygons) and get list of indices of each polygons
        indices_by_polygon_ravel: Dict[int, List[int]] = pd.DataFrame(polygons_raster.ravel()[~no_data_mask].astype(np.uint32), columns=['polygon']).groupby(by='polygon').groups
        # Compute ravel index offset from polygons_raster instead of making two groups by
        indices_by_polygon_raster: Dict[int, List[int]] = pd.DataFrame(polygons_raster.ravel(), columns=['polygon']).groupby(by='polygon').groups

        time_start = time.perf_counter()
        polygons_centroid: List[np.ndarray] = []
        for i in range(len(polygons)):
            polygons_indices = np.hstack(np.unravel_index(indices_by_polygon_raster[i].to_numpy().reshape(-1, 1), polygons_raster.shape))
            # polygons_indices is (row, col) indexed :
            # row_min and row_max are the first and the last entries
            # col_min and col_max can be inserted at the middle of the table so min() and max() are called
            rmin, rmax = polygons_indices[:, 0][[0, -1]]
            cmin, cmax = polygons_indices[:, 1].min(), polygons_indices[:, 1].max()

            # centroids with [col, line] format
            polygons_centroid.append(np.array([((cmax-cmin)/2) + cmin, ((rmax-rmin)/2) + rmin]))

        Log.info("[polygons][centroids] {0:.2f}s".format(time.perf_counter() - time_start))

        time_start = time.perf_counter()
        polygons_connection_space: List[List[int]] = []

        # for each pixel : [row, col, polygon_rc, polygon_r-1, polygon_r+1, polygon_c-1, polygon_c+1]
        points = polygons_raster.shape[0] * polygons_raster.shape[1]
        connection_space_array = np.zeros((points, 7), dtype=np.uint32)
        # As we work with uint32 array, np.nan is represented as 0. To avoid confusion, polygons starts at 1.
        polygons_raster_start_1 = polygons_raster.ravel()[~no_data_mask] + 1
        polygons_raster_1 = polygons_raster.copy() + 1
        polygons_raster_1[np.isnan(polygons_raster_1)] = 0
        Log.info("[connection_space_array] Memory usage : {0} bytes".format(connection_space_array.nbytes))
        connection_space_array[:, 0] = np.repeat(np.arange(polygons_raster.shape[0]), polygons_raster.shape[1])
        connection_space_array[:, 1] = np.tile(np.arange(polygons_raster.shape[1]), polygons_raster.shape[0])
        connection_space_array = connection_space_array[~no_data_mask]
        connection_space_array[:, 2] = polygons_raster_start_1.astype(np.uint32)

        # row - 1
        mask_arg_1 = connection_space_array[:, 0] != 0
        connection_space_array[mask_arg_1, 3] = polygons_raster_1[connection_space_array[mask_arg_1, 0] - 1, connection_space_array[mask_arg_1, 1]]

        # row + 1
        mask_arg_2 = connection_space_array[:, 0] < polygons_raster.shape[0] - 1
        connection_space_array[mask_arg_2, 4] = polygons_raster_1[connection_space_array[mask_arg_2, 0] + 1, connection_space_array[mask_arg_2, 1]]

        # col - 1
        mask_arg_3 = connection_space_array[:, 1] != 0
        connection_space_array[mask_arg_3, 5] = polygons_raster_1[connection_space_array[mask_arg_3, 0], connection_space_array[mask_arg_3, 1] - 1]

        # col + 1
        mask_arg_4 = connection_space_array[:, 1] < polygons_raster.shape[1] - 1
        connection_space_array[mask_arg_4, 6] = polygons_raster_1[connection_space_array[mask_arg_4, 0], connection_space_array[mask_arg_4, 1] + 1]

        for i in range(len(polygons)):  # type: int
            polygon_indices = indices_by_polygon_ravel[i]
            neighbors = connection_space_array[polygon_indices, 3:]
            neighbors = np.unique(neighbors)
            neighbors = neighbors[np.nonzero(neighbors)]  # remove 0 aka np.nan
            # In connection_space_array, polygons starts at 1 !
            neighbors = neighbors - 1
            neighbors = neighbors[neighbors != i]
            polygons_connection_space.append(neighbors.tolist())

        Log.info("[polygons][very fast connection space] {0:.2f}s".format(time.perf_counter() - time_start))

        time_stations = time.perf_counter()
        polygons_stations: List[List[int]] = self.__add_neighbouring_stations_to_polygons(polygons, polygons_connection_space)
        Log.info("[polygons][stations] {0:.2f}s".format(time.perf_counter() - time_stations))

        # polygons_stations : Stations are indexed [0, n]
        # polygons_stations_projected : Stations indices are reprojected to match features indices (on the shapefile)
        polygons_stations_projected = []
        for polygon_station in polygons_stations:
            polygons_stations_projected.append(list(valid_features[list(polygon_station)]))
        polygons_stations = polygons_stations_projected

        return polygons_raster, polygons_stations, polygons_connection_space, np.array(polygons_centroid)

    def get_non_null_features(self, layer: QgsVectorLayer, fields: List[str], no_data: Optional[Union[int, float]]) -> List[int]:
        """
        Filter layer features

        :param layer: Input layer
        :type layer: QgsVectorLayer

        :param fields: Fields of interest
        :type fields: List[str]

        :param no_data: Value to consider as NO_DATA
        :type no_data: Optional[Union[int, float]]

        :returns: Indices of features where at least one valid value exists on the requested fields
        :rtype: List[int]
        """
        fields_names = [field.name() for field in layer.fields()]
        fields_idx = [fields_names.index(field) for field in fields]
        valid_features: List[int] = []
        for i, feat in enumerate(layer.getFeatures()):  # type: QgsFeature
            attributes = feat.attributes()
            is_valid = False
            for fidx in fields_idx:
                fvalue = attributes[fidx]
                is_valid = is_valid or (isinstance(fvalue, (int, float)) and fvalue != no_data) or (isinstance(fvalue, QVariant) and not fvalue.isNull())
            if is_valid:
                valid_features.append(i)
        return valid_features


    def processAlgorithm(self, parameters, context, feedback):
        """
        Called when a processing algorithm is run
        """
        input_points: QgsVectorLayer = self.parameterAsVectorLayer(parameters, self.INPUT_POINTS_SHAPEFILE, context)

        qgs_version_major, qgs_version_minor = self.qgis_version()
        if qgs_version_major == 3 and qgs_version_minor >= 32:
            input_fields: List[str] = self.parameterAsStrings(parameters, self.INPUT_POINTS_FIELDS, context)
        else:
            input_fields: List[str] = self.parameterAsFields(parameters, self.INPUT_POINTS_FIELDS, context)

        input_n: int = self.parameterAsInt(parameters, self.INPUT_N, context)
        input_mask: QgsRasterLayer = self.parameterAsRasterLayer(parameters, self.INPUT_MASK, context)
        vector_no_data: Optional[Union[float, int]] = parameters[self.INPUT_NO_DATA]

        return_metadata: bool = self.parameterAsBoolean(parameters, self.INPUT_RETURN_METADATA, context)
        np_mask: np.ndarray = LandsklimUtils.raster_to_array(input_mask)
        no_data, geotransform = self.get_raster_metadata(parameters, context, input_mask)
        no_data_mask = (np_mask == no_data) | np.isnan(np_mask)

        out_srs: SpatialReference = self.get_spatial_reference(input_mask)

        out_path = self.parameterAsOutputLayer(parameters, self.OUTPUT_RASTER, context)

        # polygons_definition: np.ndarray of dims (polygons_count, n)
        raster_output, polygons_definition, polygons_connected_space, polygons_center_of_gravity = self.polygons(input_points, input_mask, input_n, input_fields, no_data_mask, vector_no_data)
        output_no_data = -1
        raster_output[no_data_mask] = output_no_data

        self.write_raster(out_path, raster_output, out_srs, geotransform, output_no_data)

        # Processing output make QGIS crash if polygons definition are too big
        # So, when algorithm_polygons is launched from Processing Toolbox, only the raster is returned
        if return_metadata:
            res_dict = {self.OUTPUT_RASTER: out_path, self.OUTPUT_POLYGONS_DEFINITION: polygons_definition, self.OUTPUT_POLYGONS_CONNECTED_SPACE: polygons_connected_space, self.OUTPUT_POLYGONS_CENTER_OF_GRAVITY: polygons_center_of_gravity}
        else:
            res_dict = {self.OUTPUT_RASTER: out_path}

        return res_dict

