from typing import List, Optional, Dict, Type
from dataclasses import dataclass

from landsklim.lk.regressor import  Regressor
from landsklim.lk.regressor_collection import RegressorCollection
from landsklim.lk.regressor_radiation import RegressorRadiation
from landsklim.lk.regressor_slope import RegressorSlope
from landsklim.lk.regressor_altitude import RegressorAltitude
from landsklim.lk.regressor_orientation import RegressorOrientation
from landsklim.lk.regressor_sine import RegressorSine
from landsklim.lk.regressor_cosine import RegressorCosine
from landsklim.lk.regressor_roughness import RegressorRoughness
from landsklim.lk.regressor_encasement import RegressorEncasement
from landsklim.lk.regressor_amplc import RegressorAmplitudeThalweg
from landsklim.lk.regressor_amplb import RegressorAmplitudeCrests
from landsklim.lk.regressor_cols import RegressorCols
from landsklim.lk.regressor_rows import RegressorRows
from landsklim.lk.regressor_raster_variable import RegressorRasterVariable
from landsklim.lk.map_layer import RasterLayer


@dataclass
class RegressorDefinition:
    regressor_name: str
    windows: List[int]
    polynomial_degree: int

    def __str__(self) -> str:
        return "<{0}> {1} - Windows : {2} - Polynomial degree : {3}".format(RegressorDefinition.__name__, self.regressor_name, self.windows, self.polynomial_degree)

class RegressorFactory:
    __registered_classes: List[Type[Regressor]] = [RegressorRows,
                                                   RegressorCols,
                                                   RegressorAltitude,
                                                   RegressorSlope,
                                                   RegressorOrientation,
                                                   RegressorRoughness,
                                                   RegressorEncasement,
                                                   RegressorSine,
                                                   RegressorCosine,
                                                   RegressorRadiation,
                                                   RegressorAmplitudeCrests,
                                                   RegressorAmplitudeThalweg]


    regressor_dependencies: Dict[str, List[str]] = {
        RegressorOrientation.class_name(): [RegressorSlope.class_name()],
        RegressorSine.class_name(): [RegressorOrientation.class_name()],
        RegressorCosine.class_name(): [RegressorOrientation.class_name()],
        RegressorRadiation.class_name(): [RegressorOrientation.class_name(), RegressorSlope.class_name()]
        # Keep RegressorSlope at end because Orientation depends on Slope
    }

    @staticmethod
    def default_prefixes() -> List[str]:
        """
        Get list of prefixes used by default regressors (far-fetched way)
        """
        return [RegressorFactory.get_regressor(reg_type.class_name(), 0, 0).prefix() for reg_type in RegressorFactory.registered_classes()]

    @staticmethod
    def registered_classes() -> List[Type[Regressor]]:
        return RegressorFactory.__registered_classes.copy()

    @staticmethod
    def get_regressor_class(name: str) -> Type[Regressor]:
        reg: Optional[Type[Regressor]] = None
        for regressor in RegressorFactory.registered_classes():
            if regressor.class_name() == name:
                reg = regressor
        if reg is None:
            reg = RegressorRasterVariable
        return reg

    @staticmethod
    def get_regressor(regressor_name: str, windows: int, polynomial_degree: int) -> Regressor:

        regressor: Optional[Regressor] = None
        for regressor_class in RegressorFactory.registered_classes():
            if regressor_name == regressor_class.class_name():
                regressor = regressor_class(windows, polynomial_degree)

        # If regressor is None, regressor is found from source variables, other than DEM
        if regressor is None:
            from landsklim.landsklim import Landsklim
            for variable in Landsklim.instance().get_landsklim_project().get_variables():  # type: RasterLayer
                # Can't change the name of the layer between opening and closing analysis dialog
                if variable.qgis_layer().name() == regressor_name:
                    regressor = RegressorRasterVariable(windows, polynomial_degree, variable)

        return regressor

    @staticmethod
    def get_regressors_from_definitions(regressor_definitions: List[RegressorDefinition]) -> RegressorCollection:
        """
        Convert definitions of regressors to a list of regressors

        :param regressor_definitions: Regressors definition to convert
        :type regressor_definitions: List[RegressorDefinition]

        :param resolve_dependencies: If True, resolve dependencies between regressors
        :type resolve_dependencies: bool

        :returns: List of created regressors
        :rtype: RegressorCollection
        """
        collection: RegressorCollection = RegressorCollection()
        for regressor_definition in regressor_definitions:  # type: RegressorDefinition
            for regressor_window in regressor_definition.windows:
                regressor = RegressorFactory.get_regressor(regressor_definition.regressor_name, regressor_window, regressor_definition.polynomial_degree)
                RegressorFactory.__register_regressor(regressor, collection)
        return collection

    @staticmethod
    def __register_regressor(regressor: Regressor, regressors: RegressorCollection, as_first: bool = False, is_dependency: bool = False):
        """
        Register a regressor on a list of regressors.
        Dependencies are resolved

        :param regressor: Regressor to register
        :type regressor: Regressor

        :param as_first: Add the regressor to the beginning of the list
        :type as_first: bool

        :param regressors: List of regressors as a RegressorCollection
        :type regressors: RegressorCollection
        """

        if not RegressorFactory.__regressor_exists(regressor, regressors):
            regressors.add_regressor(regressor, as_first, is_dependency)
        elif as_first:
            existing_regressor = [r for r in regressors.get_regressors() if r.equals(regressor)][0]
            regressors.move_to_top(existing_regressor)

        # Dependencies resolution and add them at the beginning of the list
        if regressor.class_name() in RegressorFactory.regressor_dependencies:
            for dependency in RegressorFactory.regressor_dependencies[regressor.class_name()]:
                new_regressor = RegressorFactory.get_regressor(dependency, regressor.get_windows(), regressor.get_polynomial_degree())
                RegressorFactory.__register_regressor(new_regressor, regressors, as_first=True, is_dependency=True)

                """if not RegressorFactory.__regressor_exists(new_regressor, regressors):
                    RegressorFactory.__register_regressor(new_regressor, regressors, as_first=True, is_dependency=True)
                else:
                    existing_regressor = [r for r in regressors.get_regressors() if r.equals(new_regressor)][0]
                    regressors.move_to_top(existing_regressor)"""
    @staticmethod
    def __regressor_exists(regressor: Regressor, regressors: RegressorCollection) -> bool:
        """
        :returns: True if the regressor is already registered on the project, else False
        :rtype: bool
        """
        exists: bool = False
        for r in regressors.get_regressors():  # type: Regressor
            if r.equals(regressor):
                exists = True
        return exists