import json
import os
import time
from typing import List, Tuple, Optional, Callable, Type, Dict

from PyQt5.QtCore import QDate
from qgis.core import QgsRasterLayer, QgsProject, QgsExpressionContextUtils

from landsklim.lk.map_layer import MapLayerCollection, VectorLayer, RasterLayer
from landsklim.lk.landsklim_configuration import LandsklimConfiguration

import _pickle as cPickle

from landsklim.lk.regressor import Regressor
from landsklim.lk.regressor_collection import RegressorCollection
from landsklim.lk.regressor_factory import RegressorDefinition, RegressorFactory
from landsklim.lk.utils import LandsklimUtils


class LandsklimProject:
    """
    Represents a Landsklim project containing all data needed for making spatial interpolations,
    associated with a QGIS project

    :param stations: List of samples to use as stations for interpolation.
        Each shapefile represents a list of points as stations.
    :type stations: MapLayerCollection[VectorLayer]

    :param dem: Default digital elevation model. Represents the altitude.
        Predictive variables will be extracted from this DEM.
    :type dem: RasterLayer

    :param variables: Additional variables
    :type variables: MapLayerCollection[RasterLayer]

    :param date: Date used for solar irradiance
    :type date: QDate
    """

    def __init__(self, name: str = "", stations: Optional[MapLayerCollection] = None, dem: RasterLayer = None,
                 variables: Optional[MapLayerCollection] = None, regressors: Optional[List[RegressorDefinition]] = None, date: Optional[QDate] = None, stations_no_data: Optional[float] = None):
        self._name: str = name
        self._configurations: List[LandsklimConfiguration] = []
        self._stations: MapLayerCollection = stations if stations is not None else MapLayerCollection()
        self._dem: RasterLayer = dem
        self._variables: MapLayerCollection = variables if variables is not None else MapLayerCollection()
        self._regressors: List[Regressor] = RegressorFactory.get_regressors_from_definitions(regressors).get_regressors() if regressors is not None else []
        self._additional_layers: MapLayerCollection = MapLayerCollection()
        self._date: Tuple[int, int, int] = (date.year(), date.month(), date.day()) if date is not None else (0, 0, 0)

        self._on_explicative_variables_compute_started: Optional[Callable[[int], None]] = None
        self._on_regressor_compute_finished: Optional[Callable[[Regressor, int, int, int], None]] = None
        self._on_explicative_variables_compute_finished: Optional[Callable[[], None]] = None

        self._stations_no_data: Optional[float] = stations_no_data

        self.__landsklim_version: str = LandsklimUtils.landsklim_version()

    def __getstate__(self):
        dict = self.__dict__.copy()
        dict.pop("_on_explicative_variables_compute_started", None)
        dict.pop("_on_regressor_compute_finished", None)
        dict.pop("_on_explicative_variables_compute_finished", None)
        return dict
        # return (self._name, self._configurations, self._stations, self._dem, self._variables, self._regressors, self._additional_layers)

    def __setstate__(self, state):
        for k, v in state.items():
            setattr(self, LandsklimUtils.rename_attr("LISDQSProject", "LandsklimProject", k), v)
        # (self._name, self._configurations, self._stations, self._dem, self._variables, self._regressors, self._additional_layers) = state

    def set_landsklim_version(self, version: str):
        self.__landsklim_version = version

    def get_landsklim_version(self) -> str:
        return self.__landsklim_version if hasattr(self, "_LandsklimProject__landsklim_version") else None

    def get_stations_no_data(self) -> Optional[float]:
        return self._stations_no_data

    def handle_on_explicative_variables_compute_started(self, handle_function: Callable[[int], None]):
        self._on_explicative_variables_compute_started = handle_function

    def handle_on_regressor_compute_finished(self, handle_function: Callable[[Regressor, int, int, int], None]):
        self._on_regressor_compute_finished = handle_function

    def handle_on_explicative_variables_compute_finished(self, handle_function: Callable[[], None]):
        self._on_explicative_variables_compute_finished = handle_function

    def add_configuration(self, configuration: LandsklimConfiguration):
        self._configurations.append(configuration)

    def get_regressors(self) -> List[Regressor]:
        return sorted(self._regressors.copy(), key=lambda r: "{0}_{1:08d}".format(r.prefix(), r.get_windows()))

    def get_raster_variables(self) -> MapLayerCollection:
        rasters: MapLayerCollection = MapLayerCollection()
        for regressor in self._regressors:  # type: Regressor
            if regressor.get_raster_layer() is not None:
                rasters.append(regressor.get_raster_layer())
        return rasters

    def __regressor_exists(self, regressor: Regressor) -> Regressor:
        """
        :returns: The regressor if the regressor is already registered on the project, else None
        :rtype: Regressor
        """
        exists: Optional[Regressor] = None
        for r in self._regressors:  # type: Regressor
            if r.equals(regressor):
                exists = r
        return exists

    def register_regressors(self, regressors_to_register: RegressorCollection) -> RegressorCollection:
        """
        Register regressors inside a Landsklim project from their definition

        :param regressors_to_register: List of regressors to register
        :type regressors_to_register: List[Regressor]

        :returns: List of concrete regressors
        :rtype: List[Regressor]
        """
        project_regressors: RegressorCollection = RegressorCollection()

        new_regressors: RegressorCollection = regressors_to_register
        for new_regressor in new_regressors.get_regressors():
            existing_regressor: Optional[Regressor] = self.__regressor_exists(new_regressor)
            if not existing_regressor:
                self._regressors.append(new_regressor)
                existing_regressor = new_regressor
            project_regressors.add_regressor(existing_regressor, False, not regressors_to_register.is_first_level_regressor(existing_regressor))

        # Once regressors were added, sort them to get pretty analysis datasets
        project_regressors.sort()
        return project_regressors

    def create_explicative_variables(self):
        """
        Create explicative variables defined on self._regressors
        """
        regressors = self._regressors
        total = len(regressors)
        i = 0
        if self._on_explicative_variables_compute_started is not None:
            self._on_explicative_variables_compute_started(total)
        for regressor in regressors:
            if self._on_regressor_compute_finished is not None:
                self._on_regressor_compute_finished(regressor, regressor.get_windows(), i+1, total)
            # Get source raster according to regressor "policy". Default regressors uses DEM, but RegressorRasterVariable use source variables
            source_raster: QgsRasterLayer = self.get_dem().qgis_layer() if regressor.specific_source_raster() is None else regressor.specific_source_raster().qgis_layer()
            regressor.compute(source_raster)
            i = i + 1

        if self._on_explicative_variables_compute_finished is not None:
            self._on_explicative_variables_compute_finished()

    def get_additional_layers(self) -> MapLayerCollection:
        """
        .. deprecated:: 0.4.4
            Not used anymore
        """
        return self._additional_layers

    def add_additional_layer(self, intermediate_layer: RasterLayer):
        """
        .. deprecated:: 0.4.4
            Not used anymore
        """
        self._additional_layers.append(intermediate_layer)

    def save(self) -> Tuple[bool, bool, str]:
        """
        Serialize project into disk

        :returns:

            - First boolean : True is Landsklim project was correctly saved, False otherwise,
            - Second boolean : True is Landsklim project was cloned, False otherwise
            - String : and string containing error cause
        :rtype: Tuple[bool, bool, str]
        """
        error_cause: str = ""
        was_saved: bool = False
        was_cloned: bool = False

        project_instance = QgsProject.instance()
        # file_name_pkl = "{0}.pkl".format(self._name)
        file_name_json = "{0}.json".format(self._name)
        # qgis_project_path_pkl = os.path.join(project_instance.absolutePath(), 'Landsklim', file_name_pkl)
        qgis_project_path_json = os.path.join(project_instance.absolutePath(), 'Landsklim', file_name_json)
        # existing_lk_folder_path = os.path.join(os.path.dirname(project_instance.customVariables()["landsklim_project"])) if "landsklim_project" in project_instance.customVariables() else ""
        target_lk_folder = os.path.join(os.path.dirname(qgis_project_path_json))
        path_is_valid = False
        lk_core_folders = ["interpolations", "regressors"]

        print("[qgis_project_path]", qgis_project_path_json)

        # Si le projet a déjà été enregistré
        if "landsklim_project" in project_instance.customVariables():
            folder_names = [name for name in os.listdir(target_lk_folder) if
                            os.path.isdir(os.path.join(target_lk_folder, name))]
            path_is_valid = os.path.exists(target_lk_folder)

            """# S'il manque un des deux dossiers "interpolations" et "regressors", alors le projet a été corrompu :
            # - soit le projet QGIS a été déplacé en oubliant le dossier Landsklim
            # - soit des fichiers internes on été supprimés/déplacés
            for folder_name in lk_core_folders:
                if folder_name not in folder_names:
                    path_is_valid = False"""

            if not path_is_valid:
                error_cause = "ERROR_SRC_LANDSKLIM_FOLDER_NOT_VALID"

        # Le projet Landsklim peut être sauvegardé si :
        # - aucun projet Landsklim n'est associé au projet QGIS
        # - Le projet QGIS a été déplacé et le nouveau chemin est valide
        allow_save = "landsklim_project" not in project_instance.customVariables() or path_is_valid

        if allow_save:
            was_saved = True

            # In case of Landsklim version migration, save/re-save each dependencies
            if LandsklimUtils.landsklim_version() != self.get_landsklim_version():
                self.__save_json_dependencies()

            self.set_landsklim_version(LandsklimUtils.landsklim_version())
            if not os.path.isdir(os.path.join(project_instance.absolutePath(), 'Landsklim')):
                os.mkdir(os.path.join(project_instance.absolutePath(), 'Landsklim'))
            """with open(qgis_project_path_pkl, "wb") as output_file:
                time_start = time.perf_counter()
                cPickle.dump(self, output_file)
                print("[Pickle][Encoder] {0:.4f}s".format(time.perf_counter() - time_start))"""
            with open(qgis_project_path_json, "w") as output_file:
                from landsklim.serialization.json_encoder import LandsklimEncoder
                time_start = time.perf_counter()
                json.dump(self, fp=output_file, cls=LandsklimEncoder, indent=2)
                print("[Landsklim][JSON][Encoder] {0:.4f}s".format(time.perf_counter() - time_start))
            """with open(qgis_project_path_json, "rb") as input_file:
                from landsklim.serialization.json_decoder import LandsklimDecoder
                time_start = time.perf_counter()
                landsklim_project = json.load(fp=input_file, cls=LandsklimDecoder)
                print("[JSON][Decoder] {0:.4f}s".format(time.perf_counter() - time_start))"""

            QgsExpressionContextUtils.setProjectVariable(project_instance, "landsklim_project", file_name_json)

        return was_saved, was_cloned, error_cause

    """def save_with_clone(self) -> Tuple[bool, bool, str]:
        
        # Serialize project into disk
        # 
        # .. deprecated:: 0.4.0
        #     User is responsible to move Landsklim folder with the QGIS project when moving a QGIS project.
        # 
        # :returns:
        # 
        #     - First boolean : True is Landsklim project was correctly saved, False otherwise,
        #     - Second boolean : True is Landsklim project was cloned, False otherwise
        #     - String : and string containing error cause
        # :rtype: Tuple[bool, bool, str]
        
        error_cause: str = ""
        was_saved: bool = False
        clone_to_new_path: bool = False

        project_instance = QgsProject.instance()
        file_name = "{0}.pkl".format(self._name)
        file_name_json = "{0}.json".format(self._name)
        project_path = os.path.join(project_instance.absolutePath(), 'Landsklim', file_name)
        project_path_json = os.path.join(project_instance.absolutePath(), 'Landsklim', file_name_json)
        source_lk_folder = os.path.join(os.path.dirname(project_instance.customVariables()["landsklim_project"])) if "landsklim_project" in project_instance.customVariables() else ""
        target_lk_folder = os.path.join(os.path.dirname(project_path))
        new_path_is_valid = False
        lk_core_folders = ["interpolations", "regressors"]  # ["Zone_Calculs", "Zone_Fichiers"]

        if "landsklim_project" in project_instance.customVariables():
            folder_names = [name for name in os.listdir(source_lk_folder) if os.path.isdir(os.path.join(source_lk_folder, name))]
            new_path_is_valid = os.path.exists(source_lk_folder)

            for folder_name in folder_names:
                if folder_name not in lk_core_folders:
                    new_path_is_valid = False
            if new_path_is_valid:
                if os.path.exists(target_lk_folder):
                    error_cause = "ERROR_DEST_FOLDER_ALREADY_HAVE_LANDSKLIM_FOLDER"
                    new_path_is_valid = False
                else:
                    clone_to_new_path = True
            else:
                error_cause = "ERROR_SRC_LANDSKLIM_FOLDER_NOT_VALID"

        allow_save = "landsklim_project" not in project_instance.customVariables() or os.path.normpath(source_lk_folder) == os.path.normpath(target_lk_folder) or new_path_is_valid

        if allow_save:
            was_saved = True
            if not os.path.isdir(os.path.join(project_instance.absolutePath(), 'Landsklim')):
                os.mkdir(os.path.join(project_instance.absolutePath(), 'Landsklim'))
            with open(project_path, "wb") as output_file:
                cPickle.dump(self, output_file)
            with open(project_path_json, "w") as output_file:
                from landsklim.serialization.json_encoder import LandsklimEncoder
                # json.dump(self, fp=output_file, cls=LandsklimEncoder, indent=2)
            QgsExpressionContextUtils.setProjectVariable(project_instance, "landsklim_project", project_path)

        if clone_to_new_path:
            for folder in lk_core_folders:
                source_core_lk_folder = os.path.join(source_lk_folder, folder)
                target_core_lk_folder = os.path.join(target_lk_folder, folder)
                if os.path.exists(source_core_lk_folder):
                    shutil.copytree(source_core_lk_folder, target_core_lk_folder)

        return was_saved, clone_to_new_path, error_cause"""

    def __save_json_dependencies(self):
        for analysis in self.get_analysis():  # type: "LandsklimAnalysis"
            analysis.serialize_components()

    def to_string(self) -> str:
        return self._name

    def get_configurations(self) -> List[LandsklimConfiguration]:
        return self._configurations

    def get_analysis(self) -> List["LandsklimAnalysis"]:
        analysis = []
        for configuration in self._configurations:
            analysis = analysis + configuration.get_analysis()
        return analysis

    def get_interpolations(self):
        interpolations = []
        for analysis in self.get_analysis():
            interpolations = interpolations + analysis.get_interpolations()
        return interpolations

    def get_stations_sources(self) -> MapLayerCollection:
        return self._stations

    def get_variables(self) -> MapLayerCollection:
        return self._variables

    def available_regressors(self) -> List[str]:
        """
        Get list of available regressors in the project, including default regressors and additional variables

        :returns: Available regressors
        :rtype: List[str]
        """
        regressors = []
        for base_regressor in RegressorFactory.registered_classes():  # type: Type[Regressor]
            regressors.append(base_regressor.class_name())
        for variable in self.get_variables():  # type: RasterLayer
            regressors.append(variable.qgis_layer().name())
        return regressors

    def get_dem(self) -> RasterLayer:
        return self._dem

    def get_date(self) -> Tuple[int, int, int]:
        """
        :returns: Date (for solar irradiance) on a (year, month, day) format
        :rtype: Tuple[int, int, int]
        """
        return self._date

    def update_project_definition(self, name: str, stations: MapLayerCollection, dem: RasterLayer, variables: MapLayerCollection, date: QDate, stations_no_data: Optional[float]):
        """
        Update project definition (sources and name). Accessed through Project Dialog.

        :param name: Project name
        :type name: str

        :param stations: List of samples to use as stations for interpolation.
            Each shapefile represents a list of points as stations.
        :type stations: MapLayerCollection[VectorLayer]

        :param dem: Default digital elevation model. Represents the altitude.
            Predictive variables will be extracted from this DEM.
        :type dem: RasterLayer

        :param variables: Additionnal variables
        :type variables: MapLayerCollection[RasterLayer]

        :param date: Date of the project (used for sun irradiance)
        :type date: QDate
        """
        self._name = name
        self._stations = stations
        self._dem = dem
        self._variables = variables
        self._date = (date.year(), date.month(), date.day())
        self._stations_no_data = stations_no_data


    def display(self):
        print("=== Project {0} ===".format(self.to_string()))
        print("Configurations : {0}, Analysis : {1}, Interpolations : {2}".format(len(self.get_configurations()), len(self.get_analysis()), len(self.get_interpolations())))
        print("Stations : {0}, DEM : {1}, Variables : {2}".format(self.get_stations_sources(), self._dem, self._variables))
        print("=====================")

    def to_json(self) -> Dict:
        state: Dict = self.__getstate__()
        return state
