from typing import Any, Tuple
import _pickle as cPickle

import numpy as np

from landsklim.lk.landsklim_analysis import LandsklimAnalysis
from landsklim.lk.landsklim_project import LandsklimProject
from landsklim.lk.phase_composite import PhaseComposite
from landsklim.lk.polygons_definition import PolygonsDefinition


class PearsonRResultDummyClass:
    """
    In Landsklim 0.5.1, LISDQSAnalysis.self._pearson_correlation[...][...] can be pickled as PearsonRResult.
    Following version of Scipy, PearsonRResult is sometimes a simple tuple.
    From Landsklim 0.6.0, LISDQSAnalysis.self._pearson_correlation[...][...] is always serialized as tuple
    PearsonRResultDummyClass is there to make the bridge if a user loads a project with PearsonRResult
     but the scipy version installed doesn't have an implementation of PearsonRResult
    """

    def __new__(cls, *args, **kwargs):
        obj = super().__new__(cls)
        obj.tuple = args
        return obj

    def __getitem__(self, index):
        return self.tuple[index]

    def to_json(self) -> Tuple[float, float]:
        return self.tuple

    def __repr__(self):
        return "PearsonRResultDummyClass{0}".format(self.tuple)


class LandsklimUnpickler(cPickle.Unpickler):

    remapping_global_names = {
        "LISDQSRectangle": "LandsklimRectangle",
        "PearsonRResult": "PearsonRResultDummyClass",
        "LISDQSProject": "LandsklimProject",
        "LISDQSConfiguration": "LandsklimConfiguration",
        "LISDQSAnalysis": "LandsklimAnalysis",
        "LISDQSInterpolation": "LandsklimInterpolation",
        "LISDQSAnalysisMode": "LandsklimAnalysisMode",
        "LISDQSInterpolationType": "LandsklimInterpolationType"
    }

    @staticmethod
    def postprocess(landsklim_project: LandsklimProject):
        for analysis in landsklim_project.get_analysis():  # type: LandsklimAnalysis
            polygon_raster_path = analysis.get_polygons_raster_path() if analysis.is_local() else None
            if not analysis.is_local():
                stations_count = analysis.get_stations_count()
                analysis._polygons = PolygonsDefinition(np.linspace(0, stations_count - 1, stations_count, dtype=int).reshape(1, -1), [[]], None)
            analysis._polygons.set_polygon_path(polygon_raster_path)
            for situation in analysis.get_station_situations():
                for phase in analysis.get_phases(situation):
                    setattr(phase, '_polygons', analysis.get_polygons())
                    if phase.class_name() == PhaseComposite.class_name():
                        setattr(phase, "_phases_raster_path", {})

    def find_class(self, module_name: str, global_name: str) -> Any:
        if global_name == "PearsonRResult":
            module_name = "landsklim.serialization.landsklim_unpickler"
        module_name = module_name.replace("landsklim.lisdqs.lisdqs_", "landsklim.lk.landsklim_")
        module_name = module_name.replace("landsklim.lisdqs.", "landsklim.lk.")
        if global_name in LandsklimUnpickler.remapping_global_names:
            global_name = LandsklimUnpickler.remapping_global_names[global_name]
        return super(LandsklimUnpickler, self).find_class(module_name, global_name)
