from typing import List

from landsklim.lk.regressor import Regressor


class RegressorCollection:
    """
    Represents a collection of regressors.

    The main use case of RegressorCollection is to allow the transfer of regressors definitions from the UI to
    registering regressors on the Landsklim project, keeping track of regressors registered as dependency,
    regressors that shouldn't be added to the analysis, but must be created to allow the creation of each regressors.

    Le cas d'utilisation principal RegressorCollection est de permettre le transfert de la définition des régresseurs
    à partir du tableau dans l'interface jusqu'à l'enregistrement des regresseurs dans le projet Landsklim,
    tout en gardant trace la liste des régresseurs ajoutés en tant de dépendences, permettant de créer une analyse
    avec uniquement les régresseurs souhaités, mais en ayant créé tous les régresseurs nécessaires en amont.

    :param regressors: Initialise the collection with these regressors
    :type regressors: List[Regression]
    """

    def __init__(self, regressors: List[Regressor] = None):
        self.__regressors: List[Regressor] = [] if regressors is None else regressors
        self.__first_level_regressor: List[Regressor] = self.__regressors.copy()

    def add_regressor(self, regressor: Regressor, add_on_top: bool, is_dependency: bool) -> None:
        """
        Add a regressor to the collection

        :param regressor: Regressor to add
        :type regressor: Regressor

        :param add_on_top: If true, add the regressor at the beginning of the collection
        :type add_on_top: bool

        :param is_dependency: Register the regressor as a dependency or not
        :type is_dependency: bool
        """
        if add_on_top:
            self.__regressors.insert(0, regressor)
        else:
            self.__regressors.append(regressor)
        if not is_dependency:
            self.__first_level_regressor.append(regressor)

    def move_to_top(self, regressor: Regressor):
        self.__regressors.remove(regressor)
        self.__regressors.insert(0, regressor)

    def get_regressors(self, exclude_dependencies: bool = False) -> List[Regressor]:
        """
        Returns the regressors as a list of Regressor.

        :param exclude_dependencies: If true, don't include regressors registered as dependencies
        :type exclude_dependencies: bool
        """
        return self.__first_level_regressor if exclude_dependencies else self.__regressors

    def is_first_level_regressor(self, regressor: Regressor) -> bool:
        """
        Check if the regressor is a 'first-level regressor'
        aka the regressor was chosen on the table, and is not a dependency

        :param regressor: Regressor to test
        :type regressor: Regressor

        :returns: True if the regressor is a 'first-level regressor', False otherwise
        :rtype: bool
        """
        is_fl_regressor: bool = False
        for r in self.__first_level_regressor:
            is_fl_regressor = is_fl_regressor or r.equals(regressor)
        return is_fl_regressor

    def sort(self):
        """
        Sort the collection
        """
        self.__regressors = sorted(self.__regressors, key=lambda r: "{0}_{1:08d}".format(r.prefix(), r.get_windows()))
        self.__first_level_regressor = sorted(self.__first_level_regressor, key=lambda r: "{0}_{1:08d}".format(r.prefix(), r.get_windows()))


