# -*- coding: utf-8 -*-
"""
/***************************************************************************
 Landsklim
 A QGIS plugin
 Logiciel d'interpolation de données quantitatives et spatiales

        begin                : 2023-04-03
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Laboratoire ThéMA
        email                : nicolas.lepy@univ-fcomte.fr

 ***************************************************************************/
"""
import os
import pathlib
import shutil
import time
import warnings
from math import ceil, floor
from typing import List, Dict, Callable, Union, Optional, Tuple, Type
from collections import OrderedDict
import json
import logging

import _pickle as cPickle

import numpy as np
import pandas as pd
import qgis
from PyQt5 import QtWidgets
from PyQt5.QtCore import QObject, QEvent, QDir
from PyQt5.QtWidgets import QFileDialog, QDialog
from osgeo import gdal, gdal_array
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QDate
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QToolButton, QMenu, QMessageBox, QProgressBar, QToolBar
from qgis.PyQt.QtCore import QThread, pyqtSignal
from qgis._core import QgsCoordinateReferenceSystem
from qgis.core import QgsRectangle, QgsMapLayer, QgsColorRampShader, QgsStyle, QgsColorRamp, QgsRasterShader, \
    QgsSingleBandPseudoColorRenderer, QgsRasterBandStats, QgsMapLayerType
from qgis._gui import QgsLayerTreeViewMenuProvider, QgsLayerTreeView
from qgis.PyQt.QtGui import QKeySequence
from qgis.PyQt.QtWidgets import QShortcut

# Initialize Qt resources from file resources.py
from qgis.gui import QgsMessageBarItem
from qgis.core import QgsMessageLog, Qgis, QgsProject, QgsExpressionContextUtils, QgsRasterLayer, QgsVectorLayer, \
    QgsLayerTree, QgsLayerTreeLayer, QgsLayerTreeGroup, QgsApplication

from landsklim.lk.logger import Log
from landsklim.lk.utils import LandsklimUtils
from landsklim.lk.netcdf import NetCdfExporterNetCdf4
from landsklim.lk.regressor_collection import RegressorCollection
from landsklim.landsklim_layer_tree_manager import LayerTreeContextMenuManager, LandsklimMenuProvider
from landsklim.serialization.json_decoder import LandsklimDecoder
from landsklim.serialization.landsklim_unpickler import LandsklimUnpickler
from landsklim.ui.view_save_netcdf import ViewSaveNetCdf

# landsklim_dependency_loader.LandsklimDependencyLoader.load_dependencies()

from landsklim.lk.landsklim_constants import LandsklimLayerType, LAYER_TYPE_PATH
from landsklim.lk.cache import update_qgis_project_cache, qgis_project_cache
from landsklim.lk.phase import IPhase
from landsklim.lk.phase_kriging import PhaseKriging
from landsklim.lk.phase_multiple_regression import PhaseMultipleRegression
from landsklim.lk.phase_polynomial import PhasePolynomial
from landsklim.lk.regressor import Regressor
from landsklim.lk.regressor_factory import RegressorFactory, RegressorDefinition

from landsklim.ui.view_new_project import ViewNewProject
from landsklim.ui.view_new_configuration import ViewNewConfiguration
from landsklim.ui.view_new_analysis import ViewNewAnalysis
from landsklim.ui.view_new_interpolation import ViewNewInterpolation
from landsklim.ui.view_charts import ViewCharts
from landsklim.ui.view_regressors import ViewRegressors
from landsklim.ui.widgets_phase_view import DialogPhaseView
from landsklim.ui.widget_variables_correlation import WidgetVariablesCorrelation
from landsklim.ui.widgets_analysis_charts import WidgetAnalysisSerieByPhase, WidgetAnalysisExplicativeVariableSeries,\
    WidgetAnalysisSerieByStation, WidgetAnalysisExplicativeVariable, WidgetAnalysisExplicativeVariableDistribution
from landsklim.lk.landsklim_project import LandsklimProject
from landsklim.lk.landsklim_configuration import LandsklimConfiguration
from landsklim.lk.map_layer import MapLayerCollection, RasterLayer, VectorLayer, MapLayer
from landsklim.lk.landsklim_analysis import LandsklimAnalysis, LandsklimAnalysisMode
from landsklim.lk.landsklim_interpolation import LandsklimInterpolation, LandsklimRectangle, LandsklimInterpolationType
from landsklim.lk.landsklim_constants import LandsklimLayerType
from landsklim.ui.threads import QThreadComputeVariables, QThreadComputeInterpolations, QThreadAnalysisModels, QThreadComputePolygons
from landsklim.resources import *  # Mandatory import to include resources (icons) in 'path'
from landsklim.processing.landsklim_provider import LandsklimProvider
import subprocess

MOCK_LANDSKLIM_PROJECT = None


class NoLandsklimProjectException(Exception):
    """Raised when no Landsklim project is defined but should be"""
    pass


class Landsklim:
    """
    QGIS Plugin Implementation.

    :param iface: An interface instance that will be passed to this class
        which provides the hook by which you can manipulate the QGIS
        application at run time.
    :type iface: QgsInterface

    :ivar QgsInterface iface: Save reference to the QGIS interface

    :ivar str plugin_dir: initialize plugin directory
    """

    def __init__(self, iface):
        self.iface = iface
        self.provider = None
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale', '')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            '{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        QgsApplication.messageLog().messageReceived.connect(Log.on_message)

        # Declare instance attributes
        self._landsklim_project: Optional[LandsklimProject] = None
        self._phases: List[Type[IPhase]] = [PhaseMultipleRegression,
                                   PhaseKriging]
        self._regressors: List[RegressorDefinition] = [RegressorDefinition("REGRESSOR_ROWS", [1], 1),
                                                       RegressorDefinition("REGRESSOR_COLS", [1], 1),
                                                       RegressorDefinition("REGRESSOR_ALTITUDE", [1, 3, 5], 1),
                                                       RegressorDefinition("REGRESSOR_SLOPE", [3, 5, 7], 1),
                                                       RegressorDefinition("REGRESSOR_ORIENTATION", [3, 5, 7], 1),
                                                       RegressorDefinition("REGRESSOR_ROUGHNESS", [], 1),
                                                       RegressorDefinition("REGRESSOR_ENCASEMENT", [3], 1)]
        self.actions = []
        self.menus = []
        self.toolbar: Optional[QToolBar] = None
        self.custom_menu_actions = []
        self.menu = self.tr(u'&Landsklim')

        self.action_new_project: Optional[QAction] = None
        self.action_load_project: Optional[QAction] = None
        self.action_add_regressors: Optional[QAction] = None
        self.action_new_configuration: Optional[QAction] = None
        self.action_new_analysis: Optional[QAction] = None
        self.action_new_interpolation: Optional[QAction] = None
        self.action_charts: Optional[QAction] = None
        self.tool_button: Optional[QToolButton] = None
        self.dialog_new_project: Optional[ViewNewProject] = None
        self.dialog_new_analysis: Optional[ViewNewAnalysis] = None
        self.dialog_regressors: Optional[ViewRegressors] = None
        self.dialog_new_configuration: Optional[ViewNewConfiguration] = None
        self.dialog_new_interpolation: Optional[ViewNewInterpolation] = None
        self.dialog_chart: Optional[QDialog] = None
        self.dialog_file: Optional[QFileDialog] = None
        self.message_box: Optional[QMessageBox] = None
        self._m_project: Optional[QMenu] = None
        self._m_configuration: Optional[QMenu] = None
        self._m_analysis: Optional[QMenu] = None
        self._m_interpolation: Optional[QMenu] = None

        self._progress_message_bar: Optional[QgsMessageBarItem] = None
        self._progress_bar: Optional[QProgressBar] = None
        self._thread_variables: Optional[QThread] = None
        self._thread_analysis_models: Optional[QThread] = None
        self._thread_polygons: Optional[QThread] = None
        self._thread_interpolation: Optional[QThread] = None

        # Extra callable handler for unit test to handle threads -------
        self._extra_finished_signal_receiver: Optional[Callable[[], None]] = None
        # --------------------------------------------------------------

        QgsMessageLog.logMessage("Loading Landsklim", level=Qgis.Info)
        # self.load_lib()
        missing_dependencies: List[str] = self.check_dependencies()
        if len(missing_dependencies) > 0 and self.gui_exists():
            _ = QMessageBox.critical(self.iface.parent(), "Landsklim", self.tr("Some Python dependencies are missing : {0}\nPlease install them with pip to make Landsklim work properly. Please refer to the documentation for further details.").format(", ".join(missing_dependencies)), QMessageBox.Ok)

        update_qgis_project_cache(QgsProject.instance())

        qgis_project_cache().cleared.connect(self.qgis_project_closed)
        qgis_project_cache().readProject.connect(self.qgis_project_loaded)
        qgis_project_cache().writeProject.connect(self.qgis_project_saved)

        self._context_menu_manager: Optional[LayerTreeContextMenuManager] = None

        self.__load_project_after_init_gui: bool = True

    def load_lib(self):
        """
        .. deprecated:: 0.4.4
            lisdqsapi is no longer used

        """
        try:
            from landsklim.lib import lisdqsapi
        except ImportError:
            QgsMessageLog.logMessage("lisdqsapi is not build for this system. Landsklim will use a full-Python implementation", level=Qgis.Info)


    def set_unittest_handler(self, extra_finished_signal_receiver: Callable[[], None]):
        """
        :param extra_finished_signal_receiver: Additional receiver function to call when computations are finished.
            Used during unit tests
        :type extra_finished_signal_receiver: Callable[[], None]
        """
        self._extra_finished_signal_receiver = extra_finished_signal_receiver

    def qgis_project_saved(self):
        """
        Triggered by QGIS when a project is saved
        Save the Landsklim project
        """
        self.save_landsklim_project()

    def qgis_project_closed(self):
        """
        Triggered by QGIS when a project is closed (QgsProject.cleared signal)
        Closes the associated Landsklim project
        """
        self.detach_landsklim_project()

    """def qgis_project_loaded_with_clone(self):
        
        # Triggered by QGIS when a project is loaded (QgsProject.readProject signal)
        # Open the associated Landsklim project if exists
        # 
        # .. deprecated:: 0.4.0
        #     User is responsible to move Landsklim folder with the QGIS project when moving a QGIS project.
        update_qgis_project_cache(QgsProject.instance())
        if "landsklim_project" in qgis_project_cache().customVariables():
            # Automatically retarget "landsklim_project" variable path on the active QGIS project.
            # Usually useless when the user reloads a QGIS project the path is still the same
            # Useful when the project was moved by hand on the disk by the user.
            # Project target the cloned Landsklim project under the cloned folder.
            # Warning : the user must have moved the Landsklim folder too else it will not work.
            landsklim_project_path = qgis_project_cache().customVariables()["landsklim_project"]
            base_dir, base_filename = os.path.dirname(landsklim_project_path), os.path.basename(landsklim_project_path)
            qgis_project_path = os.path.join(qgis_project_cache().absolutePath(), "Landsklim")
            # Update the project variable only if the path has changed prevent QGIS from thinking that the project has been modified
            if os.path.normcase(os.path.normpath(os.path.join(qgis_project_path, base_filename))) != os.path.normcase(os.path.normpath(landsklim_project_path)):
                landsklim_project_path = os.path.join(qgis_project_path, base_filename)
                QgsExpressionContextUtils.setProjectVariable(qgis_project_cache(), "landsklim_project", landsklim_project_path)
            self.load_landsklim_project(landsklim_project_path)"""

    def migrate_map_node(self, node, old_code: str, new_code: str):
        if node.customProperty(old_code) is not None:
            node.setCustomProperty(new_code, node.customProperty(old_code))
            node.removeCustomProperty(old_code)

    def migrate_qgs_project_to_080(self):
        """
        Migrate qgs project old names "lisdqs_project", "lisdqs_object" to new names "landsklim_project", "landsklim_object"
        """
        if "lisdqs_project" in qgis_project_cache().customVariables():
            QgsExpressionContextUtils.setProjectVariable(qgis_project_cache(), "landsklim_project", qgis_project_cache().customVariables()["lisdqs_project"])
            QgsExpressionContextUtils.removeProjectVariable(qgis_project_cache(), "lisdqs_project")
            for layer_name, layer in qgis_project_cache().mapLayers().items():  # type: str, QgsMapLayer
                self.migrate_map_node(layer, "lisdqs_code", "landsklim_code")
            groups: List[QgsLayerTreeGroup] = self.iface.layerTreeCanvasBridge().rootGroup().findGroups(True)
            for group in groups:  # type: QgsLayerTreeGroup
                self.migrate_map_node(group, "lisdqs_object", "landsklim_object")

    def qgis_project_loaded(self):
        """
        Triggered by QGIS when a project is loaded (QgsProject.readProject signal)
        Open the associated Landsklim project if exists
        """
        update_qgis_project_cache(QgsProject.instance())

        # Until 0.7.0, Landkslim project name was saved under the 'lisdqs_project' key in the project variable.
        self.migrate_qgs_project_to_080()

        if "landsklim_project" in qgis_project_cache().customVariables():
            lk_project_name: str = qgis_project_cache().customVariables()["landsklim_project"]
            qgis_project_path: str = os.path.join(qgis_project_cache().absolutePath(), "Landsklim")
            # Update the project variable only if the path has changed prevent QGIS from thinking that the project has been modified
            lk_project_path: str = os.path.join(qgis_project_path, lk_project_name)
            self.load_landsklim_project(lk_project_path)

    def detach_landsklim_project(self):
        """
        Closes the associated Landsklim project
        """
        self._landsklim_project = None
        if "landsklim_project" in qgis_project_cache().customVariables():
            QgsExpressionContextUtils.removeProjectVariable(qgis_project_cache(), "landsklim_project")
        update_qgis_project_cache(QgsProject.instance())
        self.refresh()

    @staticmethod
    def instance(iface=None):
        try:
            # QGIS context
            return qgis.utils.plugins["landsklim"]
        except Exception as e:
            # Unit test context
            global MOCK_LANDSKLIM_PROJECT
            if MOCK_LANDSKLIM_PROJECT is not None:
                return MOCK_LANDSKLIM_PROJECT
            print("[Landsklim] Create mock")
            from landsklim.tests import utilities
            if iface is None:
                QGIS_APP = utilities.get_qgis_app()
                iface = QGIS_APP[2]

            qgis_path = os.path.join(os.getcwd(), "QGIS")
            if not os.path.isdir(qgis_path):
                os.mkdir(qgis_path)

            qgs_project = QgsProject.instance()
            # QgsProject.instance().setPresetHomePath(qgis_path)
            qgs_project.setFileName(os.path.join(qgis_path, "project.qgis"))  # Set a path to tell QGIS we are on a stored project
            QgsExpressionContextUtils.removeProjectVariable(qgs_project, "landsklim_project")

            landsklim = Landsklim(iface)

            landsklim.init_processing()
            dem: QgsRasterLayer = QgsRasterLayer(os.path.join(os.path.dirname(__file__), 'tests', 'sources', 'mnt.tif'), 'MNT')
            ind_veg: QgsRasterLayer = QgsRasterLayer(os.path.join(os.path.dirname(__file__), 'tests', 'sources', 'ind_veg.tif'), 'ind_veg')
            stations: QgsVectorLayer = QgsVectorLayer(os.path.join(os.path.dirname(__file__), 'tests', 'sources', 'stations_min.shp'), 'Stations')
            crs = dem.crs()
            crs.createFromString('EPSG:32633')
            dem.setCrs(crs)
            ind_veg.setCrs(crs)
            stations.setCrs(crs)

            iface.addLayers([dem, stations, ind_veg])
            qgs_project.addMapLayers([dem, stations, ind_veg])

            mock_project_name = "Project for tests"
            landsklim._landsklim_project = LandsklimProject(mock_project_name,
                                                   MapLayerCollection.create_from_vector_layers([stations]),
                                                   RasterLayer(dem),
                                                   MapLayerCollection.create_from_raster_layers([ind_veg]),
                                                   regressors=landsklim._regressors, date=QDate(2000, 7, 16),
                                                      stations_no_data=30000)

            regressors: List[Regressor] = landsklim._landsklim_project.register_regressors(RegressorFactory.get_regressors_from_definitions(landsklim._regressors)).get_regressors()
            configuration1: LandsklimConfiguration = LandsklimConfiguration("Configuration1", landsklim._landsklim_project)
            landsklim._landsklim_project.add_configuration(configuration1)

            analysis_situations = [5, 6, 7, 8]  # Used for analysis tests
            analysis1: LandsklimAnalysis = LandsklimAnalysis("Analysis1", configuration1, LandsklimAnalysisMode.Global, 0, VectorLayer(stations), analysis_situations, False, landsklim._phases[0].class_name(), landsklim._phases[1].class_name(), regressors, landsklim._landsklim_project.get_stations_no_data())
            configuration1.add_analysis(analysis1)
            landsklim._landsklim_project.create_explicative_variables()  # analysis1.create_explicative_variables()
            analysis1.create_polygons()
            # [Interpolation not needed] interpolation1: LandsklimInterpolation = LandsklimInterpolation("Interpolation1", analysis1, False, None, [LandsklimInterpolationType.Global], True, LandsklimRectangle(dem.extent()))
            landsklim.initGui()  # TODO: Set RasterLayer to regressor after compute them to avoid landsklim.refresh() call constraint
            analysis1.construct_datasets()
            analysis1.construct_models()
            # landsklim.refresh()  # FIXME: Need to refresh because interpolation need kriged layers to be set on LandsklimAnalysis.get_kriging_layer(). These layers should be referenced directly after kriging computed
            # [Interpolation not needed] analysis1.add_interpolation(interpolation1)
            # [Interpolation not needed] analysis1.compute_interpolations(interpolation1)
            landsklim.refresh()
            QgsExpressionContextUtils.setProjectVariable(qgs_project, "landsklim_project", "{0}.pkl".format(mock_project_name))
            MOCK_LANDSKLIM_PROJECT = landsklim
            return MOCK_LANDSKLIM_PROJECT

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('Landsklim', message)

    def init_processing(self):
        print("[Register Landsklim]")
        self.provider = LandsklimProvider()
        QgsApplication.processingRegistry().removeProvider(self.provider)
        QgsApplication.processingRegistry().addProvider(self.provider)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
        shortcut=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param shortcut: Keyboard shortcut to reach this action. Defaults None.
        :type shortcut: str

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.toolbar.addAction(action)
            # self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        if shortcut is not None:
            self.iface.registerMainWindowAction(action, shortcut)

        self.actions.append(action)

        return action

    def add_menu_button(self, default_action: QAction=None) -> QToolButton:
        """
        :param default_action: Action to run when button is triggered
        :type default_action: QAction

        :returns: The button that was created. Note that button is added to menus list
        :rtype: QToolButton
        """
        tool_button = QToolButton()
        tool_button.setMenu(QMenu())
        tool_button.setPopupMode(QToolButton.MenuButtonPopup)
        tool_btn_action = self.iface.addToolBarWidget(tool_button)
        self.menus.append(tool_btn_action)
        if default_action is not None:
            tool_button.setDefaultAction(default_action)
        return tool_button

    def display_regression_model(self, analysis: LandsklimAnalysis, situation_index: int) -> None:
        self.dialog_chart = DialogPhaseView(analysis.get_station_situations()[situation_index], analysis)
        self.dialog_chart.show()
        self.dialog_chart.exec_()

    def dialog_analysis_delete(self, analysis: LandsklimAnalysis):
        self.message_box = QMessageBox.warning(self.iface.parent(), self.tr("Warning"), self.tr("Delete analysis {0} ?").format(analysis.get_name()), QMessageBox.Yes | QMessageBox.No)
        if self.message_box == QMessageBox.Yes:
            self.delete_analysis(analysis)
        self.message_box = None

    def dialog_interpolation_delete(self, analysis: LandsklimAnalysis, interpolation: LandsklimInterpolation):
        self.message_box = QMessageBox.warning(self.iface.parent(), self.tr("Warning"), self.tr("Delete interpolation {0} ({1}) ?").format(interpolation.get_name(), analysis.get_name()), QMessageBox.Yes | QMessageBox.No)
        if self.message_box == QMessageBox.Yes:
            self.delete_interpolation(analysis, interpolation)
        self.message_box = None

    def delete_analysis(self, analysis: LandsklimAnalysis) -> None:
        for configuration in self._landsklim_project.get_configurations():
            if analysis in configuration.get_analysis():
                analysis_path: str = configuration.delete_analysis(analysis)
        self.refresh()
        if os.path.isdir(analysis_path) and len(os.listdir(analysis_path)) == 0:  # If the directory is empty, delete it
            shutil.rmtree(analysis_path)

    def delete_interpolation(self, analysis: LandsklimAnalysis, interpolation: LandsklimInterpolation) -> None:
        analysis.delete_interpolation(interpolation)
        self.refresh()

    def save_raster_as_netcdf(self, source_raster: str, target_file: str, no_data: Optional[Union[int, float]]):
        # Il faudra sans doute passer par un formulaire permettant d'éditer les métadonnées du projet (comme les pas temporels, ...)
        # Nommer correctement Band1 (nom de la couche)
        # Global attributes :
        # :title = "Nom du projet QGIS/Landsklim ?"
        # :project_id : "Nom du projet QGIS/Landsklim"
        # Pour la bande exportée :
        # :standard_name = "Nom de la couche"
        # :long_name : "Nom de la couche"
        # :units = "T"
        # :missing_value = [doit être la même que :_FillValue : no_data]
        # :original_name : "Nom de la couche"
        if no_data is not None:
            gdal.Translate(target_file, source_raster, format='NetCDF', outputType=gdal.gdalconst.GDT_Float64, noData=no_data)
        else:
            gdal.Translate(target_file, source_raster, format='NetCDF', outputType=gdal.gdalconst.GDT_Float64)

    def dialog_convert_raster_to_netcdf(self, source_raster: str):
        self.dialog_file = QFileDialog()
        self.dialog_file.setFilter(self.dialog_file.filter() | QDir.Hidden)
        self.dialog_file.setDefaultSuffix('nc')

        self.dialog_file.setAcceptMode(QFileDialog.AcceptSave)
        self.dialog_file.setNameFilters([self.tr("NetCDF (*.nc)")])

        if self.dialog_file.exec():
            selected_file = self.dialog_file.selectedFiles()[0]
            if os.path.exists(source_raster):
                dataset: gdal.Dataset = gdal.Open(source_raster)
                no_data_value = dataset.GetRasterBand(1).GetNoDataValue()
                dataset = None  # Close the gdal.Dataset (in a non Pythonic way)
                self.save_raster_as_netcdf(source_raster, selected_file, no_data_value)
        self.dialog_file = None

    def dialog_convert_interpolation_to_netcdf(self, analysis: LandsklimAnalysis, interpolation: LandsklimInterpolation, interpolation_type: LandsklimInterpolationType):
        self.dialog_file = ViewSaveNetCdf(analysis, interpolation)

        if self.dialog_file.exec():
            selected_file = self.dialog_file.get_file_path() # self.dialog_file.selectedFiles()[0]
            unit = self.dialog_file.get_unit()
            long_name = self.dialog_file.get_long_variable_name()
            standard_name = self.dialog_file.get_standard_variable_name()
            dates = self.dialog_file.get_dates()
            NetCdfExporterNetCdf4().export(interpolation, interpolation_type, selected_file, unit, long_name, standard_name, dates)

        self.dialog_file = None

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/landsklim/icon.png'

        self.toolbar = self.iface.addToolBar(u'Landsklim')
        self.toolbar.setObjectName('Landsklim')

        self.action_new_project = self.add_action(':/plugins/landsklim/mActionLandsklimAdd.png', self.tr(u'New project'), self.run_new_project, add_to_menu=False, add_to_toolbar=True, status_tip=self.tr('Create a new Landsklim project'), shortcut="CTRL+F5")
        # self.action_load_project = self.add_action(icon_path, self.tr(u'Load project'), self.run_load_project, add_to_menu=False, add_to_toolbar=False, status_tip=self.tr(''), shortcut="CTRL+ALT+L")
        self.action_add_regressors = self.add_action(':/plugins/landsklim/mActionLandsklimEdit.png', self.tr(u'Add regressors'), self.run_add_regressors, add_to_menu=False, add_to_toolbar=True, status_tip=self.tr('Create new regressors'))
        # self.action_new_configuration = self.add_action(icon_path, self.tr(u'New configuration'), self.run_new_configuration, add_to_menu=False, add_to_toolbar=True, status_tip=self.tr('Create a new configuration'))
        self.action_new_analysis = self.add_action(':/plugins/landsklim/mActionLandsklimAnalysis.png', self.tr(u'New analysis'), self.run_new_analysis, add_to_menu=False, add_to_toolbar=True, status_tip=self.tr('Create a new analysis'))
        self.action_new_interpolation = self.add_action(':/plugins/landsklim/mActionLandsklimInterpolation.png', self.tr(u'New interpolation'), self.run_new_interpolation, add_to_menu=False, add_to_toolbar=True, status_tip=self.tr('Create a new interpolation'))
        self.action_charts = self.add_action(':/plugins/landsklim/mActionLandsklimCharts.png', self.tr(u'Charts'), self.run_charts, add_to_menu=False, add_to_toolbar=True, status_tip=self.tr('Show charts panel'))

        self.refresh()

        # Enable QGIS Python console
        # qgis.utils.iface.actionShowPythonDialog().trigger()
        self.init_processing()

        self._context_menu_manager = LayerTreeContextMenuManager(self.iface)
        landsklim_menu_provider: LandsklimMenuProvider = LandsklimMenuProvider(self.iface, self)
        landsklim_menu_provider.handle_on_analysis_situation_click(self.display_regression_model)
        landsklim_menu_provider.handle_on_save_as_netcdf_click(self.dialog_convert_raster_to_netcdf)
        landsklim_menu_provider.handle_on_interpolation_save_as_netcdf_click(self.dialog_convert_interpolation_to_netcdf)
        landsklim_menu_provider.handle_on_analysis_delete_click(self.dialog_analysis_delete)
        landsklim_menu_provider.handle_on_interpolation_delete_click(self.dialog_interpolation_delete)
        self._context_menu_manager.addProvider(landsklim_menu_provider)

        if self.__load_project_after_init_gui:
            self.__load_project_after_init_gui = False
            self.qgis_project_loaded()

        shortcut_style_1 = QShortcut(QKeySequence(Qt.ControlModifier + Qt.ShiftModifier + Qt.AltModifier + Qt.Key_S), self.iface.mainWindow())
        shortcut_style_1.setContext(Qt.ApplicationShortcut)
        shortcut_style_1.activated.connect(lambda: self.command("style"))
        shortcut_style_10 = QShortcut(QKeySequence(Qt.ControlModifier + Qt.ShiftModifier + Qt.AltModifier + Qt.Key_D),
                                     self.iface.mainWindow())
        shortcut_style_10.setContext(Qt.ApplicationShortcut)
        shortcut_style_10.activated.connect(lambda: self.command("style", 10))

    def layers_group_list_children(self, group_children: List):
        return {group_children[q[0]].name() + str(q[0]): q[1] for q in enumerate(group_children)}

    def sort_layers_group(self, group: QgsLayerTreeGroup):
        """
        Sort layers group alphabetically
        """
        group_children_dict: Dict = self.layers_group_list_children(group.children())
        children_keys_sorted = OrderedDict(sorted(self.layers_group_list_children(group.children()).items(), reverse=False)).keys()

        group_children_sorted = [group_children_dict[k].clone() for k in children_keys_sorted]
        group.insertChildNodes(0, group_children_sorted)
        for n in group_children_dict.values():
            group.removeChildNode(n)

    def get_layers_group(self, root: QgsLayerTree, name: str) -> QgsLayerTreeGroup:
        layers_group: QgsLayerTreeGroup = root.findGroup(name)
        if layers_group is None:
            layers_group: QgsLayerTreeGroup = root.addGroup(name)
        return layers_group

    def move_layer_to_group(self, source_layer: QgsLayerTreeLayer, group: QgsLayerTreeGroup):
        if source_layer is not None:
            clone_layer: QgsLayerTreeLayer = source_layer.clone()
            parent = source_layer.parent()
            group.insertChildNode(-1, clone_layer)
            parent.removeChildNode(source_layer)
            group.findLayer(clone_layer.layer().id()).setItemVisibilityChecked(False)
            clone_layer.setExpanded(False)

    def refresh_tree(self):
        """
        Refresh and organize layers tree content
        """

        self.clear_custom_context_menu()
        # view: QgsLayerTreeView = self.iface.layerTreeView()

        root: QgsLayerTree = self.iface.layerTreeCanvasBridge().rootGroup()

        name_explicative_variables = self.tr("Explicative variables")
        name_source_variables = self.tr("Source variables")
        name_stations = self.tr("Stations")
        name_configurations = self.tr("Configurations")
        name_other = self.tr("Other layers")

        station_group: QgsLayerTreeGroup = self.get_layers_group(root, name_stations)
        source_variables_group: QgsLayerTreeGroup = self.get_layers_group(root, name_source_variables)
        explanatory_variables_group: QgsLayerTreeGroup = self.get_layers_group(root, name_explicative_variables)
        configurations_group: QgsLayerTreeGroup = self.get_layers_group(root, name_configurations)
        additionnal_layers_group: QgsLayerTreeGroup = self.get_layers_group(root, name_other)
        # set of regressor families
        regressor_families = {regressor.name() for regressor in self._landsklim_project.get_regressors()}
        regressor_families_groups: Dict[str, QgsLayerTreeGroup] = {}
        for regressor_family in regressor_families:
            regressor_families_groups[regressor_family] = self.get_layers_group(explanatory_variables_group, regressor_family)

        # Update Stations tree
        for source in self._landsklim_project.get_stations_sources():  # type: VectorLayer
            self.move_layer_to_group(root.findLayer(source.qgis_layer().id()), station_group)

        self.move_layer_to_group(root.findLayer(self._landsklim_project.get_dem().qgis_layer().id()), source_variables_group)
        for source in self._landsklim_project.get_variables():
            self.move_layer_to_group(root.findLayer(source.qgis_layer().id()), source_variables_group)

        # Recreate Configurations tree
        # 1. Save layers by moving them to root
        for layer in configurations_group.findLayers():
            self.move_layer_to_group(layer, root)
        # 2. Erase all groups under Configurations
        configurations_group.removeAllChildren()
        # 3. Recreate configurations groups and move layers inside
        for i_configuration, configuration in enumerate(self._landsklim_project.get_configurations()):
            configuration_group: QgsLayerTreeGroup = configurations_group.addGroup(configuration.to_string())
            for i_analysis, analysis in enumerate(configuration.get_analysis()):
                analysis_group: QgsLayerTreeGroup = configuration_group.addGroup(analysis.to_string())
                analysis_group.setCustomProperty("landsklim_object", "configuration_{0}_analysis_{1}".format(i_configuration, i_analysis))
                # interpolations_group: QgsLayerTreeGroup = analysis_group.addGroup("Interpolations")
                for i_interpolation, interpolation in enumerate(analysis.get_interpolations()):  # type: LandsklimInterpolation
                    interpolation_group: QgsLayerTreeGroup = analysis_group.addGroup(interpolation.get_name())
                    interpolation_group.setCustomProperty("landsklim_object","configuration_{0}_analysis_{1}_interpolation_{2}".format(i_configuration, i_analysis, i_interpolation))
                    for i_type, itype in enumerate(interpolation.get_interpolation_types()):  # type: int, LandsklimInterpolationType
                        interpolation_type_group: QgsLayerTreeGroup = interpolation_group.addGroup(itype.str())
                        for situation in analysis.get_station_situations():  # type: int
                            interpolation_layer: QgsMapLayer
                            if situation not in interpolation.get_layers(itype):
                                lk_code_suffixes = {'C': configuration.get_name(), 'A': analysis.get_name(), 'I': interpolation.get_name(), 'P': itype.str()}
                                interpolation_layer = self.load_landsklim_layer(interpolation.get_paths(situation)[i_type], LandsklimLayerType.Interpolation, lk_code_suffixes)
                                interpolation.get_layers(itype)[situation] = RasterLayer(interpolation_layer) if interpolation.is_on_grid() else VectorLayer(interpolation_layer)  # TODO: Factory ?
                            else:
                                interpolation_layer = interpolation.get_layers(itype)[situation].qgis_layer()
                            self.move_layer_to_group(root.findLayer(interpolation_layer.id()), interpolation_type_group)

                # Load kriged layers
                """have_kriging = len([phase for phase in analysis.get_phases(analysis.get_station_situations()[0]) if phase.class_name() == PhaseKriging.class_name()]) > 0 if len(analysis.get_station_situations()) > 0 else False
                if have_kriging:
                    kriging_phase_index: PhaseKriging = [(i+1) for i, phase in enumerate(analysis.get_phases(analysis.get_station_situations()[0])) if phase.class_name() == PhaseKriging.class_name()][0]
                    for situation in analysis.get_station_situations():  # type: int
                        kriged_layer: Optional[QgsMapLayer]
                        # FIXME?: Due to questionable management of refreshing layers because they needs to be open on qgis to construct models/kriging,
                        # we must assure kriged layers were created before trying to open them
                        if os.path.exists(analysis.get_kriging_layer_path(situation)):
                            if analysis.get_kriging_layer(situation) is None:
                                lk_code_suffixes = {'C': configuration.get_name(), 'A': analysis.get_name()}  # , 'P': "Phase{0}".format(kriging_phase_index)
                                kriged_layer = self.load_landsklim_layer(analysis.get_kriging_layer_path(situation), LandsklimLayerType.Kriging, lk_code_suffixes)
                                raster_kriged_layer: RasterLayer = RasterLayer(kriged_layer)
                                analysis.set_kriging_layer(raster_kriged_layer, situation)
                            else:
                                kriged_layer = analysis.get_kriging_layer(situation).qgis_layer()
                            if kriged_layer is not None:
                                if "[C_" not in kriged_layer.customProperty('landsklim_code'):
                                    raise Exception("Kriged layer is invalid")
                                self.move_layer_to_group(root.findLayer(kriged_layer.id()), analysis_group)"""


        # Load polygons
        for analysis in self._landsklim_project.get_analysis():  # type: LandsklimAnalysis
            if analysis.is_local():
                lk_code_suffixes = {'C': analysis.get_configuration_name(), 'A': analysis.get_name()}
                self.refresh_variable_layer(LandsklimLayerType.Polygons, MapLayerCollection(), analysis.get_polygons_layer_displayed_name(), analysis.get_polygons_raster_path(), additionnal_layers_group, lk_code_suffixes)
                """if imported_lk_layer is not None:
                    analysis.set_polygons_layer(imported_lk_layer)"""

        # Load explicative variables and intermediate layers
        # Explicative variables raster are generated during analysis creation but not loaded
        # Same for intermediates rasters, but they are referenced in regressor definition, so we can get them from here
        # They are loaded and sorted now
        # [WARNING] Each regressors referenced inside each analysis must be the same instance as regressor in LandsklimProject
        #   This could become optional when Regressor's layer will be opened directly run-of-river
        #   Regressor creation is encapsulated inside Landsklim project instance
        for regressor in self._landsklim_project.get_regressors():
        # for analysis in self._landsklim_project.get_analysis():
            # for regressor in analysis.get_regressors():
            window = regressor.get_windows()
            parent_group: QgsLayerTreeGroup = regressor_families_groups[regressor.name()]
            imported_lk_layer: Optional[RasterLayer] = self.refresh_variable_layer(LandsklimLayerType.ExplicativeVariable, self._landsklim_project.get_raster_variables(), regressor.layer_name(), regressor.get_path(), parent_group, {})
            if imported_lk_layer is not None:
                regressor.set_raster_layer(imported_lk_layer)

            # Intermediate layers are no longer loaded
            """for intermediate_layer in regressor.get_intermediate_layers():  # type: str
                layer_path = os.path.join(qgis_project_cache().homePath(), 'Landsklim',
                                          LAYER_TYPE_PATH[LandsklimLayerType.AdditionalLayer],
                                          '{0}_{1}.tif'.format(intermediate_layer, window))
                layer_name = "{0}_{1}".format(intermediate_layer, window)
                imported_lk_layer: Optional[RasterLayer] = self.refresh_variable_layer(LandsklimLayerType.AdditionalLayer, self._landsklim_project.get_additional_layers(), layer_name, layer_path, additionnal_layers_group, {})
                if imported_lk_layer is not None:
                    self._landsklim_project.add_additional_layer(imported_lk_layer)"""

        self.sort_layers_group(explanatory_variables_group)
        self.sort_layers_group(source_variables_group)
        self.sort_layers_group(additionnal_layers_group)

    def refresh_variable_layer(self, layer_type: LandsklimLayerType, registered_layers: MapLayerCollection, layer_name: str, source_path: str, layer_tree_group: QgsLayerTreeGroup, lk_code_suffixes: Dict[str, str]) -> Optional[RasterLayer]:
        """
        Refresh a variable layer (variable derived from regressors or an intermediate layer)

        :param layer_type: Layer category
        :type layer_type: LandsklimLayerType
        :param registered_layers: List of already registered RasterLayers
        :type registered_layers: MapLayerCollection
        :param layer_name: Name of the layer
        :type layer_name: str
        :param source_path: Source file location on the disk of the raster, needed if the RasterLayer is not referenced yet
        :type source_path: str
        :param layer_tree_group: Target node of the layers tree
        :type layer_tree_group: QgsLayerTreeGroup
        :param lk_code_suffixes: Dictionary storing layer suffixes to make a layer owned by a specific analysis or phase
        :type lk_code_suffixes: Dict[str, str]

        :returns: The new RasterLayer if it wasn't referenced yet, None otherwise
        :rtype: Optional[RasterLayer]
        """
        new_layer: Optional[RasterLayer] = None
        qgis_layer: Optional[QgsRasterLayer] = None
        for layer in registered_layers:  # type: RasterLayer
            if layer.qgis_layer().name() == layer_name:
                qgis_layer = layer.qgis_layer()
        if qgis_layer is None and os.path.exists(source_path):
            qgis_layer = self.load_landsklim_layer(source_path, layer_type, lk_code_suffixes)
            qgis_layer.setName(layer_name)
            new_layer = RasterLayer(qgis_layer)
        if qgis_layer is not None:
            root = self.iface.layerTreeCanvasBridge().rootGroup()
            self.move_layer_to_group(root.findLayer(qgis_layer.id()), layer_tree_group)
        return new_layer

    def refresh_toolbar(self):
        """
        Refresh toolbar icons status linked with Landsklim project content
        """
        update_project_enabled: bool = self._landsklim_project is not None
        new_configuration_enabled: bool = self._landsklim_project is not None
        new_analysis_enabled: bool = self._landsklim_project is not None and len(self._landsklim_project.get_configurations()) > 0
        new_interpolation_enabled: bool = self._landsklim_project is not None and len(self._landsklim_project.get_analysis()) > 0
        sun_enabled: bool = self._landsklim_project is not None
        self.action_add_regressors.setEnabled(update_project_enabled)
        # self.action_new_configuration.setEnabled(new_configuration_enabled)
        self.action_new_analysis.setEnabled(new_analysis_enabled)
        self.action_new_interpolation.setEnabled(new_interpolation_enabled)
        self.action_charts.setEnabled(sun_enabled)

    """
    def refresh_menus(self):
        # Refresh menu content dynamically linked with Landsklim project content
        icon_path = ':/plugins/landsklim/icon.png'

        m: QMenu = self.tool_button.menu()
        m.clear()

        self._m_project = m.addMenu(QIcon(icon_path), self.tr(u'Landsklim'))
        self._m_configuration = m.addMenu(QIcon(icon_path), self.tr('Configuration'))
        self._m_analysis = m.addMenu(QIcon(icon_path), self.tr('Analysis'))
        self._m_interpolation = m.addMenu(QIcon(icon_path), self.tr('Interpolation'))
        m.addAction(self.action_about)

        self._m_project.addAction(self.action_new_project)
        if self._landsklim_project is None:
            self._m_project.addAction(self.action_load_project)
        else:
            action_project = QAction(self._landsklim_project.to_string(), m)
            action_project.triggered.connect(self.run_update_project)
            self._m_project.addSeparator()
            self._m_project.addAction(action_project)

        self._m_configuration.addAction(self.action_new_configuration)
        self._m_configuration.addSeparator()

        self._m_analysis.addAction(self.action_new_analysis)
        self._m_analysis.addSeparator()

        self._m_interpolation.addAction(self.action_new_interpolation)
        self._m_interpolation.addSeparator()

        self.action_new_project.setEnabled(QgsProject.instance() is not None)
        self.action_load_project.setEnabled(QgsProject.instance() is not None)

        self.action_new_configuration.setEnabled(self._landsklim_project is not None)
        self.action_new_analysis.setEnabled(self._landsklim_project is not None)
        self.action_new_interpolation.setEnabled(self._landsklim_project is not None)

        if self._landsklim_project is not None:
            for configuration in self._landsklim_project.get_configurations():
                self._m_configuration.addAction(QAction(configuration.to_string(), m))
            for analysis in self._landsklim_project.get_analysis():
                self._m_analysis.addAction(QAction(analysis.to_string(), m))
            for interpolation in self._landsklim_project.get_interpolations():
                self._m_interpolation.addAction(QAction(interpolation.to_string(), m)) """

    def refresh(self):
        """
        Refresh Landsklim menus and layers tree
        """
        # self.refresh_menus() [Landsklim scroll-down menu]
        self.refresh_toolbar()
        if self._landsklim_project is not None:
            self.refresh_tree()

    def load_landsklim_layer(self, path, landsklim_code_prefix: str, lk_code_suffixes: Dict[str, str]) -> QgsMapLayer:
        is_raster: bool = path.endswith('.tif')
        layer_name = os.path.basename(path).replace('.tif', '').replace('.gpkg', '')
        layer: Optional[QgsMapLayer] = None
        for layer_id, map_layer in qgis_project_cache().mapLayers().items():  # type: int, QgsMapLayer
            # Layers' source can contain some metadata defined after '|'
            if os.path.exists(map_layer.source()) and os.path.exists(str(path)) and os.path.samefile(map_layer.source().split('|')[0], str(path)):
                layer = map_layer

        if layer is None:
            layer: QgsMapLayer = QgsRasterLayer(path, layer_name) if is_raster else QgsVectorLayer(path, layer_name)
            lk_code_calculation_suffix: str = ""
            for suffix_key, suffix_value in lk_code_suffixes.items():
                lk_code_calculation_suffix = "{0}[{1}_{2}]".format(lk_code_calculation_suffix, suffix_key, suffix_value)
            landsklim_code = "{0}_{1}{2}".format(landsklim_code_prefix, layer_name, lk_code_calculation_suffix)
            layer.setCustomProperty('landsklim_code', landsklim_code)
            print("[Add from landsklim]", layer.source(), landsklim_code)
            QgsProject.instance().addMapLayer(layer)
        return layer

    def clear_custom_context_menu(self):
        pass
        """for action in self.custom_menu_actions:
            self.iface.removeCustomActionForLayerType(action)
        self.custom_menu_actions = []"""

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        QgsApplication.messageLog().messageReceived.disconnect(Log.on_message)
        for action in self.actions + self.menus:
            self.iface.removePluginMenu(
                self.tr(u'&Landsklim'),
                action)
            self.iface.removeToolBarIcon(action)
        self.clear_custom_context_menu()
        QgsApplication.processingRegistry().removeProvider(self.provider)
        del self._context_menu_manager
        self.toolbar.deleteLater()

    def run_new_project(self):
        """
        Create and show a new project dialog
        """
        if len(QgsProject.instance().homePath()) > 0:
            if self.dialog_new_project is None:
                is_update = self._landsklim_project is not None
                if self._landsklim_project is None:
                    project: LandsklimProject = LandsklimProject(regressors=self._regressors)
                    project.add_configuration(LandsklimConfiguration("Configuration", project))
                else:
                    project = self._landsklim_project
                self.dialog_new_project = ViewNewProject(project=project, is_update=is_update)
                self.dialog_new_project.show()
                result = self.dialog_new_project.exec_()
                if result:
                    self.dialog_new_project.update_project()
                    self._landsklim_project = self.dialog_new_project.get_project()
                    self.refresh()
                elif self.dialog_new_project.delete_project_enabled():
                    self.detach_landsklim_project()
                self.dialog_new_project = None
        elif self.gui_exists():
            _ = QMessageBox.critical(self.iface.parent(), "Landsklim", self.tr("Landsklim can only create a project on an existing QGIS project"), QMessageBox.Ok)

    def gui_exists(self) -> bool:
        """
        :returns: True if a GUI is available (e.g. QGIS context), otherwise False (e.g. tests context)
        :rtype: bool
        """
        return 'QT_QPA_PLATFORM' not in os.environ or os.environ['QT_QPA_PLATFORM'] != 'offscreen'

    def remove_landsklim_map_layers(self):
        """
        Remove from map every layer used by Landsklim.
        """
        layers_key = QgsProject.instance().mapLayers().keys()
        for layer_key in layers_key:
            if QgsProject.instance().mapLayers()[layer_key].customProperty('landsklim_code') is not None:
                QgsProject.instance().removeMapLayer(QgsProject.instance().mapLayers()[layer_key])

    def save_landsklim_project(self) -> Tuple[bool, bool, str]:
        """
        Save the current project

        :returns:

            - First boolean : True is Landsklim project was correctly saved, False otherwise,
            - Second boolean : True is Landsklim project was cloned, False otherwise
            - String : and string containing error cause
        :rtype: Tuple[bool, bool, str]

        """
        was_saved, was_cloned, error = False, False, ""
        if self._landsklim_project is not None:
            if LandsklimUtils.landsklim_version() != self._landsklim_project.get_landsklim_version() and self.gui_exists():
                self.message_box = QMessageBox() # QMessageBox.information was not used because tests were softlocked
                self.message_box.setText(self.tr("This project was created with an older version of Landsklim ({0}).\nMigration to the current version ({1}) may increase saving time.".format(self._landsklim_project.get_landsklim_version(), LandsklimUtils.landsklim_version())))
                self.message_box.setStandardButtons(QMessageBox.Ok)
                self.message_box.setIcon(QMessageBox.Information)
                self.message_box.setWindowTitle("Landsklim")
                self.message_box.exec()
                self.message_box = None

            was_saved, was_cloned, error = self._landsklim_project.save()
            self._landsklim_project.display()
            if not was_saved and self.gui_exists():
                displayed_error = "Landsklim project was not saved. {0}".format(error)
                self.iface.messageBar().pushMessage("Error", displayed_error, level=Qgis.Warning)
            # Update layers
            if was_cloned:
                # Must close all the Landsklim layers and refresh
                self.remove_landsklim_map_layers()
                self.refresh()
        return was_saved, was_cloned, error

    def load_landsklim_project(self, project_path: str):
        """
        Load Landsklim project associated with QGIS project

        :param project_path: Path of the project as written in landsklim_project custom attribute of QGIS project
        :type project_path: str
        """
        extension = pathlib.Path(project_path).suffix
        with open(project_path, "rb") as input_file:
            try:
                time_start = time.perf_counter()
                if extension == '.json':
                    self._landsklim_project = json.load(fp=input_file, cls=LandsklimDecoder)
                elif extension == '.pkl':
                    self._landsklim_project = LandsklimUnpickler(input_file).load()  # cPickle.load(input_file)
                    LandsklimUnpickler.postprocess(self._landsklim_project)
                else:
                    raise RuntimeError("{0} : unknown format".format(project_path))
                print("[Landsklim][{0}][Decoder] {1:.4f}s".format(extension, time.perf_counter() - time_start))
                self._landsklim_project.display()
            except Exception as e:
                import traceback
                traceback.print_exc()
                print("[Exception]", e)
                self._landsklim_project = None

        if self._landsklim_project is not None:
            self.refresh()
        elif self.gui_exists():
            msg: str = self.tr("Unable to load the Landsklim project.\nPossible cause: the project was saved on an older version of Landsklim and the migration to the new version has failed.")
            _ = QMessageBox.critical(self.iface.parent(), "Landsklim", msg, QMessageBox.Ok)

    """def run_load_project(self):
        if "landsklim_project" not in QgsProject.instance().customVariables():
            if self.gui_exists():
                _ = QMessageBox.critical(self.iface.parent(), "Critical", "ERROR_NO_LANDSKLIM_PROJECT", QMessageBox.Ok)
        else:
            file_path = QgsProject.instance().customVariables()["landsklim_project"]
            self.load_landsklim_project(file_path)"""

    def run_new_configuration(self):
        """
        Create and show a new configuration dialog
        """
        if self.dialog_new_configuration is None:
            self.dialog_new_configuration = ViewNewConfiguration()
            self.dialog_new_configuration.show()
            result = self.dialog_new_configuration.exec_()
            if result and self._landsklim_project is not None:
                configuration_name = self.dialog_new_configuration.get_configuration_name()
                lk_configuration: LandsklimConfiguration = LandsklimConfiguration(configuration_name, self._landsklim_project)
                self._landsklim_project.add_configuration(lk_configuration)
                self.refresh()
            self.dialog_new_configuration = None

    def create_progress_message_bar(self, total: Optional[int]):
        self._progress_message_bar = self.iface.messageBar().createMessage("...")
        if total is not None:
            self._progress_bar = QProgressBar()
            self._progress_bar.setMaximum(total)
            self._progress_bar.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)
            self._progress_message_bar.layout().addWidget(self._progress_bar)
        self.iface.messageBar().pushWidget(self._progress_message_bar, Qgis.Info)
        self.iface.messageBar().findChildren(QToolButton)[0].setHidden(True)

    def clear_iface_message_bar(self):
        self.iface.messageBar().clearWidgets()
        self._progress_bar = None
        self._progress_message_bar = None

    """
    Interpolation thread
    """

    def handle_on_interpolation_started(self, total: int):
        self.create_progress_message_bar(total)

    def handle_on_interpolation_step(self, situation_name: str, i: int, total: int):
        self._progress_message_bar.setText("{0} : {1}/{2}".format(situation_name, i, total))
        self._progress_message_bar.update()
        self._progress_bar.setValue(i)

    def handle_on_interpolation_failed(self, analysis: LandsklimAnalysis, interpolation: LandsklimInterpolation):
        message = self.tr("Interpolation {0} failed. See the Python console for more information.".format(interpolation.get_name()))
        _ = QMessageBox.critical(self.iface.parent(), "Landsklim", message, QMessageBox.Ok)
        self.delete_interpolation(analysis, interpolation)
        self.refresh()

    def handle_on_interpolation_finished(self):
        self.iface.messageBar().clearWidgets()
        self._progress_bar = None
        self._progress_message_bar = None
        self.refresh()

    """
    Explicative variables thread
    """

    def handle_on_explicative_variables_compute_started(self, total: int):
        self.create_progress_message_bar(total)

    def handle_on_regressor_compute_finished(self, regressor: Regressor, window_size: int, i: int, total: int):
        self._progress_message_bar.setText("{0} ({1}) : {2}/{3}".format(regressor.name(), window_size, i, total))
        self._progress_message_bar.update()
        self._progress_bar.setValue(i)

    def handle_on_explicative_variables_compute_finished(self, only_variables: bool):
        if not only_variables:
            # Launch thread aiming to build analysis dataset and models
            self._thread_polygons.start()
        else:
            self.refresh()  # Add explicative variables
            self.clear_iface_message_bar()

    """
    Polygons building thread
    """

    def handle_on_polygons_compute_start(self, message: str):
        self._progress_message_bar.setText(message)
        self._progress_message_bar.update()

    def handle_on_polygons_compute_end(self):
        # It's necessary to call self.refresh() before self.construct_datasets() because it need opened layer
        # but it's not a good pattern
        self.refresh()  # Add explicative variables and polygons

        # Launch thread aiming to build analysis dataset and models
        self._thread_analysis_models.start()

    def handle_on_polygons_compute_fail(self, analysis: LandsklimAnalysis):

        if self.gui_exists():
            message = self.tr("""An error has occured during the calculation of the polygons. The most probable cause of this error is a memory overflow :
                        - If applicable, check that a NO_DATA value is correctly entered in the DEM.
                        - The neighborhood size is too high
                        - The DEM is too large""")
            _ = QMessageBox.critical(self.iface.parent(), "Landsklim", message, QMessageBox.Ok)
        for configuration in self._landsklim_project.get_configurations():
            configuration.delete_analysis(analysis)
        self.clear_iface_message_bar()
        self.refresh()

    """
    Models building thread
    """

    def handle_on_model_compute_started(self, total: Optional[int]):
        self._progress_bar.setMaximum(total)

    def handle_on_model_compute_step(self, message: str, progress_bar_value: Optional[int]):
        self._progress_message_bar.setText(message)
        self._progress_message_bar.update()
        if progress_bar_value is not None:
            self._progress_bar.setValue(progress_bar_value)

    def handle_on_model_compute_finished(self):
        self.clear_iface_message_bar()

    def handle_on_model_compute_fail(self, analysis: LandsklimAnalysis, invalid_fields: str):
        if self.gui_exists():
            base_message = self.tr("Regression models can't be created.\n")
            if len(invalid_fields) > 0:
                sub_message = self.tr("Extent of variables {0} is not aligned with the stations layer.\nMake sure that the additional variables raster extent correspond to the extents of the stations and the DEM.".format(invalid_fields))
            else:
                sub_message = self.tr("If it's a global analysis, there is not enough valid samples.\nIf it's a local analysis, the local neighborhood is maybe smaller than predictives variables.\n\nMake sure that the additional variables raster extent correspond to the extents of the stations and the DEM.")
            message = "{0}{1}".format(base_message, sub_message)
            _ = QMessageBox.critical(self.iface.parent(), "Landsklim", message, QMessageBox.Ok)
        for configuration in self._landsklim_project.get_configurations():
            configuration.delete_analysis(analysis)
        self.refresh()


    def launch_thread_create_explicative_variables(self, analysis: Optional[LandsklimAnalysis], only_variables: bool):
        """
        Launch thread creating explicative variables of an analysis.

        :param analysis: Analysis. Not needed if only_variables is True
        :type analysis: Optional[LandsklimAnalysis]

        :param only_variables: If False, build analysis models once regressors are computed. Otherwise, only refresh QGIS
        :type only_variables: bool
        """
        self._thread_variables: QThreadComputeVariables = QThreadComputeVariables(self._landsklim_project, only_variables=only_variables)
        self._thread_variables.computationStarted.connect(self.handle_on_explicative_variables_compute_started)
        self._thread_variables.explicativeVariableComputed.connect(self.handle_on_regressor_compute_finished)
        self._thread_variables.computationFinished.connect(self.handle_on_explicative_variables_compute_finished)

        if not only_variables:
            self._thread_polygons: QThreadComputePolygons = QThreadComputePolygons(analysis)
            self._thread_polygons.computationStarted.connect(self.handle_on_polygons_compute_start)
            self._thread_polygons.computationFinished.connect(self.handle_on_polygons_compute_end)
            self._thread_polygons.computationFailed.connect(self.handle_on_polygons_compute_fail)

            self._thread_analysis_models: QThreadAnalysisModels = QThreadAnalysisModels(analysis)
            self._thread_analysis_models.computationStarted.connect(self.handle_on_model_compute_started)
            self._thread_analysis_models.computationStep.connect(self.handle_on_model_compute_step)
            self._thread_analysis_models.computationFinished.connect(self.handle_on_model_compute_finished)
            self._thread_analysis_models.computationFail.connect(self.handle_on_model_compute_fail)
            if self._extra_finished_signal_receiver is not None:
                self._thread_analysis_models.computationFinished.connect(self._extra_finished_signal_receiver)

        self._thread_variables.start()

    def run_add_regressors(self):
        """
        Create and show a dialog to edit regressors
        """
        if self.dialog_regressors is None:
            self.dialog_regressors = ViewRegressors(self._landsklim_project.get_regressors())
            self.dialog_regressors.show()
            result = self.dialog_regressors.exec_()
            if result and self._landsklim_project is not None:
                selected_regressors: RegressorCollection = RegressorFactory.get_regressors_from_definitions(self.dialog_regressors.get_explicatives_variables())
                self._landsklim_project.register_regressors(selected_regressors)
                self.launch_thread_create_explicative_variables(only_variables=True, analysis=None)
            self.dialog_regressors = None

    def run_new_analysis(self):
        """
        Create and show a new analysis dialog
        """
        if self.dialog_new_analysis is None:
            self.dialog_new_analysis = ViewNewAnalysis(default_regressors=self._landsklim_project.get_regressors())
            self.dialog_new_analysis.show()
            result = self.dialog_new_analysis.exec_()
            if result and self._landsklim_project is not None and self.dialog_new_analysis.get_configuration() is not None:
                analysis_name = self.dialog_new_analysis.get_analysis_name()
                analysis_is_local: LandsklimAnalysisMode = self.dialog_new_analysis.get_interpolation_mode()
                analysis_neighborhood_size: int = self.dialog_new_analysis.get_neighborhood_size()
                analysis_stations = VectorLayer(self.dialog_new_analysis.get_data_to_interpolate())
                analysis_station_situations = self.dialog_new_analysis.get_situations()
                analysis_use_pc = self.dialog_new_analysis.get_use_partial_correlations()
                analysis_phase1, analysis_phase2 = self.dialog_new_analysis.get_phases()
                analysis_regressors_definition: List[RegressorDefinition] = self.dialog_new_analysis.get_explicatives_variables()
                analysis_configuration = self.dialog_new_analysis.get_configuration()
                analysis_regressors: List[Regressor] = self._landsklim_project.register_regressors(RegressorFactory.get_regressors_from_definitions(analysis_regressors_definition)).get_regressors(exclude_dependencies=True)
                analysis: LandsklimAnalysis = LandsklimAnalysis(analysis_name, analysis_configuration, analysis_is_local, analysis_neighborhood_size, analysis_stations, analysis_station_situations, analysis_use_pc, analysis_phase1, analysis_phase2, analysis_regressors, self._landsklim_project.get_stations_no_data())
                analysis_configuration.add_analysis(analysis)
                # task = QgsTask.fromFunction('FromFunctionTest', self.task_compute_explicative_variables, on_finished=self.task_compute_explicative_variables_completed, analysis=analysis)
                # QgsApplication.taskManager().addTask(task)
                # self.refresh()
                self.launch_thread_create_explicative_variables(analysis=analysis, only_variables=False)

            self.dialog_new_analysis = None

    def run_new_interpolation(self):
        """
        Create and show a new interpolation dialog
        """
        if self.dialog_new_interpolation is None:
            self.dialog_new_interpolation = ViewNewInterpolation()
            self.dialog_new_interpolation.show()
            result = self.dialog_new_interpolation.exec_()
            if result and self._landsklim_project is not None:
                interpolation_name: str = self.dialog_new_interpolation.interpolation_name()
                interpolation_analysis: LandsklimAnalysis = self.dialog_new_interpolation.selected_analysis()
                interpolation_phases: List[LandsklimInterpolationType] = self.dialog_new_interpolation.selected_phases()
                interpolation_min_value = self.dialog_new_interpolation.get_minimum_value()
                interpolation_max_value = self.dialog_new_interpolation.get_maximum_value()
                interpolation_extrapolation = self.dialog_new_interpolation.selected_extrapolation_margin()
                interpolation_extrapolation_mode = self.dialog_new_interpolation.get_extrapolation_mode()
                interpolation_extrapolation_value = self.dialog_new_interpolation.get_extrapolation_value()
                interpolation_grid = self.dialog_new_interpolation.is_interpolation_on_grid()
                interpolation_extent: Union[QgsVectorLayer, QgsRectangle] = self.dialog_new_interpolation.selected_interpolation_extent()
                if not interpolation_grid:
                    interpolation_extent = VectorLayer(interpolation_extent)
                else:
                    interpolation_extent = LandsklimRectangle(interpolation_extent)
                interpolation: LandsklimInterpolation = LandsklimInterpolation(interpolation_name, interpolation_analysis, interpolation_min_value, interpolation_max_value, interpolation_extrapolation_mode, interpolation_extrapolation_value, interpolation_extrapolation, interpolation_phases, interpolation_grid, interpolation_extent)
                interpolation_analysis.add_interpolation(interpolation)

                self._thread_interpolation = QThreadComputeInterpolations(interpolation_analysis, interpolation)
                self._thread_interpolation.computationStarted.connect(self.handle_on_interpolation_started)
                self._thread_interpolation.interpolationComputed.connect(self.handle_on_interpolation_step)
                self._thread_interpolation.computationFinished.connect(self.handle_on_interpolation_finished)
                self._thread_interpolation.interpolationFailed.connect(self.handle_on_interpolation_failed)
                if self._extra_finished_signal_receiver is not None:
                    self._thread_interpolation.computationFinished.connect(self._extra_finished_signal_receiver)
                self._thread_interpolation.start()
            self.dialog_new_interpolation = None

    def run_charts(self):
        """
        Create and show the charts dialog
        """
        if self._landsklim_project is not None:
            self.dialog_chart = ViewCharts(self._landsklim_project)
            self.dialog_chart.show()
            self.dialog_chart.exec_()

    def get_landsklim_project(self) -> LandsklimProject:
        """
        Get opened Landsklim project

        :returns: Current Landsklim project
        :rtype: LandsklimProject
        """
        return self._landsklim_project

    def get_available_phases(self) -> List[Type[IPhase]]:
        """
        Phases available inside Landsklim

        :returns: List of IPhase
        :rtype: List[IPhase]
        """
        return self._phases

    def get_available_regressors(self) -> List[RegressorDefinition]:
        """
        Regressors available inside Landsklim

        :returns: List of Regressor
        :rtype: List[RegressorDefinition]

        .. deprecated:: 0.3.3

        """
        base_regressors = self._regressors.copy()
        if self._landsklim_project is not None:
            for source_variable in self._landsklim_project.get_variables():  # type: RasterLayer
                base_regressors.append(RegressorDefinition("{0}".format(source_variable.qgis_layer().name()), [], 1))
        return base_regressors

    def check_dependencies(self) -> List[str]:
        """
        Check if Python dependencies are correctly installed.

        :returns: List of missing modules
        :rtype: List[str]
        """
        missing_modules = []
        try:
            import pandas as pd
        except ModuleNotFoundError as e:
            missing_modules.append('pandas')
        try:
            import numpy as np
        except ModuleNotFoundError as e:
            missing_modules.append('numpy')
        try:
            import sklearn
        except ModuleNotFoundError as e:
            missing_modules.append('scikit-learn')
        try:
            import pykrige
        except ModuleNotFoundError as e:
            missing_modules.append('pykrige')
        try:
            import netCDF4
        except ModuleNotFoundError as e:
            missing_modules.append('netCDF4')

        return missing_modules

    def command(self, *args: str):  # pragma: no cover
        """
        Execute a command aiming to modify Landsklim behaviour.
        Accessed with :

        .. code-block::

            from qgis.utils import plugins
            plugins['landsklim'].command(<your command>)
            plugins['landsklim'].command("force_model", "A1L", "local_analysis_23_juil.txt", "local_analysis_24_juil.txt", "local_analysis_25_juil.txt")

        **Commands :**

        :code:`force_models <analysis_name> <coefficients_file>`

        Force models coefficients of a local analysis.

        - ``analysis_name`` : Name of the analysis to update models coefficients
        - ``coefficients_file`` : File name hosting models coefficients

        :code:`integer`

        Landsklim works with integers

        :code:`float`

        Landsklim works with floats

        :code:`style`

        Set a generic style on each interpolations

        """
        from landsklim.ui.quick_commands import QuickCommands
        QuickCommands.command(self, *args)
