# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Landlab Algorithm Provider - Utilities
                                 A QGIS plugin
 Earth Surface Dynamics modeling algorithms - Utility functions
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-07-25
        copyright            : (C) 2025 by Lucas Amion
        email                : lamion@dcc.uchile.cl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Lucas Amion'
__date__ = '2025-07-25'
__copyright__ = '(C) 2025 by Lucas Amion'

from osgeo import gdal, ogr
import numpy as np
from qgis.core import QgsRasterLayer, QgsProject, QgsVectorLayer, QgsProcessingException


def getGeotransform(extent, dx, dy):
    """
    Create a GDAL geotransform tuple from extent and pixel sizes.
    
    Args:
        extent: QgsRectangle with spatial extent
        dx: pixel width
        dy: pixel height
    
    Returns:
        list: GDAL geotransform [top-left x, pixel width, rotation, top-left y, rotation, -pixel height]
    """
    return [
        extent.xMinimum(),  # top-left x
        dx,                 # pixel width
        0,                  # rotation
        extent.yMaximum(),  # top-left y
        0,                  # rotation
        -dy                 # pixel height (negative for north-up)
    ]


def vectorToRaster(vector_layer, geotransform, ncols, nrows, field_to_burn=None, no_data=0):
    """
    Convert vector layer to raster array.
    
    Args:
        vector_layer: QgsVectorLayer to rasterize
        geotransform: GDAL geotransform tuple
        ncols: number of columns
        nrows: number of rows
        field_to_burn: field to burn values from (None for 1/0 mask)
        no_data: no data value
    
    Returns:
        numpy array with rasterized vector data
    """
    # Create temporary raster
    driver = gdal.GetDriverByName('MEM')
    temp_raster = driver.Create('', ncols, nrows, 1, gdal.GDT_Byte)
    temp_raster.SetGeoTransform(geotransform)
    
    # Set projection from vector layer
    temp_raster.SetProjection(vector_layer.crs().toWkt())
    
    # Initialize with no_data values
    band = temp_raster.GetRasterBand(1)
    band.Fill(no_data)
    
    # Create temporary vector dataset
    vector_driver = ogr.GetDriverByName('Memory')
    temp_vector = vector_driver.CreateDataSource('')
    
    # Create layer
    layer = temp_vector.CreateLayer('temp', geom_type=ogr.wkbPolygon)
    
    # Add features from QGIS vector layer
    for feature in vector_layer.getFeatures():
        geom = feature.geometry()
        ogr_feature = ogr.Feature(layer.GetLayerDefn())
        ogr_geom = ogr.CreateGeometryFromWkt(geom.asWkt())
        ogr_feature.SetGeometry(ogr_geom)
        layer.CreateFeature(ogr_feature)
    
    # Rasterize
    if field_to_burn is None:
        gdal.RasterizeLayer(temp_raster, [1], layer, burn_values=[1])
    else:
        gdal.RasterizeLayer(temp_raster, [1], layer, options=[f"ATTRIBUTE={field_to_burn}"])
    
    # Read as array
    array = band.ReadAsArray()
    
    # Clean up
    temp_raster = None
    temp_vector = None
    
    return array


def createProcessingMask(mask_layer, use_selected_features, extent, width, height, dx, dy, feedback=None):
    """
    Create a processing mask from a vector layer for algorithm processing.
    
    Args:
        mask_layer: QgsVectorLayer to create mask from (can be None)
        use_selected_features: bool, whether to use only selected features
        extent: QgsRectangle with spatial extent
        width: raster width in pixels
        height: raster height in pixels
        dx: pixel width
        dy: pixel height
        feedback: QgsProcessingFeedback object (optional)
    
    Returns:
        numpy array with mask (1 for processing areas, 0 for skip areas) or None if no mask
    
    Raises:
        QgsProcessingException: if mask layer is invalid or no features selected when required
    """
    if mask_layer is None:
        return None
        
    if feedback:
        feedback.pushInfo('Creating mask from vector layer...\n')
    
    # Check if mask layer is valid
    if not mask_layer.isValid():
        error_msg = 'Mask layer is not valid\n'
        if feedback:
            feedback.reportError(error_msg)
        raise QgsProcessingException(error_msg)
    
    # Get geotransform for rasterizing
    geotransform = getGeotransform(extent, dx, dy)
    
    # Create temporary layer with selected features if requested
    if use_selected_features:
        if mask_layer.selectedFeatureCount() == 0:
            error_msg = 'No features selected in mask layer\n'
            if feedback:
                feedback.reportError(error_msg)
            raise QgsProcessingException(error_msg)
        
        # Create temporary layer with selected features
        temp_layer = QgsVectorLayer(f'Polygon?crs={mask_layer.crs().authid()}', 'temp_mask', 'memory')
        temp_dp = temp_layer.dataProvider()
        temp_dp.addAttributes(mask_layer.fields().toList())
        temp_layer.updateFields()
        
        # Add selected features
        for feature in mask_layer.getSelectedFeatures():
            temp_dp.addFeature(feature)
        
        # Rasterize the temporary layer
        processing_mask = vectorToRaster(temp_layer, geotransform, width, height)
    else:
        # Rasterize all features in the mask layer
        processing_mask = vectorToRaster(mask_layer, geotransform, width, height)
    
    if feedback:
        masked_pixels = np.sum(processing_mask)
        total_pixels = processing_mask.size
        feedback.pushInfo(f'Mask created. Processing will be applied to {masked_pixels} pixels out of {total_pixels}\n')
    
    return processing_mask


def rasterToNumpy(input_layer, feedback=None):
    """
    Convert QGIS raster layer to numpy array and extract raster properties.
    
    Args:
        input_layer: QgsRasterLayer to read from
        feedback: QgsProcessingFeedback object for progress reporting (optional)
    
    Returns:
        tuple: (elevation_data, provider, extent, width, height, dx, dy)
    """
    # Read raster data
    provider = input_layer.dataProvider()
    extent = provider.extent()
    width = provider.xSize()
    height = provider.ySize()
    
    # Calculate pixel size (assuming square pixels)
    dx = extent.width() / width
    dy = extent.height() / height
    
    if feedback:
        feedback.pushInfo(f'Raster dimensions: {width}x{height}, pixel size: {dx:.2f}x{dy:.2f}\n')
    
    # Read the raster data as numpy array
    block = provider.block(1, extent, width, height)
    
    # Convert to numpy array
    elevation_data = np.zeros((height, width), dtype=np.float64)
    for row in range(height):
        for col in range(width):
            elevation_data[row, col] = block.value(row, col)
    
    return elevation_data, provider, extent, width, height, dx, dy


def createOutputRaster(output_data, output_path, extent, width, height, dx, dy, input_layer, layer_name="Processed Raster", feedback=None):
    """
    Save processed data to a raster file and create QGIS layer.
    
    Args:
        output_data: 2D numpy array with processed data
        output_path: path to save the output raster
        extent: QgsRectangle with spatial extent
        width: raster width in pixels
        height: raster height in pixels
        dx: pixel width
        dy: pixel height
        input_layer: original QgsRasterLayer for copying properties
        layer_name: name for the output layer
        feedback: QgsProcessingFeedback object (optional)
    
    Returns:
        QgsRasterLayer: the created output layer
    """
    # Create output raster
    # Create a new GDAL dataset
    driver = gdal.GetDriverByName('GTiff')
    output_dataset = driver.Create(output_path, width, height, 1, gdal.GDT_Float64)
    
    # Set geotransform and projection
    geotransform = getGeotransform(extent, dx, dy)
    output_dataset.SetGeoTransform(geotransform)
    output_dataset.SetProjection(input_layer.crs().toWkt())
    
    # Write the data
    band = output_dataset.GetRasterBand(1)
    band.WriteArray(output_data)
    
    # Set no data value if available
    provider = input_layer.dataProvider()
    no_data_value = provider.sourceNoDataValue(1)
    if no_data_value is not None:
        band.SetNoDataValue(no_data_value)
    
    # Close the dataset
    output_dataset = None
    
    # Create a QgsRasterLayer from the output file
    output_layer = QgsRasterLayer(output_path, layer_name)
    
    # Copy the renderer (color scheme) from the input layer to the output layer
    if input_layer.renderer():
        # Clone the renderer from the input layer
        renderer_clone = input_layer.renderer().clone()
        output_layer.setRenderer(renderer_clone)
        
        # Trigger a repaint to apply the new renderer
        output_layer.triggerRepaint()
        
        if feedback:
            feedback.pushInfo('Applied color scheme from input layer to output layer\n')
    
    # Add the layer to the project to make the color scheme visible
    QgsProject.instance().addMapLayer(output_layer)
    
    return output_layer
