# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Landlab Algorithm Provider
                                 A QGIS plugin
 Earth Surface Dynamics modeling algorithms
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-07-25
        copyright            : (C) 2025 by Lucas Amion
        email                : lamion@dcc.uchile.cl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Lucas Amion'
__date__ = '2025-07-25'
__copyright__ = '(C) 2025 by Lucas Amion'


from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessingAlgorithm,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterRasterDestination,
                       QgsProcessingException,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterVectorLayer,
                       QgsProcessingParameterBoolean,
                       QgsWkbTypes)

from landlab import RasterModelGrid
from landlab.components import FlowAccumulator, StreamPowerEroder as LandlabStreamPowerEroder
from ..utils import (rasterToNumpy, createOutputRaster, createProcessingMask)


class StreamPowerEroder(QgsProcessingAlgorithm):
  """
  This algorithm applies stream power erosion to a raster layer.
  
  It takes a raster layer (typically a digital elevation model) and
  applies stream power erosion calculations to simulate the erosional
  processes that shape landscapes over time.

  All Processing algorithms should extend the QgsProcessingAlgorithm
  class.
  """

  # Constants used to refer to parameters and outputs. They will be
  # used when calling the algorithm from another algorithm, or when
  # calling from the QGIS console.

  OUTPUT = 'OUTPUT'
  INPUT = 'INPUT'
  K_SP = 'K_SP'
  DT = 'DT'
  MASK_LAYER = 'MASK_LAYER'
  SELECTED_FEATURES = 'SELECTED_FEATURES'

  def initAlgorithm(self, config):
    """
    Here we define the inputs and output of the algorithm, along
    with some other properties.
    """

    # We add the input raster layer.
    self.addParameter(
      QgsProcessingParameterRasterLayer(
        self.INPUT,
        self.tr('Input raster layer')
      )
    )

    # Add parameter for erosion constant K_sp
    self.addParameter(
      QgsProcessingParameterNumber(
        self.K_SP,
        self.tr('Erosion constant (K_sp)'),
        type=QgsProcessingParameterNumber.Double,
        defaultValue=1.0,
        minValue=0.0
      )
    )

    # Add parameter for time step dt
    self.addParameter(
      QgsProcessingParameterNumber(
        self.DT,
        self.tr('Time step (dt)'),
        type=QgsProcessingParameterNumber.Double,
        defaultValue=1.0,
        minValue=0.0
      )
    )

    # Add optional vector layer parameter for masking
    self.addParameter(
      QgsProcessingParameterVectorLayer(
        self.MASK_LAYER,
        self.tr('Mask layer'),
        types=[QgsWkbTypes.PolygonGeometry],
        optional=True
      )
    )

    # Add parameter for using selected features only
    self.addParameter(
      QgsProcessingParameterBoolean(
        self.SELECTED_FEATURES,
        self.tr('Use selected features only'),
        defaultValue=False
      )
    )

    # We add a raster destination in which to store our processed raster
    # (this usually takes the form of a newly created raster layer when the
    # algorithm is run in QGIS).
    self.addParameter(
      QgsProcessingParameterRasterDestination(
        self.OUTPUT,
        self.tr('Output raster layer')
      )
    )

  def processAlgorithm(self, parameters, context, feedback):
    """
    Here is where the processing itself takes place.
    """

    # Retrieve the input raster layer
    input_layer = self.parameterAsRasterLayer(parameters, self.INPUT, context)
    
    if input_layer is None:
      raise QgsProcessingException(self.tr('Input raster layer is not valid'))

    # Get parameters
    k_sp = self.parameterAsDouble(parameters, self.K_SP, context)
    dt = self.parameterAsDouble(parameters, self.DT, context)
    output_path = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)
    
    # Get optional mask layer
    mask_layer = self.parameterAsVectorLayer(parameters, self.MASK_LAYER, context)
    use_selected_features = self.parameterAsBool(parameters, self.SELECTED_FEATURES, context)

    feedback.pushInfo(self.tr(f'Processing raster: {input_layer.source()}\n'))
    feedback.pushInfo(self.tr(f'K_sp: {k_sp}, dt: {dt}\n'))
    if mask_layer:
        feedback.pushInfo(self.tr(f'Using mask layer: {mask_layer.name()}\n'))
        feedback.pushInfo(self.tr(f'Use selected features only: {use_selected_features}\n'))
    feedback.pushInfo(self.tr(f'Output will be saved to: {output_path}\n'))
    
    # Read raster data using utility function
    elevation_data, provider, extent, width, height, dx, dy = rasterToNumpy(input_layer, feedback)
    
    feedback.setProgress(20)
    
    # Create Landlab RasterModelGrid
    mg = RasterModelGrid((height, width), xy_spacing=(dx, dy))
    
    # Add elevation field to the grid
    mg.add_field("topographic__elevation", elevation_data.flatten(), at="node")
    
    # Create mask if vector layer is provided using utility function
    erosion_mask = createProcessingMask(mask_layer, use_selected_features, extent, width, height, dx, dy, feedback)
    
    feedback.setProgress(40)
    
    # Initialize flow accumulator and stream power eroder
    fr = FlowAccumulator(mg, flow_director="D8")
    sp = LandlabStreamPowerEroder(mg, K_sp=k_sp)
    
    feedback.pushInfo(self.tr('Running flow accumulation...\n'))
    fr.run_one_step()
    
    feedback.setProgress(60)
    
    # Store original elevation for masking
    if erosion_mask is not None:
        original_elevation = mg.at_node["topographic__elevation"].copy()
    
    feedback.pushInfo(self.tr('Running stream power erosion...\n'))
    sp.run_one_step(dt=dt)
    
    # Apply mask if provided - restore original elevation outside mask
    if erosion_mask is not None:
        feedback.pushInfo(self.tr('Applying mask to erosion results...\n'))
        mask_flat = erosion_mask.flatten()
        eroded_elevation = mg.at_node["topographic__elevation"]
        
        # Restore original elevation where mask is 0 (outside erosion areas)
        eroded_elevation[mask_flat == 0] = original_elevation[mask_flat == 0]
        
        # Update the grid with masked results
        mg.at_node["topographic__elevation"] = eroded_elevation
    
    feedback.setProgress(80)
    
    # Get the modified elevation data
    modified_elevation = mg.at_node["topographic__elevation"].reshape((height, width))
    
    # Save the processed data to raster file using utility function
    createOutputRaster(
        modified_elevation, output_path, extent, width, height, dx, dy, 
        input_layer, "Stream Power Eroded DEM", feedback
    )
    
    feedback.setProgress(100)

    # Return the output path
    return {self.OUTPUT: output_path}

  def name(self):
    """
    Returns the algorithm name, used for identifying the algorithm. This
    string should be fixed for the algorithm, and must not be localised.
    The name should be unique within each provider. Names should contain
    lowercase alphanumeric characters only and no spaces or other
    formatting characters.
    """
    return 'stream_power_eroder'

  def displayName(self):
    """
    Returns the translated algorithm name, which should be used for any
    user-visible display of the algorithm name.
    """
    return 'Stream Power Eroder'


  def group(self):
    """
    Returns the name of the group this algorithm belongs to. This string
    should be localised.
    """
    return self.tr('Erosion')

  def groupId(self):
    """
    Returns the unique ID of the group this algorithm belongs to. This
    string should be fixed for the algorithm, and must not be localised.
    The group id should be unique within each provider. Group id should
    contain lowercase alphanumeric characters only and no spaces or other
    formatting characters.
    """
    return 'erosion'
  
  def icon(self):
    """
    Returns the icon for the algorithm, which is used in the Processing
    toolbox.
    """
    return QIcon(":/plugins/landlab_algorithm_provider/img/icon.png")

  def tr(self, string):
    return QCoreApplication.translate('Processing', string)

  def createInstance(self):
    return StreamPowerEroder()

  def shortHelpString(self):
    """
    Returns a localised short helper string for the algorithm. This string
    should provide a basic description about what the algorithm does and the
    parameters and outputs associated with it.
    """
    return self.tr("""
    This algorithm applies stream power erosion to a topographic surface.

    The <b>Stream Power Eroder</b> component simulates erosional processes that shape landscapes over time using the stream power law. It calculates erosion rates based on drainage area, slope, and user-defined parameters to model landscape evolution.

    <b>Parameters:</b>
      <b>- Input raster layer:</b> Digital elevation model to process
      <b>- Erosion constant (K_sp):</b> Stream power erosion coefficient that controls erosion rate
      <b>- Time step (dt):</b> Duration of erosion simulation in model time units
      <b>- Mask layer (optional):</b> Vector polygon layer to restrict erosion to specific areas
      <b>- Use selected features only:</b> Apply erosion only within selected features of the mask layer

    <b>Output:</b>
      <b>- Eroded DEM:</b> Raster layer showing the topography after stream power erosion

    <b>Note:</b> This algorithm requires flow accumulation to be calculated internally using D8 flow routing before applying erosion. When a mask layer is provided, erosion will only be applied within the masked areas, leaving the rest of the topography unchanged.
    """)
