# -*- coding: utf-8 -*-

"""
/***************************************************************************
 Landlab Algorithm Provider
                                 A QGIS plugin
 Earth Surface Dynamics modeling algorithms
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-07-25
        copyright            : (C) 2025 by Lucas Amion
        email                : lamion@dcc.uchile.cl
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Lucas Amion'
__date__ = '2025-07-25'
__copyright__ = '(C) 2025 by Lucas Amion'


from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessingAlgorithm,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterRasterDestination,
                       QgsProcessingException,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterVectorLayer,
                       QgsWkbTypes)

from landlab import RasterModelGrid
from landlab.components import SinkFiller as LandlabSinkFiller
from ..utils import rasterToNumpy, createOutputRaster, createProcessingMask


class SinkFiller(QgsProcessingAlgorithm):
  """
  This algorithm fills depressions (sinks) in a topographic surface.
  
  It takes a raster layer (typically a digital elevation model) and
  identifies depressions in the topographic surface, then fills them.
  No attempt is made to conserve sediment mass. User may specify whether
  the holes should be filled to flat, or with a gradient downwards towards
  the depression outlet.

  All Processing algorithms should extend the QgsProcessingAlgorithm
  class.
  """

  # Constants used to refer to parameters and outputs. They will be
  # used when calling the algorithm from another algorithm, or when
  # calling from the QGIS console.

  OUTPUT = 'OUTPUT'
  INPUT = 'INPUT'
  ROUTING = 'ROUTING'
  APPLY_SLOPE = 'APPLY_SLOPE'
  FILL_SLOPE = 'FILL_SLOPE'
  MASK_LAYER = 'MASK_LAYER'
  SELECTED_FEATURES = 'SELECTED_FEATURES'

  def initAlgorithm(self, config):
    """
    Here we define the inputs and output of the algorithm, along
    with some other properties.
    """

    # We add the input raster layer.
    self.addParameter(
      QgsProcessingParameterRasterLayer(
        self.INPUT,
        self.tr('Input raster layer')
      )
    )

    # Add parameter for routing method
    self.addParameter(
      QgsProcessingParameterEnum(
        self.ROUTING,
        self.tr('Routing method'),
        options=['D8', 'D4'],
        defaultValue=0  # D8 is default
      )
    )

    # Add parameter for apply slope
    self.addParameter(
      QgsProcessingParameterBoolean(
        self.APPLY_SLOPE,
        self.tr('Apply slope to filled areas'),
        defaultValue=False
      )
    )

    # Add parameter for fill slope
    self.addParameter(
      QgsProcessingParameterNumber(
        self.FILL_SLOPE,
        self.tr('Fill slope (m/m)'),
        type=QgsProcessingParameterNumber.Double,
        defaultValue=1e-05,
        minValue=0.0
      )
    )

    # Add optional vector layer parameter for masking
    self.addParameter(
      QgsProcessingParameterVectorLayer(
        self.MASK_LAYER,
        self.tr('Mask layer'),
        types=[QgsWkbTypes.PolygonGeometry],
        optional=True
      )
    )

    # Add parameter for using selected features only
    self.addParameter(
      QgsProcessingParameterBoolean(
        self.SELECTED_FEATURES,
        self.tr('Use selected features only'),
        defaultValue=False
      )
    )

    # We add a raster destination in which to store our processed raster
    # (this usually takes the form of a newly created raster layer when the
    # algorithm is run in QGIS).
    self.addParameter(
      QgsProcessingParameterRasterDestination(
        self.OUTPUT,
        self.tr('Output raster layer')
      )
    )

  def processAlgorithm(self, parameters, context, feedback):
    """
    Here is where the processing itself takes place.
    """

    # Retrieve the input raster layer
    input_layer = self.parameterAsRasterLayer(parameters, self.INPUT, context)
    
    if input_layer is None:
      raise QgsProcessingException(self.tr('Input raster layer is not valid'))

    # Get parameters
    routing_index = self.parameterAsEnum(parameters, self.ROUTING, context)
    routing_methods = ['D8', 'D4']
    routing = routing_methods[routing_index]
    
    apply_slope = self.parameterAsBool(parameters, self.APPLY_SLOPE, context)
    fill_slope = self.parameterAsDouble(parameters, self.FILL_SLOPE, context)
    output_path = self.parameterAsOutputLayer(parameters, self.OUTPUT, context)
    
    # Get optional mask layer
    mask_layer = self.parameterAsVectorLayer(parameters, self.MASK_LAYER, context)
    use_selected_features = self.parameterAsBool(parameters, self.SELECTED_FEATURES, context)

    feedback.pushInfo(self.tr(f'Processing raster: {input_layer.source()}\n'))
    feedback.pushInfo(self.tr(f'Routing: {routing}, Apply slope: {apply_slope}, Fill slope: {fill_slope}\n'))
    if mask_layer:
        feedback.pushInfo(self.tr(f'Using mask layer: {mask_layer.name()}\n'))
        feedback.pushInfo(self.tr(f'Use selected features only: {use_selected_features}\n'))
    feedback.pushInfo(self.tr(f'Output will be saved to: {output_path}\n'))
    
    # Read raster data using utility function
    elevation_data, provider, extent, width, height, dx, dy = rasterToNumpy(input_layer, feedback)
    
    feedback.setProgress(20)
    
    # Create Landlab RasterModelGrid
    mg = RasterModelGrid((height, width), xy_spacing=(dx, dy))
    
    # Add elevation field to the grid
    mg.add_field("topographic__elevation", elevation_data.flatten(), at="node")
    
    # Create mask if vector layer is provided using utility function
    processing_mask = createProcessingMask(mask_layer, use_selected_features, extent, width, height, dx, dy, feedback)
    
    feedback.setProgress(40)
    
    # Initialize sink filler
    sf = LandlabSinkFiller(mg, routing=routing, apply_slope=apply_slope, fill_slope=fill_slope)
    
    # Store original elevation for masking
    if processing_mask is not None:
        original_elevation = mg.at_node["topographic__elevation"].copy()
    
    feedback.pushInfo(self.tr('Running sink filling...\n'))
    sf.run_one_step()
    
    # Apply mask if provided - restore original elevation outside mask
    if processing_mask is not None:
        feedback.pushInfo(self.tr('Applying mask to sink filling results...\n'))
        mask_flat = processing_mask.flatten()
        filled_elevation = mg.at_node["topographic__elevation"]
        
        # Restore original elevation where mask is 0 (outside processing areas)
        filled_elevation[mask_flat == 0] = original_elevation[mask_flat == 0]
        
        # Update the grid with masked results
        mg.at_node["topographic__elevation"] = filled_elevation
    
    feedback.setProgress(80)
    
    # Get the modified elevation data
    modified_elevation = mg.at_node["topographic__elevation"].reshape((height, width))
    
    # Save the processed data to raster file using utility function
    createOutputRaster(
        modified_elevation, output_path, extent, width, height, dx, dy, 
        input_layer, "Sink-Filled DEM", feedback
    )
    
    feedback.setProgress(100)

    # Return the output path
    return {self.OUTPUT: output_path}

  def name(self):
    """
    Returns the algorithm name, used for identifying the algorithm. This
    string should be fixed for the algorithm, and must not be localised.
    The name should be unique within each provider. Names should contain
    lowercase alphanumeric characters only and no spaces or other
    formatting characters.
    """
    return 'sink_filler'

  def displayName(self):
    """
    Returns the translated algorithm name, which should be used for any
    user-visible display of the algorithm name.
    """
    return 'Sink Filler'

  def group(self):
    """
    Returns the name of the group this algorithm belongs to. This string
    should be localised.
    """
    return self.tr('Preprocessing')

  def groupId(self):
    """
    Returns the unique ID of the group this algorithm belongs to. This
    string should be fixed for the algorithm, and must not be localised.
    The group id should be unique within each provider. Group id should
    contain lowercase alphaneric characters only and no spaces or other
    formatting characters.
    """
    return 'preprocessing'
  
  def icon(self):
    """
    Returns the icon for the algorithm, which is used in the Processing
    toolbox.
    """
    return QIcon(":/plugins/landlab_algorithm_provider/img/icon.png")

  def tr(self, string):
    return QCoreApplication.translate('Processing', string)

  def createInstance(self):
    return SinkFiller()

  def shortHelpString(self):
    """
    Returns a localised short helper string for the algorithm. This string
    should provide a basic description about what the algorithm does and the
    parameters and outputs associated with it.
    """
    return self.tr("""
    This algorithm identifies depressions in a topographic surface and fills them.

    The <b>SinkFiller</b> component fills depressions (sinks) in digital elevation models. No attempt is made to conserve sediment mass. You can specify whether the holes should be filled to flat, or with a gradient downwards towards the depression outlet.

    <b>Parameters:</b>
      <b>- Input raster layer:</b> Digital elevation model to process
      <b>- Routing method:</b> D8 (8-direction) or D4 (4-direction) flow routing
      <b>- Apply slope:</b> If checked, applies a slope to filled areas for flow routing
      <b>- Fill slope:</b> The slope applied to filled areas (only used if Apply slope is checked)
      <b>- Mask layer (optional):</b> Vector polygon layer to restrict sink filling to specific areas
      <b>- Use selected features only:</b> Apply sink filling only within selected features of the mask layer

    <b>Output:</b>
      <b>- Sink-filled DEM:</b> Raster layer with depressions removed

    <b>Note:</b> When a mask layer is provided, sink filling will only be applied within the masked areas, leaving the rest of the topography unchanged.
    """)
