# -*- coding: utf-8 -*-

"""
/***************************************************************************
 LandUseAnalyzer
                                 A QGIS plugin
 A plugin for Land Use spatial analysis tools
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-08-09
        copyright            : (C) 2023 by Alexandros Voukenas
        email                : avoukenas@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'Alexandros Voukenas'
__date__ = '2023-08-09'
__copyright__ = '(C) 2023 by Alexandros Voukenas'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (QgsProcessing,
                       QgsFeatureSink,
                       QgsVectorLayer,QgsProject,
                       QgsProcessingException,
                       QgsProcessingAlgorithm,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterField,
                       QgsProcessingParameterEnum,
                       QgsProcessingParameterNumber,
                       QgsProcessingParameterString,
                       QgsProcessingParameterBoolean,
                       QgsProcessingParameterFolderDestination,
                       QgsField,
                       QgsFields,
                       QgsFeature,
                       QgsGeometry,
                       QgsPoint,QgsWkbTypes)
import processing
import os
import inspect
import csv

class ZonalAnalysis(QgsProcessingAlgorithm):
    
    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    
    INPUT_LU = 'INPUT LAND USE LAYER'
    INPUT_LU_FIELD = 'LAND USE TYPE FIELD'
    
    INPUT_ZONES = 'INPUT ZONES LAYER'
    INPUT_ZONES_FIELD = 'INPUT ZONES LAYER FIELD'
    
    INPUT_ALGORITHM = 'INPUT ALGORITHM'

    def initAlgorithm(self, config):

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_LU,
                self.tr('Input Land Use Layer'),
                [QgsProcessing.TypeVectorPolygon]
            )
        )

        self.addParameter(
        QgsProcessingParameterField(
                self.INPUT_LU_FIELD,
                'Field defining Land Use type',
                '',
                self.INPUT_LU)
        )
        
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_ZONES,
                self.tr('Input Zones Layer'),
                [QgsProcessing.TypeVectorPolygon]
            )
        )
        
        self.addParameter(
        QgsProcessingParameterField(
                self.INPUT_ZONES_FIELD,
                'Unique ID for Zones features',
                '',
                self.INPUT_ZONES)
        )
        
        self.addParameter(
        QgsProcessingParameterEnum(
        self.INPUT_ALGORITHM, 
        self.tr('Analysis method'), 
        options=['Land Use Mix (Entropy)','Land Use Mix (HHI)','Total number of classes','Most dominant class by count','Most dominant class by area','Ratios'], 
        allowMultiple=True, 
        usesStaticStrings=False, 
        defaultValue=[])
        )
        
        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Zonal Analysis')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        source_LU = self.parameterAsSource(parameters, self.INPUT_LU, context)
        
        LU_field=self.parameterAsString(parameters,self.INPUT_LU_FIELD,context)
        
        source_ZONES=self.parameterAsSource(parameters, self.INPUT_ZONES, context)
        
        ZONES_field=self.parameterAsString(parameters,self.INPUT_ZONES_FIELD,context)
        
        lu_mix_algorithm=self.parameterAsEnums(parameters,self.INPUT_ALGORITHM,context)
        
        #get the type of the field for the unique ID of the zones, it will be useful later.
        for field in source_ZONES.fields():
            if field.name()==ZONES_field:
                ZONES_field_type=field.type()
                
        #some geoprocessing. First, an intersection so that the land use polygons get the id of the zone in which they fall inside
        intersection=processing.run("qgis:intersection", 
        {'INPUT':parameters[self.INPUT_LU],
        'OVERLAY':parameters[self.INPUT_ZONES],
        'INPUT_FIELDS':[parameters[self.INPUT_LU_FIELD]],
        'OVERLAY_FIELDS':[parameters[self.INPUT_ZONES_FIELD]],
        'OVERLAY_FIELDS_PREFIX':'Zone_',
        'OUTPUT':'TEMPORARY_OUTPUT',
        'GRID_SIZE':None},
        context=context,feedback=None)['OUTPUT']
        
        ZONES_id='Zone_'+ZONES_field
        
        #then, a dissolve so that all polygons of same class and zone id are meged together
        dissolution=processing.run("qgis:dissolve",
        {'INPUT':intersection,
        'FIELD':[parameters[self.INPUT_LU_FIELD],ZONES_id],
        'OUTPUT':'TEMPORARY_OUTPUT'},
        context=context,feedback=None)['OUTPUT']
        
        field_names=[]
        
        #depending on the analysis method chosen, make an appropriate field calculation for the measurement, and add the field name to a list
        if 0 in lu_mix_algorithm:
            feedback.pushInfo('Analyzing Entropy')
            dissolution=processing.run("qgis:fieldcalculator", 
               {'INPUT':dissolution,
               'FIELD_NAME':'Entropy',
               'FIELD_TYPE':0,
               'FIELD_LENGTH':0,
               'FIELD_PRECISION':0,
               'FORMULA':'0-round(sum(($area/sum($area,\"{0}\"))*ln($area/sum($area,\"{1}\")),\"{2}\")/ln(count_distinct(\"{3}\",\"{4}\")),2)'.format(ZONES_id,ZONES_id,ZONES_id,parameters[self.INPUT_LU_FIELD],ZONES_id),
               'OUTPUT':'TEMPORARY_OUTPUT'},
               context=context,feedback=None)['OUTPUT']
           
            field_names.append('Entropy')
           
        if 1 in lu_mix_algorithm:
            feedback.pushInfo('Analyzing ΗΗΙ')
            dissolution=processing.run("qgis:fieldcalculator", 
               {'INPUT':dissolution,
               'FIELD_NAME':'HHI',
               'FIELD_TYPE':0,
               'FIELD_LENGTH':0,
               'FIELD_PRECISION':0,
               'FORMULA':'sum((100*($area/sum($area,\"{0}\")))^2,\"{1}\")'.format(ZONES_id,ZONES_id),
               'OUTPUT':'TEMPORARY_OUTPUT'},
               context=context,feedback=None)['OUTPUT']
               
            field_names.append('HHI')    
           
        if 2 in lu_mix_algorithm:
            feedback.pushInfo('Analyzing Total number of classes')
            dissolution=processing.run("qgis:fieldcalculator", 
               {'INPUT':dissolution,
               'FIELD_NAME':'Total_number_of_classes',
               'FIELD_TYPE':1,
               'FIELD_LENGTH':0,
               'FIELD_PRECISION':0,
               'FORMULA':'count(\"{0}\",\"{1}\")'.format(ZONES_id,ZONES_id),
               'OUTPUT':'TEMPORARY_OUTPUT'},
               context=context,feedback=None)['OUTPUT']
           
            field_names.append('Total_number_of_classes')
               
        if 3 in lu_mix_algorithm:
            feedback.pushInfo('Analyzing Most dominant class by count')
            dissolution=processing.run("qgis:fieldcalculator", 
               {'INPUT':dissolution,
               'FIELD_NAME':'Most_dominant_class_by_count',
               'FIELD_TYPE':2,
               'FIELD_LENGTH':255,
               'FIELD_PRECISION':0,
               'FORMULA':'array_first(array_agg( \"{0}\",\"{1}\",order_by:=1/num_geometries($geometry)))'.format(parameters[self.INPUT_LU_FIELD],ZONES_id),
               'OUTPUT':'TEMPORARY_OUTPUT'},
               context=context,feedback=None)['OUTPUT']
               
            field_names.append('Most_dominant_class_by_count')    
           
        if 4 in lu_mix_algorithm:
            feedback.pushInfo('Analyzing Most dominant class by area')
            dissolution=processing.run("qgis:fieldcalculator", 
                {'INPUT':dissolution,
                'FIELD_NAME':'Most_dominant_class_by_area',
                'FIELD_TYPE':2,
                'FIELD_LENGTH':255,
                'FIELD_PRECISION':0,
                'FORMULA':'array_first(array_agg( \"{0}\",\"{1}\",order_by:=1/$area))'.format(parameters[self.INPUT_LU_FIELD],ZONES_id),
                'OUTPUT':'TEMPORARY_OUTPUT'},
                context=context,feedback=None)['OUTPUT']
                
            field_names.append('Most_dominant_class_by_area') 
            
        #if 'Ratios' were selected, things are a bit more complicated. 
        if 5 in lu_mix_algorithm:
            feedback.pushInfo('Analyzing Ratios')
            
            #the area_pct field denotes the ratio of the area of each dissolved polygon to the total area of the polygons in the same zone
            dissolution=processing.run("qgis:fieldcalculator", 
               {'INPUT':dissolution,
               'FIELD_NAME':'area_pct',
               'FIELD_TYPE':0,
               'FIELD_LENGTH':0,
               'FIELD_PRECISION':0,
               'FORMULA':'100*$area/sum($area,\"{0}\")'.format(ZONES_id),
               'OUTPUT':'TEMPORARY_OUTPUT'},
               context=context,feedback=None)['OUTPUT']
            
            #then, we need to transpose the result
            # Get unique IDs and unique land use codes
            unique_ids = set(feat[ZONES_id] for feat in dissolution.getFeatures())
            unique_codes = set(feat[parameters[self.INPUT_LU_FIELD]] for feat in dissolution.getFeatures())

            # Create a dictionary to store the data
            result_dict = {}

            # Iterate over unique IDs
            for id in unique_ids:
                result_dict[id] = {}
                
                # Get area_pct values for each land use code
                for code in unique_codes:
                    area_pct = sum(feat["area_pct"] for feat in dissolution.getFeatures('\"{0}\" = {1} AND \"{2}\" = {3}'.format(ZONES_id,id,parameters[self.INPUT_LU_FIELD],code)))
                    result_dict[id][code] = area_pct
                        

            # Create a new memory layer
            layer_name = "SummaryShapefile"
            vl = QgsVectorLayer("Point?crs=EPSG:4326", layer_name, "memory")
            pr = vl.dataProvider()

            # Add fields to the layer
            pr.addAttributes([QgsField(ZONES_id, ZONES_field_type)])
            for code in unique_codes:
                pr.addAttributes([QgsField(str(code), 6)])
            vl.updateFields()

            # Create features for each ID and add them to the layer with dummy coordinates
            for id, codes in result_dict.items():
                feat = QgsFeature(vl.fields())
                feat[ZONES_id] = id
                
                # Create a dummy point geometry
                feat.setGeometry(QgsPoint(0, 0))
                
                # Set attributes for each maes_4_12 codesetGeom
                for code, value in codes.items():
                    feat[str(code)] = value
                
                # Add the feature to the layer
                pr.addFeature(feat)
            
            #finally, a join is required with the original zones layer
            dissolution=processing.run("qgis:joinattributestable", 
            {'INPUT':dissolution,
            'FIELD':ZONES_id,
            'INPUT_2':vl,
            'FIELD_2':ZONES_id,
            'FIELDS_TO_COPY':[],'METHOD':1,'DISCARD_NONMATCHING':False,'PREFIX':'',
            'OUTPUT':'TEMPORARY_OUTPUT',
            '--overwrite': True})['OUTPUT']
            
            for code in list(unique_codes):
                field_names.append(str(code)) 
        
        
        analysis_layer=processing.run("qgis:joinattributestable", 
        {'INPUT':parameters[self.INPUT_ZONES],
        'FIELD':parameters[self.INPUT_ZONES_FIELD],
        'INPUT_2':dissolution,
        'FIELD_2':ZONES_id,
        'FIELDS_TO_COPY':field_names,'METHOD':1,'DISCARD_NONMATCHING':False,'PREFIX':'',
        'OUTPUT':'TEMPORARY_OUTPUT',
        '--overwrite': True})['OUTPUT']
        
        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
                context, analysis_layer.fields(), analysis_layer.wkbType(), analysis_layer.sourceCrs())
        
        for i,f in enumerate(analysis_layer.getFeatures()):
            new_feature =  QgsFeature()
            new_feature.setGeometry(f.geometry())
            attributes_list=[f[element] for element in analysis_layer.fields().names()]
            new_feature.setAttributes(attributes_list)
            sink.addFeature(new_feature, QgsFeatureSink.FastInsert)
        
        return {self.OUTPUT: dest_id}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'zonal_analysis'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr('Zonal Analysis')
    
    def icon(self):
        cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]
        icon = QIcon(os.path.join(os.path.join(cmd_folder, 'land_use_mix_logo')))
        return icon

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return ''

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return ZonalAnalysis()
    
    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr(r"""The Zonal Analysis tool is essentially an extension of Zonal Statistics tool. It provides additional measurements, which are useful for the analysis of land use/land cover of a landscape.Those measurements are:
        Entropy: A measurement of land use mix. The higher the Entropy, the higher the mix. The Entropy Index varies from 0 to 1.
        HHI: Similar measurement to Entropy, but the higher the HHI, the lower the mix. Its values are not restricted in a scale.
        Total number of classes: For each zone, the total unique number of classes are calculated.
        Most dominant class by count: For each zone, identifies the class which appears the most times, in terms of polygon count.
        Most dominant class by area: For each zone, identifies the class which covers the most area.
        Ratios: For each zone, calculates the percentage of presence of each land use class (in terms of area).
        
        Inputs: The Land Use layer (polygon), the field that defines the class, the Zones layer, and a field that serves as a unique ID for each zone. The fields can be of any type (Text, Integer or Double).
        Analysis method: The desired measurements to be calculated.
        Output: The Zones layer, supplemented with all the measurements that have been selected.
        The Land Use layer does not need to have an "area" field, it is done within the tool.
        """)
