# -*- coding: utf-8 -*-

"""
/***************************************************************************
 LandProductivityAnalysisTool
                                 A QGIS plugin
 The plugin analyzes and assesses land productivity
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-06-14
        copyright            : (C) 2020 by James Katwere
        email                : katxeus@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

__author__ = 'James Katwere'
__date__ = '2020-06-14'
__copyright__ = '(C) 2020 by James Katwere'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (QgsProcessing,
                       QgsProject,
                       QgsFeatureSink,
                       QgsProcessingAlgorithm,
                       QgsProcessingUtils,
                       QgsProcessingParameterDefinition,
                       QgsProcessingParameterRasterLayer,
                       QgsProcessingParameterFeatureSource,
                       QgsProcessingParameterFeatureSink,
                       QgsProcessingParameterRasterDestination)

from qgis import processing
from qgis.utils import iface
import os

class LandProductivityAnalysisToolAlgorithm(QgsProcessingAlgorithm):
    """
    This is an example algorithm that takes a vector layer and
    creates a new identical one.

    It is meant to be used as an example of how to create your own
    algorithms and explain methods and variables used to do it. An
    algorithm like this will be available in all elements, and there
    is not need for additional work.

    All Processing algorithms should extend the QgsProcessingAlgorithm
    class.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'
    SAVI_INPUT = 'SAVI_INPUT'
    SAVI_OUTPUT = 'SAVI_OUTPUT'
    SOIL_INPUT = 'SOIL_INPUT'
    SOIL_OUTPUT = 'SOIL_OUTPUT'
    EROSIVITY_RASTER = 'EROSIVITY_RASTER'
    ERODIBILITY_RASTER = 'ERODIBILITY_RASTER'
    LS_RASTER = 'LS_RASTER'
    C_RASTER = 'C_RASTER'
    SOILDEPTHOUT = 'SOILDEPTHOUT'
    CECOUTPUT = 'CECOUTPUT'
    BSPHOUTPUT = 'BSPHOUTPUT'
    OMOUTPUT = 'OMOUTPUT'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.SOIL_INPUT,
                self.tr('Soil Productivity Index'),
                [QgsProcessing.TypeVectorAnyGeometry]
            )
        )
        self.addParameter(
            QgsProcessingParameterRasterLayer(
                self.SAVI_INPUT,
                self.tr('Net Primary Productivity Index')))

        # erosivity_field_param = QgsProcessingParameterRasterLayer(self.EROSIVITY_RASTER,
        #                                                           self.tr('Rainfall Erosivity Factor (R)'),
        #                                                           optional=True)
        # erosivity_field_param.setFlags(erosivity_field_param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        # self.addParameter(erosivity_field_param)
        #
        # erodibility_field_param = QgsProcessingParameterRasterLayer(self.ERODIBILITY_RASTER,
        #                                                             self.tr('Soil Erodibility Factor (K)'),
        #                                                             optional=True
        #                                                             )
        # erodibility_field_param.setFlags(erodibility_field_param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        # self.addParameter(erodibility_field_param)
        #
        # ls_field_param = QgsProcessingParameterRasterLayer(self.LS_RASTER,
        #                                                    self.tr(
        #                                                        'Topographic Slope Length and Slope Steepness Factors (LS)'),
        #                                                    optional=True)
        # ls_field_param.setFlags(ls_field_param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        # self.addParameter(ls_field_param)
        #
        # c_field_param = QgsProcessingParameterRasterLayer(self.C_RASTER,
        #                                                   self.tr('Crop and Management Factor (C)'),
        #                                                   optional=True)
        # c_field_param.setFlags(c_field_param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        # self.addParameter(c_field_param)


        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.SOIL_OUTPUT,
                self.tr('Soil Productivity Output layer')
            )
        )

        self.addParameter(
            QgsProcessingParameterRasterDestination(
                self.SAVI_OUTPUT,
                self.tr('Net Primary Productivity Output Layer')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        # source = self.parameterAsSource(parameters, self.SAVI_INPUT, context)
        # (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
        #         context, source.fields(), source.wkbType(), source.sourceCrs())

        # Compute the number of steps to display within the progress bar and
        # get features from source
        # total = 100.0 / source.featureCount() if source.featureCount() else 0
        # features = source.getFeatures()

        # for current, feature in enumerate(features):
        #     # Stop the algorithm if cancel button has been clicked
        #     if feedback.isCanceled():
        #         break
        #
        #     # Add a feature in the sink
        #     sink.addFeature(feature, QgsFeatureSink.FastInsert)
        #
        #     # Update the progress bar
        #     feedback.setProgress(int(current * total))

        soildepth_factor_rating_layer = processing.run(
            "qgis:advancedpythonfieldcalculator",
            {
                'FIELD_LENGTH': 5,
                'FIELD_NAME': 'depth_fr',
                'FIELD_PRECISION': 3,
                'FIELD_TYPE': 0,
                'FORMULA': 'if <depth> ==\'Deep\': \n value = 85\nelif <depth> ==\'Fairly deep\':\n value = 75\nelif <depth> ==\'Very deep\':\n value = 100\nelif <depth> ==\'Shallow\':\n value = 50\nelif <depth> ==\'Very shallow\':\n value = 20\nelse:\n value = 0',
                'GLOBAL': '',
                'INPUT': parameters['SOIL_INPUT'],
                'OUTPUT': 'SOILDEPTHOUT'
            }, is_child_algorithm=True, context=context, feedback=feedback)

        cec_factor_rating_layer = processing.run(
            "qgis:advancedpythonfieldcalculator",
            {
                'FIELD_LENGTH': 5,
                'FIELD_NAME': 'cec_fr',
                'FIELD_PRECISION': 3,
                'FIELD_TYPE': 0,
                'FORMULA': 'if <CEC_topsoi> <=5: \n value = 20\nelif <CEC_topsoi> > 5 and <CEC_topsoi> <= 20:\n value = 50\nelif <CEC_topsoi> > 20 and <CEC_topsoi> <= 40:\n value = 75\nelif <CEC_topsoi> > 40:\n value = 100',
                'GLOBAL': '',
                'INPUT': soildepth_factor_rating_layer['OUTPUT'],
                'OUTPUT': 'CECOUTPUT'
            }, is_child_algorithm=True, context=context, feedback=feedback)

        bsph_factor_rating_layer = processing.run(
            "qgis:advancedpythonfieldcalculator",
            {
                'FIELD_LENGTH': 5,
                'FIELD_NAME': 'BS_pH_fr',
                'FIELD_PRECISION': 3,
                'FIELD_TYPE': 0,
                'FORMULA': 'if <BS_and_pH> ==\'N1\': \n value = 20\nelif <BS_and_pH> ==\'N2\': \n value = 50\nelif <BS_and_pH> ==\'N5\': \n value = 70\nelif <BS_and_pH> ==\'N3\': \n value = 85\nelif <BS_and_pH> ==\'N4\': \n value = 100\nelse: \n value = 10',
                'GLOBAL': '',
                'INPUT': cec_factor_rating_layer['OUTPUT'],
                'OUTPUT': 'BSPHOUTPUT'
            }, is_child_algorithm=True, context=context, feedback=feedback)

        om_factor_rating_layer = processing.run(
            "qgis:advancedpythonfieldcalculator",
            {
                'FIELD_LENGTH': 5,
                'FIELD_NAME': 'om_fr',
                'FIELD_PRECISION': 3,
                'FIELD_TYPE': 0,
                'FORMULA': 'if <OM> <1: \n value = 20\nelif <OM> >= 1 and <OM> < 2:\n value = 50\nelif <OM> >= 2 and <OM> < 5:\n value = 70\nelif <OM> >= 5 and <OM> < 6:\n value = 85\nelif <OM> >= 6:\n value = 100\nelse:\n value = 0',
                'GLOBAL': '',
                'INPUT': bsph_factor_rating_layer['OUTPUT'],
                'OUTPUT': 'OMOUTPUT'
            }, is_child_algorithm=True, context=context, feedback=feedback)

        spi = processing.run(
            "qgis:advancedpythonfieldcalculator",
            {
                'FIELD_LENGTH': 5,
                'FIELD_NAME': 'LPI',
                'FIELD_PRECISION': 3,
                'FIELD_TYPE': 0,
                'FORMULA': 'import math \nvalue = 100*math.sqrt(<depth_fr>*<cec_fr>*<BS_pH_fr>*<om_fr>*0.00000001)\n ',
                'GLOBAL': '',
                'INPUT': om_factor_rating_layer['OUTPUT'],
                'OUTPUT': parameters['SOIL_OUTPUT']
            }, is_child_algorithm=True, context=context, feedback=feedback)
        self.spi = spi

        npp = processing.run(
            "native:reclassifybytable", {
                'INPUT_RASTER': parameters['SAVI_INPUT'],
                'NODATA_FOR_MISSING': False,
                'NO_DATA': -9999,
                'RANGE_BOUNDARIES': 1,
                'RASTER_BAND': 1,
                'TABLE': [0.,0.05,0.05,0.05,0.1,0.1,0.1,0.15,0.15,0.15,0.2,0.2,0.2,0.25,0.25,0.25,0.33,0.33,0.33,0.36,0.36,0.36,1,1],
                'OUTPUT': parameters['SAVI_OUTPUT']
            }, context=context, feedback=feedback)

        self.npp = npp

        # Return the results of the algorithm. In this case our only result is
        # the feature sink which contains the processed features, but some
        # algorithms may return multiple feature sinks, calculated numeric
        # statistics, etc. These should all be included in the returned
        # dictionary, with keys matching the feature corresponding parameter
        # or output names.
        return {self.SAVI_OUTPUT: npp['OUTPUT'], self.SOIL_OUTPUT: spi['OUTPUT']}

    def postProcessAlgorithm(self, context, feedback):
        spioutput = QgsProcessingUtils.mapLayerFromString(self.spi['OUTPUT'], context)
        spi_style_path = os.path.join(os.path.dirname(__file__), 'styles/spi.qml')
        spioutput.loadNamedStyle(spi_style_path)
        spioutput.triggerRepaint()
        iface.mapCanvas().refresh()
        iface.mapCanvas().refreshAllLayers()
        iface.layerTreeView().refreshLayerSymbology(spioutput.id())

        nppoutput = QgsProcessingUtils.mapLayerFromString(self.npp['OUTPUT'], context)
        npp_style_path = os.path.join(os.path.dirname(__file__), 'styles/pdvty.qml')
        nppoutput.loadNamedStyle(npp_style_path)
        nppoutput.triggerRepaint()
        iface.mapCanvas().refresh()
        iface.mapCanvas().refreshAllLayers()
        iface.layerTreeView().refreshLayerSymbology(nppoutput.id())

        for layer in QgsProject.instance().mapLayersByName('Reclassified raster'):
            layer.setName("Land Productivity")

        return {self.SAVI_OUTPUT:self.npp, self.SOIL_OUTPUT: self.spi}

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'landproductivityanalysistool'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return 'Land Productivity Analysis Tool'

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return ''

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr(
                       "Using a well developed methodology, the LPAT plugin aims to enhance the "
                       "assessment of extent and severity of land productivity using earth "
                       "observation datasets.\n\n "
                       "<hr>"
                       "<h3>Soil Productivity Index</h3>"
                       "To compute the Soil Productivity Index (SPI), the tool requires a soils dataset."
                       "This holds soil physio-chemical properties as field attributes."
                       "The SPI was defined based on the FAO soil classification and a computed Land Productivity Index (LPI)"
                       " The FAO soil classification provides inherent agricultural productivity of different soils."
                       " This represents the potential of a given soil to produce crops or biomass."
                       "<h3>Net Productivity Index</h3>"
                       "To compute Net Productivity Index, the tool requires a savi dataset as input."
                       "This holds the net primary productivity vegetation indice for the above-ground productivity state."
                       "integrates it with the appropriate soil productivity indices. \n\n Aggregated Land "
                       "Productivity is mathematically computed from combinatorial algebraic formulations derived "
                       "from parameters of soil and above ground productivity constrained with socio-economic and "
                       "infrastructure layers. \n\n "
                       "<hr>"
                       "<h3>Computed Outputs</h3>"
                       "<style>"
                       "table, th, td {"
                       "border: 2px solid black;"
                       "border-collapse: collapse;"
                       "}"
                       "th, td {"
                       "padding: 5px;"
                       "text-align: left;"
                       "}"
                       "</style>"
                       "\n\n "
                       "<table>"
                       "<tr>"
                       "<th><h3>Productivity</h3></th>"
                       "<th><h3>Intepretation</h3></th>"
                       "</tr>"
                       "<tr>"
                       "<td><b>Extremely Low</b></td>"
                       "<td>These are portions of land  that are  entirely bare or rocky</td>"
                       "</tr>"
                       "<tr>"
                       "<td><b>Very Low</b></td>"
                       "<td>These are portions of land  that are also bare to some extent</td>"
                       "</tr>"
                       "<tr>"
                       "<td><b>Low</b></td>"
                       "<td>These are portions of land that are characterized with sparse grasslands and shrubs and with bare patches</td>"
                       "</tr>"
                       "<tr>"
                       "<td><b>Slightly Moderate</b></td>"
                       "<td>Portions of land with slightly moderate standing biomass (closed vegetation e.g. Closed evergreen or deciduous forest, Mosaic vegetation (grassland/shrubland/forest) / cropland)</td>"
                       "</tr>"
                       "<tr>"
                       "<td><b>Slightly High</b></td>"
                       "<td>Portions of land with slightly high standing biomass (closed vegetation e.g. Closed evergreen or deciduous forest, Mosaic vegetation (grassland/shrubland/forest) / cropland)</td>"
                       "</tr>"
                       "<tr>"
                       "<td><b>High</b></td>"
                       "<td>Portions of land with high standing biomass (closed vegetation e.g. Closed evergreen or deciduous forest, Mosaic vegetation (grassland/shrubland/forest) / cropland)</td>"
                       "</tr>"
                       "<tr>"
                       "<td><b>Very High</b></td>"
                       "<td>Portions of land with very high standing biomass (closed vegetation e.g. Closed evergreen or deciduous forest, Mosaic vegetation (grassland/shrubland/forest) / cropland)</td>"
                       "</tr>"
                       "</table>"

        )

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return LandProductivityAnalysisToolAlgorithm()

    def prepLayers(self):
        for layer in QgsProject.instance().mapLayersByName('Reclassified raster'):
            layer.setName("Land Productivity")
